

import gmsh


def GeneratePCTraces(layer_start_pos, row_start_pos):

    # From https://www.raypcb.com/20-layer-pcb/  Typical PC board thickness for a 20 layer board is 4.8mm (0.188976)
    # Also see https://www.4pcb.com/pcb-manufacturing-custom-standard.html?gclid=Cj0KCQjw2_OWBhDqARIsAAUNTTFNxB_BvCqSDgZ8vUUes3CdI6zo6NjSIApcSEq-izmzeFYNPKuUFDoaApNOEALw_wcB
    pc_board_thickness = 0.188976    

    pc_trace_height = .5 * pc_board_thickness / 20       # height is actually 2 times this value.
    pc_trace_height_gap = 1.5 * pc_board_thickness / 20
    pc_trace_length = 0.005   # length is actually 2 times this value.  According to "Pcbcart" https://www.pcbcart.com/article/content/copper-trace-and-capacity-relationship.html
                              # 1 oz, .010 trace will handle 1 amp.
    pc_trace_length_gap = 0.015   # According to https://resources.altium.com/p/pcb-trace-and-pad-clearance-low-vs-high-voltage, we need .005 seperation between traces.

    N_num_trace_rows = 21  # For the smallest Micronix Linear stage we have a maximum of 8 mm travel (.315 inches) for the 20 mm length state.
    N_num_trace_layers = 10   #(For now, evaluate as a 10 layer board.)

    


    gmsh.model.occ.synchronize()
    
    # Arangment of traces is based on document C:\Simulation-Development\InductionPositioner\Fenics\Linear_Induction_Motor_Papers\umi-uta-1889.pdf


    traces = []
    rtn_vals = []
    
        # https://gitlab.onelab.info/gmsh/gmsh/blob/gmsh_4_10_4/examples/api/naca_boundary_layer_2d.py#L118
        ## (Break the 2 lenght lines up to get more vertec's)
    for layer in range(N_num_trace_layers):    
        for row in range(N_num_trace_rows):
            
            skip_row_generation = False            
            
            if layer >= 6:
                if (row < 3) or (row > ( N_num_trace_rows - 4)):
                    skip_row_generation = True
                 
                
            if not skip_row_generation:
                layer_p = pc_trace_height_gap * layer + layer_start_pos
                row_p = pc_trace_length_gap * row + row_start_pos

                p1 = gmsh.model.occ.addPoint(-pc_trace_length+row_p, -pc_trace_height+layer_p, 0)
                p1_5 = gmsh.model.occ.addPoint(-pc_trace_length*.5+row_p, -pc_trace_height+layer_p, 0)
                p1_0 = gmsh.model.occ.addPoint(row_p, -pc_trace_height+layer_p, 0)

                p2_5 = gmsh.model.occ.addPoint(pc_trace_length*.5+row_p, -pc_trace_height+layer_p, 0)
                p2 = gmsh.model.occ.addPoint(pc_trace_length+row_p, -pc_trace_height+layer_p, 0)


                p3 = gmsh.model.occ.addPoint(pc_trace_length+row_p, pc_trace_height+layer_p, 0)
                p3_5 = gmsh.model.occ.addPoint(pc_trace_length*.5+row_p, pc_trace_height+layer_p, 0)
                p3_0 = gmsh.model.occ.addPoint(row_p, pc_trace_height+layer_p, 0)

                p4_5 = gmsh.model.occ.addPoint(-pc_trace_length*.5+row_p, pc_trace_height+layer_p, 0)
                p4 = gmsh.model.occ.addPoint(-pc_trace_length+row_p, pc_trace_height+layer_p, 0)



                l1 = gmsh.model.occ.addLine(p1, p1_5)
                l1_1 = gmsh.model.occ.addLine(p1_5, p1_0)
                l1_2 = gmsh.model.occ.addLine(p1_0, p2_5)
                l1_3 = gmsh.model.occ.addLine(p2_5, p2)

                l2 = gmsh.model.occ.addLine(p2, p3)

                l3 = gmsh.model.occ.addLine(p3, p3_5)
                l3_1 = gmsh.model.occ.addLine(p3_5, p3_0)
                l3_2 = gmsh.model.occ.addLine(p3_0, p4_5)
                l3_3 = gmsh.model.occ.addLine(p4_5, p4)


                l4 = gmsh.model.occ.addLine(p4, p1)

                cl2 = gmsh.model.occ.addCurveLoop([l1, l1_1, l1_2, l1_3, l2, l3, l3_1, l3_2, l3_3, l4])
                traces.append(gmsh.model.occ.addPlaneSurface([cl2]))
    
    traces_mass = (pc_trace_height*(pc_trace_length/2)*8)    
    
    rtn_vals.append(traces_mass)
    rtn_vals.append(traces)
    return rtn_vals

    
    
    
