

import gmsh


def GeneratePCTraces_Rev_3(layer_start_pos, row_start_pos):

    # From https://www.raypcb.com/20-layer-pcb/  Typical PC board thickness for a 20 layer board is 4.8mm (0.188976)
    # Also see https://www.4pcb.com/pcb-manufacturing-custom-standard.html?gclid=Cj0KCQjw2_OWBhDqARIsAAUNTTFNxB_BvCqSDgZ8vUUes3CdI6zo6NjSIApcSEq-izmzeFYNPKuUFDoaApNOEALw_wcB
    pc_board_thickness = 0.188976    

    pc_trace_height = .0014  # 35 um       
    pc_trace_height_gap = pc_trace_height + .0087   #thickness of trace + pre-preg or FR-4 core which is approx. .22 mm
    pc_trace_length = 0.025   # According to "Pcbcart" https://www.pcbcart.com/article/content/copper-trace-and-capacity-relationship.html
                              # 1 oz, .010 trace will handle 1 amp.
                              # tooth - trace a - tooth - trace b - tooth - trace c - tooth   distributed over .315 inchs. This gives .06 inchs per tooth (without tolerance)
    pc_trace_length_gap = pc_trace_length +  0.06 +.005   
    
                              # According to https://resources.altium.com/p/pcb-trace-and-pad-clearance-low-vs-high-voltage, we need .005 seperation between traces.

    N_num_trace_rows = 4  # For the smallest Micronix Linear stage we have a maximum of 12 mm travel or (8 mm motor, .315 inches) for the 20 mm length state.
                          # (Rev_3 version, only 3 teeth.
    N_num_trace_layers = 10   #(For now, evaluate as a 10 layer board.)

    


    gmsh.model.occ.synchronize()
    
    # Arangment of traces is based on document C:\Simulation-Development\InductionPositioner\Fenics\Linear_Induction_Motor_Papers\umi-uta-1889.pdf
    
    
    N_num_x_incrs = 5
    N_num_y_incrs = 2
    
    traces = []
    rtn_vals = []
    
    
    
    
    for layer in range(N_num_trace_layers): 
        for row in range(N_num_trace_rows):
            
            point = []
            
            for x_inc in range(N_num_x_incrs):
                x_pos = (pc_trace_length*x_inc/N_num_x_incrs)+(pc_trace_length_gap*row)+row_start_pos
                y_pos = (0)+(pc_trace_height_gap*layer)+layer_start_pos
                pnt = gmsh.model.occ.addPoint(x_pos, y_pos, 0)
                point.append(pnt)
                
            for y_inc in range(N_num_y_incrs):
                x_pos = (pc_trace_length)+(pc_trace_length_gap*row)+row_start_pos
                y_pos = (pc_trace_height*y_inc/N_num_y_incrs)+(pc_trace_height_gap*layer)+layer_start_pos
                pnt = gmsh.model.occ.addPoint(x_pos, y_pos, 0)
                point.append(pnt)    
                
            for x_inc in range(N_num_x_incrs):
                x_pos = (pc_trace_length*(1.0 - x_inc/N_num_x_incrs))+(pc_trace_length_gap*row)+row_start_pos
                y_pos = (pc_trace_height)+(pc_trace_height_gap*layer)+layer_start_pos
                pnt = gmsh.model.occ.addPoint(x_pos, y_pos, 0)
                point.append(pnt)            
                
            for y_inc in range(N_num_y_incrs):
                x_pos = (0)+(pc_trace_length_gap*row)+row_start_pos
                y_pos = (pc_trace_height*(1.0 - y_inc/N_num_y_incrs))+(pc_trace_height_gap*layer)+layer_start_pos
                pnt = gmsh.model.occ.addPoint(x_pos, y_pos, 0)
                point.append(pnt)    

            line = []   

            for pnt_inc in range(2*N_num_x_incrs + 2*N_num_y_incrs):
                if pnt_inc == 2*N_num_x_incrs + 2*N_num_y_incrs - 1:
                    line_seg = gmsh.model.occ.addLine(point[pnt_inc], point[0])
                else:   
                    line_seg = gmsh.model.occ.addLine(point[pnt_inc], point[pnt_inc+1])
                line.append(line_seg)
                


            cl = gmsh.model.occ.addCurveLoop(line)
            traces.append(gmsh.model.occ.addPlaneSurface([cl]))   
            


    
    traces_mass = pc_trace_height*pc_trace_length  
    
    rtn_vals.append(traces_mass)
    rtn_vals.append(traces)
    return rtn_vals

    
    
    
