
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#define PI 3.1415926535897932384626433832795


//Calculation of Mutual Inductance Matrix for stator windings.



// Table B.1: Prototype machine stator summary specification. measured at the winding
//            synchronous speed with a single pole pitch search coil (see page 249)
//
// Parameter 						Value
//
// Frame size 						180
// Stator core length 					190mm
// Stator slots 					48
// Stator winding 1: Poles 				4
//		     coils per phase 			16 (series connected)
//		     turns per coil 			10
//		     Configuration 			star
//		     wire diameter 			1.2mm
//		     rated current 			6.78Arms (with 6A/mm2)
//		     Measured air gap flux density 	0.31Trms at 238Vrms
//		     at 50Hz, at phase voltage given 	0.12Trms at 95Vrms
//
// Stator winding 2  Poles 				8
// 		     coils per phase 			16 (series connected)
//		     turns per coil 			20
//		     Configuration 			star
//		     wire diameter 			1.2mm
//		     rated current 			6.78Arms (with 6A/mm2)
//		     Measured air gap flux density 	0.30Trms at 238Vrms
//		     at 50Hz, at phase voltage given 	0.12Trms at 95Vrms
//


// Table B.3: Stator winding details and fundamental winding factors (see page 252)
//
// No. slots 		No. slots    No. 	Turns 		No. coils 	No. turns 	Wire 	Wire 	Fund. 		Fund. 		Fund. 		Fund.
// short pitch. 	/phase/pole  layers 	per coil 	per phase 	per phase 	dia. 	area 	dist. fac. 	pitch fac. 	slot w. fac. 	winding fac.
//
//    Ms		    Mp 							   Nph 				   Kd 		  Kp 		   Ks 		   K
//											        (mm)    (mm**2)
//
// Stator 4 pole winding
//     2 		     4         2 	  10 		  16 		  160 		1.2 	 1.13    0.9577 	0.9659 		0.9998 		 0.9248
// Stator 8 pole winding
//     1 		     2 	       2 	  20 		  16 		  320 		1.2 	 1.13 	 0.9659 	0.9659 		0.9991 		 0.9322
//
// (NOTE: The calculations below use the single coil inductance equation defined in Eq 2.27 on page 46, Section 2.4.3. No winding factors are used).



// Stator (3 values are given for three different air gaps) (see page 284, Table C.2)
//
//diameter 	Stack length 	rotor-stator 	No. slots 	Carter Factor 	Carter Factor 	Carter Factor 	air gap	  air gap 	slot mouth 	air gap
//				stack overlap 					(Approx.) 	(Ossana Approx.) 	  (tol.) 	width 		(eff.)
//
//   d             w                                                K                                             g                       wo             geff
//  (mm)          (mm)             (mm)                                                                          (mm)     +/-(mm)        (mm)            (mm)
//
//  175.065 	 195.5 		    - 		   48 		 1.179 		   1.193 	   1.253 	0.547 	    - 		  3.2 		  -
//  175.065 	 195.5 		    - 		   48 		 1.178 		   1.192 	   1.252 	0.555 	    - 		  3.2 		  -
//  175.065 	 195.5 		    - 		   48 		 1.186 		   1.198 	   1.259 	0.510 	    - 		  3.2 		  -





 
// Parameters for 4 pole calculation:


#define w  .1955
#define d  .175065
#define Nk  10.0
#define Nj  10.0
#define g  .000555
#define u0  (4.0*PI*1.0e-7)   //Permeability of vacuum.

#define Ack  (2.0*PI*10.0/48.0)     //From Figure 2.2 on page 42.      
#define Wsk  (2.0*PI*.0032/(2.0*PI*d/2.0))   //

    
#define Acj  Ack	       //These the same as the k'th circuit.
#define Wsj  Wsk               //




#define slot_pitch (2.0*PI/48.0)  //degrees center-to-center of two consecutive slots.
 
FILE * fp_stator_stator_mu_ind_h;



double Bk;    	   	  //This changes depending on which row of the mutual inductance matrix is being calculated.
double Bj;		       //This becomes the variable for computing each matrix element of a given row.
double M[48][48] = {0};		       //One row of the mutual inductance matrix for the 4 pole calculation and 8 pole calculation
 
double Theta;
#define dTheta (2.0*PI/360.0)      //One degree integration.

double integral_cos_cos = 0;


double Ts_4[3][48] =  {{1,  1,  1,  1, -1, -1, -1, -1,  1,  1,  1,  1, -1, -1, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
		       {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1, -1, -1, -1, -1,  1,  1,  1,  1, -1, -1, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0}, 
                       {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1, -1, -1, -1, -1,  1,  1,  1,  1, -1, -1, -1, -1}}; 


double Mss_4_r[3][48];
double Mss_4[3][3];

 
//Invocation:
//
// gcc -g Section_B_2.c -lm



main()
{
	int n,i,j,k;

	//We use the mutual inductance equation defined in Eq 2.27 on page 46, Section 2.4.3 because there is no coil groups.
	//This seems to be the method taken by the paper when reviewing the equations listed in Eq. B.8, page 256. However
	//what is misleading is the addition of parameters Kd,  Kp and Ks winding factors which tend to suggest the use of 
	//of equations Eq. 2.36 through Eq. 2.39 on pages 52 and 53.)

	//Create include file for simulation program "App_InductionMotor.hpp" 
	fp_stator_stator_mu_ind_h = fopen("Stator_Stator_Mutual_Inductance.h", "w");
	fprintf(fp_stator_stator_mu_ind_h, "\n\n\ndouble Mss[3][3] = {\n");

	for(i = 0; i < 48; i++)
	{


		if(i < 16)
		{					//Start of Uout
	
			if(i < 4)
			{
			    Bk = slot_pitch * (double) i;
			}
			else if(i < 8)
			{
			    Bk = 12.0 * slot_pitch + slot_pitch * (double) (i - 4);
			}
			else if(i < 12)
			{
			    Bk = 24.0 * slot_pitch + slot_pitch * (double) (i - 8);
			}
			else
			{
			    Bk = 36.0 * slot_pitch + slot_pitch * (double) (i - 12);	
			}


		}
		else if(i < 32)
		{					//Start of Vout

			if(i < 20)
			{
			    Bk = slot_pitch * (double) (i - 16) + slot_pitch * 8.0;
			}
			else if(i < 24)
			{
			    Bk = 12.0 * slot_pitch + slot_pitch * (double) (i - 20)  + slot_pitch * 8.0;
			}
			else if(i < 28)
			{
			    Bk = 24.0 * slot_pitch + slot_pitch * (double) (i - 24)  + slot_pitch * 8.0;
			}
			else
			{
			    Bk = 36.0 * slot_pitch + slot_pitch * (double) (i - 28) + slot_pitch * 8.0;	
			}
	 

		} 
		else
		{					//Start of Wout
			if(i < 36)
			{
			    Bk = slot_pitch * (double) (i - 32) + slot_pitch * 16.0;
			}
			else if(i < 40)
			{
			    Bk = 12.0 * slot_pitch + slot_pitch * (double) (i - 36)  + slot_pitch * 16.0;
			}
			else if(i < 44)
			{
			    Bk = 24.0 * slot_pitch + slot_pitch * (double) (i - 40)  + slot_pitch * 16.0;
			}
			else
			{
			    Bk = 36.0 * slot_pitch + slot_pitch * (double) (i - 44) + slot_pitch * 16.0;	
			}
	 


		}




		for(k = 0; k < 48; k++)
		{ 

			// Refer to Figure B.2 on page 255
			// to calculate Bj.
			// First 16 coils (e.g. 1st phase or "U"), with Bj interate by "one slot" for each coil in the set. For the second, third and forth 
			// set shift 12, 24 and 36 slots alternating the polarity each time.
			// The transitions to the second and third 16 coils sets (eg., 2nd and 3rd phase), we shift and addition 4 and 8 slots respectively.

		   if(k < 16)
		   {								//Start of Uout
		
			if(k < 4)
		        {
			    Bj = slot_pitch * (double) k;
			}
			else if(k < 8)
		        {
			    Bj = 12.0 * slot_pitch + slot_pitch * (double) (k - 4);
			}
		        else if(k < 12)
		        {
			    Bj = 24.0 * slot_pitch + slot_pitch * (double) (k - 8);
			}
			else
		        {
			    Bj = 36.0 * slot_pitch + slot_pitch * (double) (k - 12);	
			}


		   }
		   else if(k < 32)
		   {								//Start of Vout

			if(k < 20)
		        {
			    Bj = slot_pitch * (double) (k - 16) + slot_pitch * 8.0;
			}
			else if(k < 24)
		        {
			    Bj = 12.0 * slot_pitch + slot_pitch * (double) (k - 20)  + slot_pitch * 8.0;
			}
		        else if(k < 28)
		        {
			    Bj = 24.0 * slot_pitch + slot_pitch * (double) (k - 24)  + slot_pitch * 8.0;
			}
			else
		        {
			    Bj = 36.0 * slot_pitch + slot_pitch * (double) (k - 28) + slot_pitch * 8.0;	
			}
	 

		   } 
		   else
		   {								//Start of Wout
			if(k < 36)
		        {
			    Bj = slot_pitch * (double) (k - 32) + slot_pitch * 16.0;
			}
			else if(k < 40)
		        {
			    Bj = 12.0 * slot_pitch + slot_pitch * (double) (k - 36)  + slot_pitch * 16.0;
			}
		        else if(k < 44)
		        {
			    Bj = 24.0 * slot_pitch + slot_pitch * (double) (k - 40)  + slot_pitch * 16.0;
			}
			else
		        {
			    Bj = 36.0 * slot_pitch + slot_pitch * (double) (k - 44) + slot_pitch * 16.0;	
			}
	 


		   }

		   


		   for(n = 1; n < 100; n++)
		   {
		  
		  	for(Theta = 0; Theta < 2.0*PI; Theta = Theta + dTheta)
		        {
			    integral_cos_cos = integral_cos_cos + cos((double) n * (Theta - Bk - Ack/2.0 - Wsk/2.0)) * cos((double) n * (Theta - Bj - Acj/2.0 - Wsj/2.0)) * dTheta;
			}
	     
			M[i][k] = M[i][k] + ((w*d*2.0*Nk*u0*2.0*Nj)/(2.0*PI*g*PI))*sin((double) n * Ack/2.0) * sin((double) n * Wsk/2.0) * sin((double) n * Acj/2.0) * sin((double) n * Wsj/2.0) *
					           integral_cos_cos /
							  (((double) n * (double) n * Wsk/2.0) * ((double) n * (double) n * Wsj/2.0)); 

			integral_cos_cos = 0;


	 	   }	

                     //select the row to be printed...
	   	   if(i == 0)
	  	       printf("%g\n", M[i][k]); 

	        }



	  
	}

	// double M[48][48] = {0};
	// double Ts_4[3][48];
	// double Mss_4_r[3]48];
	// double Mss_4[3][3];


	for(k = 0; k < 48; k++)
        { 

	   for(j = 0; j < 3; j++)
           {
	      for(i = 0; i < 48; i++)
	      {
	    
	    
		   Mss_4_r[j][k] = Mss_4_r[j][k] + M[k][i]*Ts_4[j][i];
	      }

	   }
        }
 

	for(k = 0; k < 3; k++)
        { 
	   for(j = 0; j < 3; j++)
           {
	      for(i = 0; i < 48; i++)
	      {
		 Mss_4[k][j] = Mss_4[k][j] + Ts_4[j][i]*Mss_4_r[k][i];

	      }
           }
         }
	
	 printf("\n\n\n");	
	 printf("%g\t%g\t%g\n", Mss_4[0][0],  Mss_4[0][1],  Mss_4[0][2]); 
	 printf("%g\t%g\t%g\n", Mss_4[1][0],  Mss_4[1][1],  Mss_4[1][2]); 
	 printf("%g\t%g\t%g\n", Mss_4[2][0],  Mss_4[2][1],  Mss_4[2][2]); 




	 fprintf(fp_stator_stator_mu_ind_h, "\t\t{%g\t,%g\t,%g},\n", Mss_4[0][0],  Mss_4[0][1],  Mss_4[0][2]); 
	 fprintf(fp_stator_stator_mu_ind_h, "\t\t{%g\t,%g\t,%g},\n", Mss_4[1][0],  Mss_4[1][1],  Mss_4[1][2]); 
	 fprintf(fp_stator_stator_mu_ind_h, "\t\t{%g\t,%g\t,%g},\n", Mss_4[2][0],  Mss_4[2][1],  Mss_4[2][2]); 
	 fprintf(fp_stator_stator_mu_ind_h, "\t\t};\n");


	 fclose(fp_stator_stator_mu_ind_h);


// This is the result. Close to that specified for Mss upper left diagonal block specified in Equation B.9
// Note that the ratio -96.3/210.0 in Equation B.9 is the same as -0.128702/0.279723  (-0.4585 and -.460 respectively)

//	0.278875	-0.127427	-0.127854
//	-0.127427	0.279723	-0.128702
//	-0.127854	-0.128702	0.280149




}
