#include "global.h"


#if defined(_SOLOIST_)
#define MAX_SOLOIST_CAM_ITERATIONS 2 // CTRL-22042
#endif

//=============================================================================
// Axis_GearCam()
//
// Performs Soloist Gearing and Camming.
//
// This is called by InterpolateSubMsecData().
//=============================================================================
#pragma CODE_SECTION(Axis_GearCam, ".internalCode")
void Axis_GearCam(unsigned int axisnum)
{
	double IncVelUnscaled;
#if defined(_SOLOIST_)
	double ThisMasterPos;
	double NextMasterPos;
	double CamPosition;
	int ThisOutputValue;
	short flag_CamLimitHigh = 0;
	short flag_CamLimitLow = 0;
#endif

	// Use "Gear/Cam Source" to determine a velocity to additionally factor in.
	switch (GearCamSource[axisnum])
	{
		case 0:         // open-loop global double value
			if (GearCamIndex[axisnum] <= CfgDoubleRegisterActive)
			{
				IncVelUnscaled = DoubleRegisters[GearCamIndex[axisnum]];
			}
			else
			{
				IncVelUnscaled = 0;
			}

			break;

#if (NUM_AUX_ENCODER_CHANNELS > 0)
		case 1:         // external encoder (default)
			IncVelUnscaled = AuxEncoderPosition[THIS_AXIS_INDEX] - AuxEncoderPositionPrevious[axisnum];      // AuxEncoderPositionPrevious will be initialized with GEAR command
			AuxEncoderPositionPrevious[axisnum] = AuxEncoderPosition[THIS_AXIS_INDEX];
			break;
#endif

		case 2:         // analog input #1
			IncVelUnscaled = (double)AnalogIn[0];

			if ((IncVelUnscaled < GearCamAnalogDeadband[axisnum]) && (IncVelUnscaled > -GearCamAnalogDeadband[axisnum]))
			{
				IncVelUnscaled = 0;
			}
			break;

		case 3:         // analog input #2
			IncVelUnscaled = (double)AnalogIn[1];

			if ((IncVelUnscaled < GearCamAnalogDeadband[axisnum]) && (IncVelUnscaled > -GearCamAnalogDeadband[axisnum]))
			{
				IncVelUnscaled = 0;
			}
			break;

		default:        // unsupported value -- do nothing
			IncVelUnscaled = 0;
			break;
	}

	// Unfiltered gearing/camming position
	GearCamPosUnfilt[axisnum] += IncVelUnscaled * GearCamScaleFactor[axisnum];

	// CTRL-23136: Shift gearing/camming position history (represents positions after filtering)
	// - 0, 1: Oldest positions
	// - 2:    Newest position (will be filled immediately below)
	GearCamPositionHistory[axisnum][0] = GearCamPositionHistory[axisnum][1];
	GearCamPositionHistory[axisnum][1] = GearCamPositionHistory[axisnum][2];

#if defined(_SOLOIST_)
	if (flag_CamActive)
	{
		BOOLEAN bIntervalFound = false;
		int NumIterations = MAX_SOLOIST_CAM_ITERATIONS;
		double NotFoundSlavePos;

		// read in all the data here, to be as efficient as possible.  Once the data is loaded
		// from external RAM, it will all run on the stack in internal RAM.
		ThisMasterPos = pCamData[CamTableIndex].MasterValue;
		NextMasterPos = pCamData[CamTableIndex + 1].MasterValue;

		CamPosition = GearCamPosUnfilt[SOLOIST_AXIS_INDEX] + CamOffset[SOLOIST_AXIS_INDEX];       // factor in the CamOffset parameter here

		if (NextMasterPos > 0)      // ascending table
		{
			if (CamPosition >= NextMasterPos)
			{
				while (NumIterations)
				{
					CamTableIndex++;
					// see if we're at the end of the table, then check for rollover
					if (CamTableIndex == EndCamTableIndex)
					{
						// check for rollover
						if (CamRolloverEnable[SOLOIST_AXIS_INDEX])
						{
							CamTableIndex = 0;
							GearCamPosUnfilt[SOLOIST_AXIS_INDEX] -= NextMasterPos;
							CamPosition -= NextMasterPos;
							NextMasterPos = 0;
						}
						else
						{
							// we're outside of the table, do nothing
							CamTableIndex -= 1;
							flag_CamLimitHigh = 1;
							break;
						}
					}
					else if (CamPosition >= NextMasterPos && CamPosition < pCamData[CamTableIndex + 1].MasterValue)
					{
						ThisMasterPos = NextMasterPos;
						bIntervalFound = true;
						break;
					}

					ThisMasterPos = NextMasterPos;
					NextMasterPos = pCamData[CamTableIndex + 1].MasterValue;
					NumIterations--;
				}

				// store since we didn't find the right interval
				NotFoundSlavePos = pCamData[CamTableIndex + 1].SlaveValue;
			}
			else if (CamPosition < ThisMasterPos)
			{
				while (NumIterations)
				{
					CamTableIndex--;
					// see if we're at the beginning of the table, then check for rollover
					if (CamTableIndex == -1)
					{
						// check for rollover
						if (CamRolloverEnable[SOLOIST_AXIS_INDEX])
						{
							CamTableIndex += EndCamTableIndex;
							NextMasterPos = pCamData[EndCamTableIndex].MasterValue;
							GearCamPosUnfilt[SOLOIST_AXIS_INDEX] += NextMasterPos;
							CamPosition += NextMasterPos;
						}
						else
						{
							// we're outside of the table, do nothing
							CamTableIndex = 0;
							flag_CamLimitLow = 1;
							break;
						}
					}
					else if (CamPosition >= pCamData[CamTableIndex].MasterValue && CamPosition < ThisMasterPos)
					{
						ThisMasterPos = pCamData[CamTableIndex].MasterValue;
						bIntervalFound = true;
						break;
					}

					NextMasterPos = ThisMasterPos;
					ThisMasterPos = pCamData[CamTableIndex].MasterValue;
					NumIterations--;
				}

				// store since we didn't find the right interval
				NotFoundSlavePos = pCamData[CamTableIndex].SlaveValue;
			}
			else
			{
				bIntervalFound = true;
			}
		}
		else                        // descending table
		{
			if (CamPosition <= NextMasterPos)
			{
				while (NumIterations)
				{
					CamTableIndex++;
					// see if we're at the end of the table, then check for rollover
					if (CamTableIndex == EndCamTableIndex)
					{
						// check for rollover
						if (CamRolloverEnable[SOLOIST_AXIS_INDEX])
						{
							CamTableIndex = 0;
							GearCamPosUnfilt[SOLOIST_AXIS_INDEX] -= NextMasterPos;
							CamPosition -= NextMasterPos;
							NextMasterPos = 0;
						}
						else
						{
							// we're outside of the table, do nothing
							CamTableIndex -= 1;
							flag_CamLimitHigh = 1;
							break;
						}
					}
					else if (CamPosition <= NextMasterPos && CamPosition > pCamData[CamTableIndex + 1].MasterValue)
					{
						ThisMasterPos = NextMasterPos;
						bIntervalFound = true;
						break;
					}

					ThisMasterPos = NextMasterPos;
					NextMasterPos = pCamData[CamTableIndex + 1].MasterValue;
					NumIterations--;
				}

				// store since we didn't find the right interval
				NotFoundSlavePos = pCamData[CamTableIndex + 1].SlaveValue;
			}
			else if (CamPosition > ThisMasterPos)
			{
				while (NumIterations)
				{
					CamTableIndex--;
					// see if we're at the beginning of the table, then check for rollover
					if (CamTableIndex == -1)
					{
						// check for rollover
						if (CamRolloverEnable[SOLOIST_AXIS_INDEX])
						{
							CamTableIndex += EndCamTableIndex;
							NextMasterPos = pCamData[EndCamTableIndex].MasterValue;
							GearCamPosUnfilt[SOLOIST_AXIS_INDEX] += NextMasterPos;
							CamPosition += NextMasterPos;
						}
						else
						{
							// we're outside of the table, do nothing
							CamTableIndex = 0;
							flag_CamLimitLow = 1;
							break;
						}
					}
					else if (CamPosition <= pCamData[CamTableIndex].MasterValue && CamPosition > ThisMasterPos)
					{
						ThisMasterPos = pCamData[CamTableIndex].MasterValue;
						bIntervalFound = true;
						break;
					}

					NextMasterPos = ThisMasterPos;
					ThisMasterPos = pCamData[CamTableIndex].MasterValue;
					NumIterations--;
				}

				// store since we didn't find the right interval
				NotFoundSlavePos = pCamData[CamTableIndex].SlaveValue;
			}
			else
			{
				bIntervalFound = true;
			}
		}

		if (flag_CamLimitHigh)
		{
			GearCamPositionHistory[axisnum][2] = pCamData[EndCamTableIndex].SlaveValue;   // force last value
		}
		else if (flag_CamLimitLow)
		{
			GearCamPositionHistory[axisnum][2] = pCamData[0].SlaveValue;                  // force first value
		}
		else if (bIntervalFound)
		{
			CamPosition -= ThisMasterPos;

			if (pCamData[CamTableIndex + 1].InterpType == 1)    // linear spline
			{
				GearCamPositionHistory[axisnum][2] = pCamData[CamTableIndex].SlaveDerivative * CamPosition;       // y = m * x
			}
			else                        // cubic spline
			{
				GearCamPositionHistory[axisnum][2] = ((((pCamData[CamTableIndex].CoeffA * CamPosition) + pCamData[CamTableIndex].CoeffB) * CamPosition) + pCamData[CamTableIndex].SlaveDerivative) * CamPosition;
			}
			GearCamPositionHistory[axisnum][2] += pCamData[CamTableIndex].SlaveValue;     // + b for linear, + d for cubic
		}
		else
		{
			// We searched but couldn't make it to the correct CAM table interval within 3 iterations
			// Use the slave position at the closest master position that was found
			GearCamPositionHistory[axisnum][2] = NotFoundSlavePos;
		}

		// Set outputs -- if CamOutputMask is 0, this will never set any outputs, but run this anyways
		ThisOutputValue = pCamData[CamTableIndex].OutData;
		if (ThisOutputValue != -1)      // -1 means no outputs are to be changed this segment
		{
			SetDigitalOutputs(SOLOIST_AXIS_INDEX, CamOutputMask[SOLOIST_AXIS_INDEX], ThisOutputValue);
		}
	}
	else
#endif
	{
		if (flag_GearFilterActive[axisnum])
		{
			// NOTE: 100 Hz low-pass filter on position
			GearCamPositionHistory[axisnum][2] =  0.067455273889 * GearCamPosUnfilt[axisnum] + 0.134910547778 * GearFiltIn1[axisnum];
			GearCamPositionHistory[axisnum][2] += 0.067455273889 * GearFiltIn2[axisnum];
			GearCamPositionHistory[axisnum][2] -= (-1.142980502540) * GearFiltOut1[axisnum];
			GearCamPositionHistory[axisnum][2] -= 0.412801598096  * GearFiltOut2[axisnum];

			// Time shift input and output samples
			GearFiltIn2[axisnum] = GearFiltIn1[axisnum];
			GearFiltIn1[axisnum] = GearCamPosUnfilt[axisnum];
			GearFiltOut2[axisnum] = GearFiltOut1[axisnum];
			GearFiltOut1[axisnum] = GearCamPositionHistory[axisnum][2];
		}
		else
		{
			GearCamPositionHistory[axisnum][2] = GearCamPosUnfilt[axisnum];
		}
	}

	// CTRL-23136: The gearing/camming logic fills in the most recent position
	// into the GearCamPositionHistory[] array (index 2).  However, in order to
	// approximate an analytic velocity, the position that we output will be
	// delayed by 1 millisecond (we will output index 1).
	GearCamPosition[axisnum] = GearCamPositionHistory[axisnum][1];

	// CTRL-23136: Update the most recent incremental position array (with 1 msec old gearing/camming position delta).
	// To approximate analytic velocity, we will use the Catmull-Rom spline method.
	AxisIncVelMsec[axisnum][2] += (GearCamPositionHistory[axisnum][1] - GearCamPositionHistory[axisnum][0]);
	AxisAnalyticVelocity[axisnum][2] += (0.5 * (GearCamPositionHistory[axisnum][2] - GearCamPositionHistory[axisnum][0]));
}


