#if !defined(DO_NOT_LINT_Firmware2)

#include "global.h"

#if defined(FEATURE_SERVO_LOOP)



#pragma CODE_SECTION(GenerateLoopTransExcitation, ".internalCode")
void GenerateLoopTransExcitation(float RecipRate)
{
#if defined(_A3200_) && !defined(FEATURE_INDEPENDENT_SERVO_INTERRUPT)
	// CTRL-11902: Synchronize generation of the excitation across multiple axes to the
	// same absolute timeslice.  This allows for synchronization of axes in gantry mode.
	if (!LoopTransSync)
	{
		if (ServoTimeslice == 7)
		{
			LoopTransSync = ON;
		}
	}
	else
#endif
	{
#if defined(_A3200_)
		// CTRL-13579: Set a flag indicating a valid loop trans excitation has been calculated. This
		// flag works with the LoopTransSync flag to make sure all axes wait until the next abs timeslice 0 to begin
		LoopTransGenerated = ON;
#endif
		// generate a sinusoidal disturbance
		if (ExcitationType == LTINPUT_SINUSOID)
		{
			float LoopTransInc;
			float LoopTransSinVal;
			float LoopTransCosVal;

			// calculate the increment for the angle based on frequency
			LoopTransInc = LoopTransFrequency * RecipRate * (float)0.001 * (float)TWO_PI;

			// Check to see if the next increment would cause us to start the next cycle
			if ((LoopTransAngleRadians + LoopTransInc) >= ((float)TWO_PI + RAD_IN_270_DEG))
			{
				// CTRL-22735: If we have reached the next cycle, reduce the loop
				// transmission angle by one cycle to prevent a decrease in precision
				LoopTransAngleRadians -= (float)TWO_PI;
				
				// if there has been a change to the parameters, we need to update them
				// at the beginning of the next full cycle
				if (LoopTransParamUpdate)
				{
					// update the parameters (frequency, amplitude and increment)
					LoopTransFrequency = NewLoopTransFrequency;
					LoopTransAmplitude = NewLoopTransAmplitude;
					LoopTransInc = LoopTransFrequency * RecipRate * (float)0.001 * (float)TWO_PI;

					// reset the update flag
					LoopTransParamUpdate = OFF;
				}
			}

			// increment the angle
			LoopTransAngleRadians += LoopTransInc;

			// calculate the sine
			LoopTransSinVal = sinf(LoopTransAngleRadians);

			// calculate the disturbance and put it in the disturbance array
			LoopTransDisturb[0] = (double)(LoopTransAmplitude * LoopTransSinVal);

			// calculate the cosine
			LoopTransCosVal = cosf(LoopTransAngleRadians);

			// calculate first derivative of the disturbance
			LoopTransDisturb[1] = (double)((float)TWO_PI * ((float)0.001 * RecipRate) * LoopTransFrequency * LoopTransAmplitude * LoopTransCosVal);
		}
		else if (ExcitationType == LTINPUT_WHTNOISE)
		{
			// temporary variable to help compiler optimizations
			float randomNoise;
			float filteredNoise;

			// generate random number disturbance
			randomNoise = ((float)2.0 * ((float)rand() / (float)RAND_MAX)) - (float)1.0;

			// run the low-pass filter on the data
			filteredNoise  = WhiteNoiseFilterN0 * randomNoise + WhiteNoiseFilterN1 * WhiteNoiseInHist0;
			filteredNoise += WhiteNoiseFilterN2 * WhiteNoiseInHist1;
			filteredNoise -= WhiteNoiseFilterD1 * WhiteNoiseOutHist0;
			filteredNoise -= WhiteNoiseFilterD2 * WhiteNoiseOutHist1;

			// time shift input and output samples
			WhiteNoiseInHist1  = WhiteNoiseInHist0;
			WhiteNoiseInHist0  = randomNoise;
			WhiteNoiseOutHist1 = WhiteNoiseOutHist0;
			WhiteNoiseOutHist0 = filteredNoise;

			// at this point the disturbance should be +/-1, so we multiply by the
			// desired excitation amplitude which comes from the LoopTransmission command
			LoopTransDisturb[0] = (double)(filteredNoise * LoopTransAmplitude);

			// calculate first derivative of the disturbance
			LoopTransDisturb[1] = (LoopTransDisturb[0] - LoopTransDisturbPrevious);

			// save previous disturbance values (for calculating derivatives)
			LoopTransDisturbPrevious = LoopTransDisturb[0];
		}
	}
}



#pragma CODE_SECTION(InjectPositionLTOpenLoopExcitation, ".internalCode")
void InjectPositionLTOpenLoopExcitation(unsigned int axisnum)
{
#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

#if defined(FEATURE_PIEZO)
	if (ServoMode[axisnum] == OFF)
	{
		// CTRL-21121: When performing a loop transmission in "servo off" mode, we are
		// measuring the plant response instead, so the output is the position feedback
		// from the axis instead of the servo output before the disturbance.
		// CTRL-21262: To report the proper phase, this must be inverted.  At some point in
		// the future, we will hopefully decide on the appropriate behavior and clean this up.
		LoopTransOutput[axisnum] = -PosLoopPositionFbk[axisnum];

		// add the calculated loop transmission disturbance into the current command
		ServoOutput[axisnum] += LoopTransDisturb[0];

		// capture the previous "disturbed" current command
		LoopTransInput[axisnum] = ServoOutputPrevious[axisnum];

		// CTRL-21121: The input measurement should be returned in Volts for a piezo drive.
		LoopTransInput[axisnum] *= PIEZO_AMP_COMMAND_SCALE;
	}
	else
#endif
	{
		// capture the current command before adding the disturbance
		LoopTransOutput[axisnum] = ServoOutput[axisnum];

		// add the calculated loop transmission disturbance into the current command
		ServoOutput[axisnum] += LoopTransDisturb[0];

		// capture the "disturbed" current command
		LoopTransInput[axisnum] = ServoOutput[axisnum];
	}
}



#pragma CODE_SECTION(InjectPositionLTClosedLoopExcitation, ".internalCode")
void InjectPositionLTClosedLoopExcitation(unsigned int axisnum)
{
#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

	// if we are trying to get a closed loop response from the position loop,
	// overwrite the position, velocity and acceleration commands with the
	// output of the loop transmission excitation

	// capture the feedback position as the "before" value
	LoopTransOutput[axisnum] = PosLoopPositionFbk[axisnum];

	// The disturbance is in the range [-LoopTransAmplitude, +LoopTransAmplitude].
	// Offset the disturbance so that it has the range [0, +2*LoopTransAmplitude].
	// This means that the sinusoidal disturbance will be zero at its lowest point (270 degrees).
	// When starting a loop transmission, we begin generating the disturbance at 270 degrees
	// (see the DigitalLoopTrans command). This is done to avoid injecting a step change into
	// the position command. Furthermore, the derivative of the sinusoid at 270 degrees is zero,
	// so we also avoid injecting a step change into the velocity command.
	LoopTransDisturb[0] += (double)LoopTransAmplitude;

	// The first time a closed loop transmission is injected, we need to save the current primary loop
	// command to serve as an offset for the disturbance.
	if (ClosedLoopTransmissionInitialized[axisnum] == false)
	{
		ClosedLoopTransmissionOffset[axisnum] = PositionCmd[axisnum];
		ClosedLoopTransmissionInitialized[axisnum] = true;
	}

	// Add in our saved offset
	PositionCmd[axisnum] = LoopTransDisturb[0] + ClosedLoopTransmissionOffset[axisnum];

	// also add in disturbances to velocity command
	VelocityCmdServo[axisnum] = LoopTransDisturb[1];

	// CTRL-13572: Update the acceleration command as well, so that the ANALOG TRACK
	// command will work as expected when doing a closed-loop loop transmission.
	AccelerationCmdServo[axisnum] = (VelocityCmdServo[axisnum] - VelocityCmdServoPrevious[axisnum]) * ServoRateKhz * ServoRateKhz;
	// CTRL-24829: Save the "velocity" command for use in next cycle's acceleration calculation.
	// This MUST be done here after the loop trans disturb has been calculated because
	// VelocityCmdServo is recalculated in DeterminePositionCommand every cycle prior to this
	// function call (probably calculated to zero) if we are doing a loop transmission.
	VelocityCmdServoPrevious[axisnum] = VelocityCmdServo[axisnum];

#if defined(FEATURE_INDEPENDENT_SERVO_INTERRUPT)
	// For the CLS, we must also calculate jerk command for jerk feed forward during drive motion
	JerkCmdServo[axisnum] = (AccelerationCmdServo[axisnum] - AccelerationCmdServoPrevious[axisnum]) * ServoRateKhz * ServoRateKhz * ServoRateKhz;
	AccelerationCmdServoPrevious[axisnum] = AccelerationCmdServo[axisnum];
#endif

#if defined(_ENSEMBLE_) || defined (_SOLOIST_)
	// The standalone drives use separate feedforward specific velocity and acceleration variables. We must overwrite these here.
	FeedforwardVelocity[axisnum] = VelocityCmdServo[axisnum];
	FeedforwardAccel[axisnum] = AccelerationCmdServo[axisnum];
#endif

	// capture the command position as the "after" value
	LoopTransInput[axisnum] = PositionCmd[axisnum];
}



#if defined(FEATURE_CURRENT_LOOP)
#pragma CODE_SECTION(InjectCurrentLTOpenLoopExcitation, ".internalCode")
void InjectCurrentLTOpenLoopExcitation(unsigned int axisnum)
{
#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

	// The output is set to the negative of the last cycle's current feedback.
	// The negative is to account for the -180 degrees of phase shift by the scope.
	LoopTransOutput[axisnum] = (double)-IdFeedback[axisnum];

	// Overwrite the PID loop output with the disturbance. This is to avoid over current faults
	// by bounding the amount of current output to the motor (as opposed to injecting the disturbance
	// into the error).
	// CTRL-19067: During current loop transmission excite using direct current.  Also return direct current
	// values to the SMC.
	DirectCurrentOut[axisnum] = (float)LoopTransDisturb[0];
	CurrentPIDOut[axisnum] = 0;

	// The input is set to the last cycle's current error.
	LoopTransInput[axisnum] = (double)IdErrorPrev[axisnum];
}
#endif



#if defined(FEATURE_AUTOFOCUS) && !defined(FEATURE_PIEZO)
#pragma CODE_SECTION(InjectAutoFocusLTOpenLoopExcitation, ".internalCode")
void InjectAutoFocusLTOpenLoopExcitation(unsigned int axisnum)
{
#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

	// capture the auto-focus loop output as the "before" value
	LoopTransOutput[axisnum] = PositionCmd[axisnum];

	// add the calculated loop transmission disturbance into the auto-focus command
	// NOTE: auto-focus currently does not make use of the feed-forward terms
	PositionCmd[axisnum] += LoopTransDisturb[0];
	VelocityCmdServo[axisnum] += LoopTransDisturb[1];

	// capture the "disturbed" auto-focus command at the "after" value
	LoopTransInput[axisnum] = PositionCmd[axisnum];
}
#endif



#if defined(FEATURE_AUTOFOCUS) && !defined(FEATURE_PIEZO)
#pragma CODE_SECTION(InjectAutoFocusLTClosedLoopExcitation, ".internalCode")
void InjectAutoFocusLTClosedLoopExcitation(unsigned int axisnum, int AutofocusFeedback)
{
#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

	// capture the auto-focus feedback as the "before" value
	LoopTransOutput[axisnum] = AutofocusFeedback * HALF_AIN_ADC_COUNTS;

	// The disturbance is in the range [-LoopTransAmplitude, +LoopTransAmplitude].
	// Offset the disturbance so that it has the range [0, +2*LoopTransAmplitude].
	// This means that the sinusoidal disturbance will be zero at its lowest point (270 degrees).
	// When starting a loop transmission, we begin generating the disturbance at 270 degrees
	// (see the DigitalLoopTrans command). This is done to avoid injecting a step change into
	// the position command. Furthermore, the derivative of the sinusoid at 270 degrees is zero,
	// so we also avoid injecting a step change into the velocity command.
	LoopTransDisturb[0] += (double)LoopTransAmplitude;

	// we want to start with the user's current selected target value
	// so we add in the starting auto focus target value
	AutofocusTarget[axisnum] = (float)LoopTransDisturb[0] + StartingAutofocusTarget[axisnum];

	// capture the auto-focus command as the "after" value
	LoopTransInput[axisnum] = (double)((AutofocusTarget[axisnum] - (float)HALF_AIN_ADC_COUNTS) * (float)HALF_AIN_ADC_COUNTS);
}
#endif




#pragma CODE_SECTION(ResetLoopTransmission, ".internalCode")
void ResetLoopTransmission(unsigned int axisnum)
{
#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

	// Disable generation of the loop transmission excitation
	PositionExcitationEnabled = OFF;
#if defined(FEATURE_CURRENT_LOOP)
	CurrentExcitationEnabled = OFF;
	// CTRL-19067: If there was a current loop transmission we were supplying current at
	// 0 degrees and must set the direct current back to zero.
	DirectCurrentOut[axisnum] = (float)0.0;
#endif

	// If we are aborting a current loop transmission, we need to prevent the
	// axis from jumping. We do this by "enabling" the axis to reset interpolation, integrators, etc.
	if (LoopTransType[axisnum] == LOOPTRANSMISSIONTYPE_CurrentLoop)
	{
		ResetServoLoopState(axisnum);
	}

	// Reset internal loop transmission state variables
	LoopTransType[axisnum] = LOOPTRANS_NONE;
	LoopTransParamUpdate = OFF;

#if defined(_A3200_)
	LoopTransGenerated = OFF;
#endif

#if defined(_A3200_) && !defined(_CLS_) && !defined(_FLS_) && !defined(_GCL_)
	// Reset the signal that is used to synchronize the excitation to an absolute timeslice
	LoopTransSync = OFF;
#endif

	// CTRL-24970: Clear closed loop initialization, forcing a new offset to be calculated on the first injection
	ClosedLoopTransmissionInitialized[axisnum] = false;

	// Reset the loop transmission excitation outputs to zero
	LoopTransDisturb[0] = 0.0;
	LoopTransDisturb[1] = 0.0;
}

#pragma CODE_SECTION(CalculateLowPass, ".internalCode")
//#pragma CODE_SECTION(CalculateLowPass, LOC_TEXT)
void CalculateLowPass(float cutoffFreq)
{
	float a;
	float num1, num2;
	float sampleFrequency = ServoRateKhz * (float)1000.0;

	if (cutoffFreq <= (float)0.0)
	{
#if !defined(_A3200_)
		// Cutoff Frequency must be greater than zero
		TaskError(TaskError_ArgumentOutOfBounds, 64);
#endif
	}
	else if (cutoffFreq >= (sampleFrequency / (float)2.0))
	{
		// Pass through filter; No band limit
		WhiteNoiseFilterD1 = (float)0.0;
		WhiteNoiseFilterD2 = (float)0.0;
		WhiteNoiseFilterN0 = (float)1.0;
		WhiteNoiseFilterN1 = (float)0.0;
		WhiteNoiseFilterN2 = (float)0.0;
	}
	else
	{
		a = (float)2.0 * atan2f(((float)PI * cutoffFreq) , sampleFrequency);
		num1 = (float)(SQRT_2 / 2.0) * sinf(a);
		num2 = ((float)1.0 - num1) / ((float)1.0 + num1);
		WhiteNoiseFilterD1 = -((float)1.0 + num2) * cosf(a);
		WhiteNoiseFilterD2 = num2;
		WhiteNoiseFilterN0 = (((float)1.0 + WhiteNoiseFilterD1) + WhiteNoiseFilterD2) / (float)4.0;
		WhiteNoiseFilterN1 = (float)2.0 * WhiteNoiseFilterN0;
		WhiteNoiseFilterN2 = WhiteNoiseFilterN0;
	}

	// initialize the filter back values now, before we start running
	WhiteNoiseInHist1  = 0;
	WhiteNoiseInHist0  = 0;
	WhiteNoiseOutHist1 = 0;
	WhiteNoiseOutHist0 = 0;

	// and initialize for derivatives
	LoopTransDisturbPrevious = 0.0;
}

#endif

#endif
