#if !defined(DO_NOT_LINT_Firmware2)

#include "global.h"


#if defined(FEATURE_SERVO_LOOP)

#if defined(FEATURE_ENHANCED_THROUGHPUT)
static inline float EnhancedThroughput(unsigned int axisnum);
#endif
// CTRL-22910: EnhancedTrackingControl is not declared as inline since it 
// must be called from multiple places in the servo loop.
static FILTER_FP EnhancedTrackingControl(unsigned int axisnum, SERVO_FP etcFilterInput);
static inline FILTER_FP FeedForward(unsigned int axisnum);
static inline void FilterServoOutput(void);
static inline void ServoLoopInputTransformation(void);
static inline void ServoLoopOutputTransformation(void);

// Use defines for the constant coefficients of the Enhanced Tracking Control
// filters rather than allocating memory for global variables.
#if defined(FEATURE_PIEZO)
#define etcFilterCoeffN2                (0)
#define etcDelayCoeffN0                 (1 - ETC_SAMPLE_DELAY_COMPENSATION)
#define etcDelayCoeffN1                 (ETC_SAMPLE_DELAY_COMPENSATION)
#define etcDelayCoeffN2                 (0)
#define etcDelayCoeffD1                 (0)
#define etcDelayCoeffD2                 (0)
#else
#define etcFilterCoeffN2                (0)
#define etcDelayCoeffN0                 (0)
#define etcDelayCoeffN1                 (1 - ETC_SAMPLE_DELAY_COMPENSATION)
#define etcDelayCoeffN2                 (ETC_SAMPLE_DELAY_COMPENSATION)
#define etcDelayCoeffD1                 (0)
#define etcDelayCoeffD2                 (0)
#endif



// This function transforms the position command/feedback values
// into servo loop inputs to allow for servo transformations from "SMC space"
#pragma CODE_SECTION(ServoLoopInputTransformation, ".internalCode")
static inline void ServoLoopInputTransformation(void)
{
	unsigned int axisnum;

	LOOP_OVER_ALL_AXES
	{
#if defined(_A3200_) && defined(FEATURE_GANTRY)
		// // Special transformation for gantry yaw control
		if (GantryYawControlEnable[axisnum] == ON)
		{
			// Check initialization (master and slave)
			switch(GantryYawControlInitState[axisnum])
			{
			case GANTRY_YAW_INIT_IDLE:
				// Wait in idle until we are told to start by an enable/home command
				break;

			case GANTRY_YAW_INIT_START:
				// On the timeslice that we start, a step change has occurred in command/ feedback.
				// We must wait until next timeslice for the step change to propagate across the gantry
				// comm channel.
				GantryYawControlInitState[axisnum] = GANTRY_YAW_INIT_ALIGN;
				break;

			case GANTRY_YAW_INIT_ALIGN:
				// Once we are here, the step change has propagated across the gantry comm channels and
				// we can now recompute our offset if we need to.
				// IF we are already homed, we have no need to compute an offset.
				if (AxisIsHomed[axisnum])
				{
					GantryYawControlPositionOffset[axisnum] = 0.0;
				}
				// IF we are not homed, we need to compute and apply an offset between the axes based on their current position
				// commands. This will prevent us from commanding a non-zero yaw command when we enable/home for the first time.
				else
				{
					if (isGantryMaster[axisnum])
					{
						GantryYawControlPositionOffset[axisnum] = (GantryYawControlPositionCmdDelay[axisnum] - GantryReceivePositionCmd[axisnum]);
					}
					else if (isGantrySlave[axisnum])
					{
						GantryYawControlPositionOffset[axisnum] = (GantryReceivePositionCmd[axisnum] - GantryYawControlPositionCmdDelay[axisnum]);
					}
				}

				// Since our position command/feedback just step changed and we compute velocity/acceleration inputs ourselves,
				// we must delay one more timeslice to prevent any velocity spikes from being calculated and sent into the servo loop.
				GantryYawControlInitState[axisnum] = GANTRY_YAW_INIT_MASK_VEL;
				break;

			case GANTRY_YAW_INIT_MASK_VEL:
				// We are now done synchronizing, the full yaw control mode will be enabled below.
				GantryYawControlInitState[axisnum] = GANTRY_YAW_INIT_DONE;
				break;

			case GANTRY_YAW_INIT_DONE:
				// Sit here in the done state until a step change in position occurs, forcing us to resynch.
			default:
				break;
			}

			// If we are not fully synchronized, we must make sure to prevent any step changes from injecting disturbances into the loop
			if (GantryYawControlInitState[axisnum] != GANTRY_YAW_INIT_DONE)
			{
				if (isGantryMaster[axisnum])
				{
					// CTRL-25467 If we are already homed, we allow the servo loop to use the average of the inputs like in normal operation.
					// We do have to use the frozen command value to avoid any weirdness due to gantry packet delays.
					if (AxisIsHomed[axisnum])
					{
						PrimaryLoopCommand[axisnum] = GantryYawControlPositionCmdFreeze[axisnum];
						PrimaryLoopFeedback[axisnum] = (GantryYawControlPosLoopPosFbkDelay[axisnum] + GantryReceivePosLoopPositionFbk[axisnum]) * 0.5;
						GantryYawControlSecondaryLoopFeedback[axisnum] = (GantryYawControlVelLoopPosFbkDelay[axisnum] + GantryReceiveVelLoopPositionFbk[axisnum] + GantryYawControlPositionOffset[axisnum]) * 0.5;
					}
					// If we are not homed just use the master's position input as R loop input, vel/acc are 0
					else
					{
						PrimaryLoopCommand[axisnum] = GantryYawControlPositionCmdDelay[axisnum];
						PrimaryLoopFeedback[axisnum] = GantryYawControlPosLoopPosFbkDelay[axisnum];
						GantryYawControlSecondaryLoopFeedback[axisnum] = GantryYawControlVelLoopPosFbkDelay[axisnum];
					}
				}
				else if (isGantrySlave[axisnum])
				{
					// CTRL-25467 If we are already homed, we allow the servo loop to use the difference of the inputs like in normal operation.
					// We do have to use the frozen command value to avoid any weirdness due to gantry packet delays.
					if (AxisIsHomed[axisnum])
					{
						PrimaryLoopCommand[axisnum] = GantryYawControlPositionCmdFreeze[axisnum];
						PrimaryLoopFeedback[axisnum] = GantryReceivePosLoopPositionFbk[axisnum] - GantryYawControlPosLoopPosFbkDelay[axisnum];
						GantryYawControlSecondaryLoopFeedback[axisnum] = GantryReceiveVelLoopPositionFbk[axisnum] - GantryYawControlVelLoopPosFbkDelay[axisnum];
					}
					// If we are not homed all theta loop inputs are 0
					else
					{
						PrimaryLoopCommand[axisnum] = 0;
						PrimaryLoopFeedback[axisnum] = 0;
						GantryYawControlSecondaryLoopFeedback[axisnum] = 0;
					}

				}
				
				// All vel/acc inputs are held at zero until we are synchronized
				PrimaryLoopCommandDerivative[axisnum] = 0;
				PrimaryLoopFeedbackDerivative[axisnum] = 0;
				PrimaryLoopCommandDerivative2[axisnum] = 0;
				SecondaryLoopFeedbackDerivative[axisnum] = 0;
			}
			else
			{
				// Master axis is R loop (averaged)
				if (isGantryMaster[axisnum])
				{
					// Primary loop command for the R loop is the average of the delayed master command and the received slave command.
					PrimaryLoopCommand[axisnum] = (GantryYawControlPositionCmdDelay[axisnum] + GantryReceivePositionCmd[axisnum] + GantryYawControlPositionOffset[axisnum]) * 0.5;

					// Primary loop feedback for the R loop is the average of the delayed master feedback and the received slave feedback
					PrimaryLoopFeedback[axisnum] = (GantryYawControlPosLoopPosFbkDelay[axisnum] + GantryReceivePosLoopPositionFbk[axisnum] + GantryYawControlPositionOffset[axisnum]) * 0.5;

					// Primary loop command derivative for the R loop is the backwards difference of the calculated R command
					PrimaryLoopCommandDerivative[axisnum] = PrimaryLoopCommand[axisnum] - GantryYawControlPrimaryLoopCommandPrevious[axisnum];

					// Primary loop feedback derivative for the R loop is the backwards difference of the calculated R feedback
					PrimaryLoopFeedbackDerivative[axisnum] = PrimaryLoopFeedback[axisnum] - GantryYawControlPrimaryLoopFeedbackPrevious[axisnum];

					// Primary loop command derivative 2 for the R loop is the backwards difference of the calculated R velocity feedback
					PrimaryLoopCommandDerivative2[axisnum] = PrimaryLoopCommandDerivative[axisnum] - GantryYawControlPrimaryLoopCommandDerivativePrevious[axisnum];

					// Secondary loop feedback for the R loop is the average of the delayed master feedback and the received slave feedback
					GantryYawControlSecondaryLoopFeedback[axisnum] = (GantryYawControlVelLoopPosFbkDelay[axisnum] + GantryReceiveVelLoopPositionFbk[axisnum] + GantryYawControlPositionOffset[axisnum]) * 0.5;

					// Secondary loop feedback derivative for the R loop is the backwards difference of the calculated R feedback
					SecondaryLoopFeedbackDerivative[axisnum] = GantryYawControlSecondaryLoopFeedback[axisnum] - GantryYawControlSecondaryLoopFeedbackPrevious[axisnum];
				}
				// Slave axis is theta loop (difference)
				else if (isGantrySlave[axisnum])
				{
					// Primary loop command for the theta loop is the difference of the received master command and the delayed slave command.
					PrimaryLoopCommand[axisnum] = ((GantryReceivePositionCmd[axisnum] - GantryYawControlPositionCmdDelay[axisnum]) - GantryYawControlPositionOffset[axisnum]);

					// Primary loop feedback for the theta loop is the difference of the received master feedback and the delayed slave feedback
					PrimaryLoopFeedback[axisnum] = ((GantryReceivePosLoopPositionFbk[axisnum] - GantryYawControlPosLoopPosFbkDelay[axisnum]) - GantryYawControlPositionOffset[axisnum]);

					// Primary loop command derivative for the theta loop is the backwards difference of the calculated theta command
					PrimaryLoopCommandDerivative[axisnum] = PrimaryLoopCommand[axisnum] - GantryYawControlPrimaryLoopCommandPrevious[axisnum];

					// Primary loop feedback derivative for the theta loop is the backwards difference of the calculated theta feedback
					PrimaryLoopFeedbackDerivative[axisnum] = PrimaryLoopFeedback[axisnum] - GantryYawControlPrimaryLoopFeedbackPrevious[axisnum];

					// Primary loop command derivative 2 for the theta loop is the backwards difference of the calculated theta velocity feedback
					PrimaryLoopCommandDerivative2[axisnum] = PrimaryLoopCommandDerivative[axisnum] - GantryYawControlPrimaryLoopCommandDerivativePrevious[axisnum];

					// Secondary loop feedback for the theta loop is the difference of the received master feedback and the delayed slave feedback
					GantryYawControlSecondaryLoopFeedback[axisnum] = ((GantryReceiveVelLoopPositionFbk[axisnum] - GantryYawControlVelLoopPosFbkDelay[axisnum]) - GantryYawControlPositionOffset[axisnum]);

					// Secondary loop feedback derivative for the theta loop is the the backwards difference of the calculated theta feedback
					SecondaryLoopFeedbackDerivative[axisnum] = GantryYawControlSecondaryLoopFeedback[axisnum] - GantryYawControlSecondaryLoopFeedbackPrevious[axisnum];
				}
			}

			// Delay this axis's inputs to align with comm channel inputs (accounting for 1 gantry packet delay)
			GantryYawControlPositionCmdDelay[axisnum] = PositionCmd[axisnum];
			GantryYawControlPosLoopPosFbkDelay[axisnum] = PosLoopPositionFbk[axisnum];
			GantryYawControlVelLoopPosFbkDelay[axisnum] = VelLoopPositionFbk[axisnum];

			// Keep history of calculated R/theta calculated values to compute backwards differences (to minimize math)
			GantryYawControlPrimaryLoopCommandPrevious[axisnum] = PrimaryLoopCommand[axisnum];
			GantryYawControlPrimaryLoopFeedbackPrevious[axisnum] = PrimaryLoopFeedback[axisnum];
			GantryYawControlPrimaryLoopCommandDerivativePrevious[axisnum] = PrimaryLoopCommandDerivative[axisnum];
			GantryYawControlSecondaryLoopFeedbackPrevious[axisnum] = GantryYawControlSecondaryLoopFeedback[axisnum];
		}
		else
#endif
		{
			PrimaryLoopCommand[axisnum] = PositionCmd[axisnum];
			PrimaryLoopFeedback[axisnum] = PosLoopPositionFbk[axisnum];
			PrimaryLoopCommandDerivative[axisnum] = VelocityCmdServo[axisnum];
			PrimaryLoopFeedbackDerivative[axisnum] = PosLoopVelocityFbk[axisnum];
			PrimaryLoopCommandDerivative2[axisnum] = AccelerationCmdServo[axisnum];
			SecondaryLoopFeedbackDerivative[axisnum] = VelLoopVelocityFbk[axisnum];
		}
		
	}
}



//////////////////////////////////////////////////////////////////////////
//
// This function implements the core servo loop, essentially what is
// shown in the servo loop block diagram in the help file.  It assumes
// that all global variables representing the position/velocity command
// and feedback have already been updated elsewhere.
//
//////////////////////////////////////////////////////////////////////////
#pragma CODE_SECTION(ServoLoop, ".internalCode")
void ServoLoop(void)
{
	unsigned int axisnum;
	SERVO_FP PrimaryLoopError[MAX_NUM_AXES];
	SERVO_FP PrimaryLoopDerivativeError[MAX_NUM_AXES];
	SERVO_FP SecondaryLoopError[MAX_NUM_AXES];
	SERVO_FP SecondaryLoopDerivativeError[MAX_NUM_AXES];
	SERVO_FP PrimaryProportionalGain_Active[MAX_NUM_AXES];
	SERVO_FP PrimaryIntegralGain_Active[MAX_NUM_AXES];
	SERVO_FP PrimaryDerivativeGain_Active[MAX_NUM_AXES];
	SERVO_FP SecondaryProportionalGain_Active[MAX_NUM_AXES];
	SERVO_FP SecondaryDerivativeGain_Active[MAX_NUM_AXES];
	SERVO_FP etcFilterInput[MAX_NUM_AXES];

	// CTRL-25469: HACK! To minimize the amount of work needed to get back to injecting a closed loop transmission
	// the correct way (into the servo loop variables directly, not position command etc), I am simply moving the injection
	// BEFORE the input transformation function. The injection function will be changed to use position command again, but when
	// we fix this, we will simply need to move this function after the input transformation again.
#if defined(_A3200_)
	// CTRL-13579: We should not inject a loop trans disturbance until after it has been generated
	if (LoopTransGenerated == ON)
#endif
	{
		LOOP_OVER_ALL_AXES
		{
			if (LoopTransType[axisnum] == LOOPTRANSMISSIONTYPE_ClosedLoop)
			{
				InjectPositionLTClosedLoopExcitation(axisnum);
			}
		}
	}

	// Transform servo loop inputs
	ServoLoopInputTransformation();

	LOOP_OVER_ALL_AXES
	{
		// Calculate primary loop error
		PrimaryLoopError[axisnum] = PrimaryLoopCommand[axisnum] - PrimaryLoopFeedback[axisnum];

		// Calculate the input to the Enhanced Tracking Control filter
#if defined(FEATURE_PIEZO)
		// CTRL-21291: Enhanced Tracking Control feature needs updated to support piezo drives
		etcFilterInput[axisnum] = PrimaryLoopError[axisnum];
#else
		// CTRL-19954: The input to the Enhanced Tracking Control filter is
		// the derivative of PrimaryLoopError. To take the backward difference,
		// we use the previous cycle's PrimaryLoopError before overwriting it.
		etcFilterInput[axisnum] = PrimaryLoopError[axisnum] - PrimaryLoopErrorPrevious[axisnum];
#endif

		// Store last loop error for ETC input
		PrimaryLoopErrorPrevious[axisnum] = PrimaryLoopError[axisnum];
	
		// Calculate the error terms needed to implement PID control for the primary loop
		PrimaryLoopDerivativeError[axisnum] = PrimaryLoopCommandDerivative[axisnum] - PrimaryLoopFeedbackDerivative[axisnum];
	}

	LOOP_OVER_ALL_AXES
	{
		// Determine whether or not to continue integrating
		if (!((ServoSetup[axisnum] & SERVO_KPI_OFF_MOVING) && (CommandedMotionDir[axisnum] != MOTION_NONE)))
		{
			// CTRL-21111: Only stop integrating if clamped in the direction we are trying to integrate
#if !defined(COMPILING_FIRMWARE2)
			if ((ServoOutputClamped[axisnum] * PrimaryLoopError[axisnum]) <= 0)
#else
			if (!ServoOutputClamped[axisnum])
#endif
			{
				PrimaryLoopErrorIntegral[axisnum] += PrimaryLoopError[axisnum];
			}
		}
	}

	LOOP_OVER_ALL_AXES
	{
		// Calculate the error terms needed to implement PD control for the secondary loop
		// CTRL-15059: Take into account the position offset between the primary loop and the secondary loop.
		SecondaryLoopDerivativeError[axisnum] = (PrimaryLoopCommandDerivative[axisnum] * GainKv[axisnum]) - SecondaryLoopFeedbackDerivative[axisnum];
		
		// CTRL-21111: Only stop integrating if clamped in the direction we are trying to integrate
#if !defined(COMPILING_FIRMWARE2)
		if ((ServoOutputClamped[axisnum] * SecondaryLoopDerivativeError[axisnum]) <= 0)
#else
		if (!ServoOutputClamped[axisnum])
#endif
		{
			SecondaryLoopIntegral[axisnum] += SecondaryLoopDerivativeError[axisnum];
		}
		SecondaryLoopError[axisnum] = SecondaryLoopIntegral[axisnum];
	}

#if defined(FEATURE_GAIN_SCHEDULING)
	// PCC-2807: These features are part of the DCT, which is a controller option
	if (DynamicControlsEnabled)
	{
		LOOP_OVER_ALL_AXES
		{
			// Adjust gains dynamically
			DynamicGainScheduling(axisnum, SecondaryLoopError[axisnum], SecondaryLoopDerivativeError[axisnum]);

			// Adjust gains based on thresholds
			ThresholdGainScheduling(axisnum);
		}
	}
#endif

	LOOP_OVER_ALL_AXES
	{
#if defined(FEATURE_GAIN_SCHEDULING)
		// Determine the active PID gains for the primary loop, as these may depend on the
		// values other parameters and/or both dynamic and position-based gain scheduling.
		PrimaryProportionalGain_Active[axisnum] = GainKp1[axisnum] * PositionGainKp1Scale[axisnum];
		PrimaryIntegralGain_Active[axisnum] = GainKpi[axisnum] * PositionGainKpiScale[axisnum];
		PrimaryDerivativeGain_Active[axisnum] = GainKd1[axisnum] * PositionGainKd1Scale[axisnum];

		// Determine the active PD gains for the secondary loop, as these may depend on the
		// values other parameters and/or both dynamic and position-based gain scheduling.
		SecondaryProportionalGain_Active[axisnum] = GainKi[axisnum] * GainKiScale[axisnum] * PositionGainKiScale[axisnum];
		SecondaryDerivativeGain_Active[axisnum] = GainKp[axisnum] * GainKpScale[axisnum] * PositionGainKpScale[axisnum];
#else
		// CTRL-23583/CTRL-26687: Removing support for dynamic and position gain scheduling on CLS and FLS.
		PrimaryProportionalGain_Active[axisnum] = GainKp1[axisnum];
		PrimaryIntegralGain_Active[axisnum] = GainKpi[axisnum];
		PrimaryDerivativeGain_Active[axisnum] = GainKd1[axisnum];

		SecondaryProportionalGain_Active[axisnum] = GainKi[axisnum];
		SecondaryDerivativeGain_Active[axisnum] = GainKp[axisnum];
#endif
	}

	LOOP_OVER_ALL_AXES
	{
		// CTRL-21121: Save the previous state of the servo output
		ServoOutputPrevious[axisnum] = ServoOutput[axisnum];

		// Calculate the servo loop contribution from the primary PID loop
		ServoOutput[axisnum] = PrimaryLoopError[axisnum] * PrimaryProportionalGain_Active[axisnum];
		ServoOutput[axisnum] += PrimaryLoopErrorIntegral[axisnum] * PrimaryIntegralGain_Active[axisnum];
		ServoOutput[axisnum] += PrimaryLoopDerivativeError[axisnum] * PrimaryDerivativeGain_Active[axisnum];
	}

	LOOP_OVER_ALL_AXES
	{
		// Implement GainKpos
		if ((ServoSetup[axisnum] & SERVO_KPOS_ALWAYS_ON) || AxisInPosition[axisnum])
		{
			SERVO_FP GainKposContribution;

			// Calculate the contribution to the secondary loop due to GainKpos (and its integral)
#if defined(FEATURE_GAIN_SCHEDULING)
			GainKposContribution = PrimaryLoopError[axisnum] * (SERVO_FP)GainKpos[axisnum] * GainKposScale[axisnum] * PositionGainKposScale[axisnum];
#else
			// CTRL-23583/CTRL-26687: Removing support for dynamic and position gain scheduling on CLS and FLS.
			GainKposContribution = PrimaryLoopError[axisnum] * (SERVO_FP)GainKpos[axisnum];
#endif

			// CTRL-21111: Only stop integrating if clamped in the direction we are trying to integrate
#if !defined(COMPILING_FIRMWARE2)
			if ((ServoOutputClamped[axisnum] * GainKposContribution) <= 0)
#else
			if (!ServoOutputClamped[axisnum])
#endif
			{
				GainKposIntegral[axisnum] += GainKposContribution;
			}

			// Add the GainKpos contribution to the secondary loop inputs
			SecondaryLoopError[axisnum] += GainKposIntegral[axisnum];
			SecondaryLoopDerivativeError[axisnum] += GainKposContribution;
		}
	}

	LOOP_OVER_ALL_AXES
	{
		// Add in the contribution from the secondary PD loop as well
		ServoOutput[axisnum] += SecondaryLoopError[axisnum] * SecondaryProportionalGain_Active[axisnum];
		ServoOutput[axisnum] += SecondaryLoopDerivativeError[axisnum] * SecondaryDerivativeGain_Active[axisnum];

#if defined(_A3200_)
		// CTRL-27389: Add in a new derivative gain that is in the feedback path.
		ServoOutput[axisnum] -= SecondaryLoopFeedbackDerivative[axisnum] * GainKdFeedback[axisnum];
#endif
	}

#if defined(FEATURE_SERVO_PARALLEL_INTEGRATORS)
	LOOP_OVER_ALL_AXES
	{
		// CTRL-20346: Add additional integrators to the servo loop for piezo stages
#if !defined(COMPILING_FIRMWARE2)
		if ((ServoOutputClamped[axisnum] * ServoOutput[axisnum]) <= 0)
#else
		if (!ServoOutputClamped[axisnum])
#endif
		{
			ServoIntegral1[axisnum] += ServoOutput[axisnum];
		}
		ServoOutput[axisnum] += ServoIntegral1[axisnum] * ((SERVO_FP)GainKsi1[axisnum]) + ServoIntegral1Offset[axisnum];
	}

	LOOP_OVER_ALL_AXES
	{
#if !defined(COMPILING_FIRMWARE2)
		if ((ServoOutputClamped[axisnum] * ServoOutput[axisnum]) <= 0)
#else
		if (!ServoOutputClamped[axisnum])
#endif
		{
			ServoIntegral2[axisnum] += ServoOutput[axisnum];
		}
		ServoOutput[axisnum] += ServoIntegral2[axisnum] * ((SERVO_FP)GainKsi2[axisnum]) + ServoIntegral2Offset[axisnum];
	}
#endif

	LOOP_OVER_ALL_AXES
	{
#if defined(_A3200_)
		// CTRL-13579: We should not inject a loop trans disturbance until after it has been generated
		if (LoopTransGenerated == ON)
#endif
		{
#if defined(FEATURE_PIEZO)
			if (ServoMode[axisnum] == ON)
#endif
			{
				if (LoopTransType[axisnum] == LOOPTRANSMISSIONTYPE_OpenLoop)
				{
					InjectPositionLTOpenLoopExcitation(axisnum);
				}
			}
		}

		// CTRL-19954: Add drive firmware support for the Enhanced Tracking Control feature
		if (EnhancedTrackingControlEnabled && (EnhancedTrackingSetup[axisnum] & ETCSETUP_ENABLED))
		{
			// CTRL-20411: Add a bit to control where the ETC contribution is injected
			if (EnhancedTrackingSetup[axisnum] & ETCSETUP_BEFORE_FILTERS)
			{
				ServoOutput[axisnum] += EnhancedTrackingControl(axisnum, etcFilterInput[axisnum]);
			}
		}

#if defined(FEATURE_HARMONIC_CANCELLATION)
		// PCC-2807: Harmonic cancellation is part of the DCT, which is a controller option
		if (DynamicControlsEnabled && (HarmonicCancellationSetup[axisnum]))
		{
			// PCC-1543: Harmonic cancellation algorithm MUST go before the feedforward
			ServoOutput[axisnum] += HarmonicCancellation(axisnum);
		}
#endif
	}
#if defined(FEATURE_PIEZO)
	LOOP_OVER_ALL_AXES
	{
		// CTRL-20343: Switch ServoOutput to zero if we a running servo control off.
		// This must go before servo filters, but after H.C and E.T.C
		if (ServoMode[axisnum] == OFF)
		{
			ServoOutput[axisnum] = PositionCmd[axisnum] - GainPffOffset[axisnum];
			ServoOutput[axisnum] *= PiezoVoltsPerUnit[axisnum] * fabs(RecipCountsPerUnit[axisnum]) * ((FILTER_FP)PIEZO_AMP_COMMAND_SCALE_RECIP);
		}
	}
#endif
	LOOP_OVER_ALL_AXES
	{
#if defined(FEATURE_PIEZO)
		// Calculate the feedforward contribution
		// Feedforward applies only in "servo on" mode
		if (ServoMode[axisnum] == ON)
		{
			FeedforwardOutput[axisnum] = FeedForward(axisnum);
		}
		else
		{
			FeedforwardOutput[axisnum] = 0;
		}
#else
		FeedforwardOutput[axisnum] = FeedForward(axisnum);
#endif

		// PCC-2005: Add a bit to control where the feedforward terms are injected
		if (ServoSetup[axisnum] & SERVO_AFF_BEFORE_FILTERS)
		{
			ServoOutput[axisnum] += FeedforwardOutput[axisnum];
		}
	}

	// Apply the servo loop filters
	FilterServoOutput();

	LOOP_OVER_ALL_AXES
	{
		// CTRL-19954: Add drive firmware support for the Enhanced Tracking Control feature
		if (EnhancedTrackingControlEnabled && (EnhancedTrackingSetup[axisnum] & ETCSETUP_ENABLED))
		{
			// CTRL-20411: Add a bit to control where the ETC contribution is injected
			if (!(EnhancedTrackingSetup[axisnum] & ETCSETUP_BEFORE_FILTERS))
			{
				ServoOutput[axisnum] += EnhancedTrackingControl(axisnum, etcFilterInput[axisnum]);
			}
		}
	}

	LOOP_OVER_ALL_AXES
	{
		// PCC-2005: Add a bit to control where the feedforward terms are injected
		if (!(ServoSetup[axisnum] & SERVO_AFF_BEFORE_FILTERS))
		{
			ServoOutput[axisnum] += FeedforwardOutput[axisnum];
		}
	}

#if !defined(_CLS_) && !defined(_FLS_) && !defined(_GCL_) && !defined(FEATURE_PIEZO)
	LOOP_OVER_ALL_AXES
	{
#if defined(FEATURE_ENHANCED_THROUGHPUT)
		// PCC-2807: Enhanced throughput support is a controller option
		if (EnhancedThroughputEnabled && (EnhancedThroughputGain[axisnum] != 0))
		{
			ServoOutput[axisnum] += EnhancedThroughput(axisnum);
		}
#endif

		// add in the static friction compensation
		if (CommandedMotionDirPrevious[axisnum] == MOTION_CW)
		{
			ServoOutput[axisnum] += (SERVO_FP)(StaticFrictionCompensation[axisnum] * RecipAmpMaxCurrent[axisnum]);
		}
		else if (CommandedMotionDirPrevious[axisnum] == MOTION_CCW)
		{
			ServoOutput[axisnum] -= (SERVO_FP)(StaticFrictionCompensation[axisnum] * RecipAmpMaxCurrent[axisnum]);
		}
	}
#endif

#if defined(FEATURE_PIEZO)
	LOOP_OVER_ALL_AXES
	{
		// CTRL-21121: When performing a loop transmission in "servo off" mode, we want
		// to inject the disturbance after the servo filters.
		if ((ServoMode[axisnum] == OFF) && (LoopTransType[axisnum] == LOOPTRANSMISSIONTYPE_OpenLoop))
		{
			InjectPositionLTOpenLoopExcitation(axisnum);
		}
	}

	LOOP_OVER_ALL_AXES
	{
		// Add offset from transitioning open to closed servo loop.
		ServoOutput[axisnum] += ServoTransitionOffset[axisnum];
	}
#endif

	// Transform servo loop outputs
	ServoLoopOutputTransformation();
}



// This function takes the output of the servo loop in "transformation space"
// and returns it to normal "SMC space" (current/voltage output)
#pragma CODE_SECTION(ServoLoopOutputTransformation, ".internalCode")
static inline void ServoLoopOutputTransformation(void)
{
	unsigned int axisnum;

#if defined(FEATURE_PIEZO)
	LOOP_OVER_ALL_AXES
	{
		if (flag_RampVoltageCmd[axisnum])
		{
			// CTRL-24537: Ramp at 750 V/s
			float disableRampRate = 0.75 * RecipServoRateKhz;                    
			if (PiezoVoltageCommand[axisnum] > disableRampRate)
			{
				PiezoVoltageCommand[axisnum] -= disableRampRate;
			}
			else if (PiezoVoltageCommand[axisnum] < -disableRampRate)
			{
				PiezoVoltageCommand[axisnum] += disableRampRate;
			}
			else
			{
				flag_DisableRampDone[axisnum] = ON;
				PiezoVoltageCommand[axisnum] = 0.0;
			}
		}
		else
		{
			// Set to output servo voltage when enabled.
			PiezoVoltageCommand[axisnum] = ServoOutput[axisnum] * PIEZO_AMP_COMMAND_SCALE * AmpEnabled[axisnum];
		}

	}
#else
	LOOP_OVER_ALL_AXES
	{
#if defined(_SOLOIST_)
		if ((ControlType[axisnum] != VELOCITY_MODE) && (ControlType[axisnum] != TORQUE_MODE))
#endif
		{
#if defined(_A3200_) && defined(FEATURE_GANTRY)
			// Special output transformation for gantry yaw control
			if (GantryYawControlEnable[axisnum] == ON)
			{
				// Delay and compute current commands for master (R + theta)
				if (isGantryMaster[axisnum])
				{
					CurrentCmd[axisnum] = (float)(GantryYawControlServoOutputDelay[axisnum] + GantryReceiveServoOutput[axisnum]);
					GantryYawControlServoOutputDelay[axisnum] = ServoOutput[axisnum];
				}
				// Delay and compute current commands for slave (R - theta)
				if (isGantrySlave[axisnum])
				{
					CurrentCmd[axisnum] = (float)(GantryReceiveServoOutput[axisnum] - GantryYawControlServoOutputDelay[axisnum]);
					GantryYawControlServoOutputDelay[axisnum] = ServoOutput[axisnum];
				}
			}
			else
#endif
			{
				CurrentCmd[axisnum] = (float)ServoOutput[axisnum];
			}
		}
	}
#endif
}



#pragma CODE_SECTION(FeedForward, ".internalCode")
static inline FILTER_FP FeedForward(unsigned int axisnum)
{
	FILTER_FP feedForwardOutput = 0;

#if defined(_A3200_)
	float AccelGain = GainAff[axisnum];
#endif

#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

#if defined(_A3200_)
#if defined(FEATURE_INDEPENDENT_SERVO_INTERRUPT)
	// CTRL-19604: Current feedforward needs to be advanced on the CLS
	double feedforwardAccel;
	double feedforwardJerk;
	double feedforwardVelocity;

#if defined(FEATURE_INDEPENDENT_SERVO_INTERRUPT)
	unsigned char feedforwardOff = flag_ResetInterpolation[axisnum];

#if defined(_CLS_) || defined(_FLS_) || defined(_GCL_)
	// CTRL-22822: Feedforward should be turned off while the drive is removing the IFOV offset.
	feedforwardOff = feedforwardOff || isHoldingIfovRegister[axisnum];
#elif defined(_QDe_)
	feedforwardOff = feedforwardOff || (EnableState[axisnum] != ENABLE_SEQUENCE_NONE);
#endif
	
	if (feedforwardOff)
	{
		// CTRL-20366: Prevent a spike in feedforward current due to a step change in position command.
		// We do not apply feedforward during the millisecond after the interpolator is reset in the FireWire interrupt.
		feedforwardVelocity = 0;
		feedforwardAccel = 0;
		feedforwardJerk = 0;

		// CTRL-20764: Any time the interpolation needs to be reset, there is
		// the potential for the position command to step change, so we need to
		// set a flag that will be communicated to the SMC.
		// CTRL-21048: The enable is now a special case, so we do not set this if the enable state machine is active.
		if (!EnableState[axisnum])
		{
			PositionCommandStepChange[axisnum] = ON;
#if defined(_CLS_) || defined(_FLS_) || defined(_GCL_)
			if (isHoldingIfovRegister[axisnum])
			{
				// CTRL-23144: The position feedback will also step change because
				// it is now being reported in scanner space instead of IFOV space.
				PositionFeedbackStepChange[axisnum] = ON;
			}
#endif
		}
	}
	else
#endif
	{
		// If we are doing a closed loop transmission, we must use servo variables to compute feedforward
		if ((LoopTransGenerated == ON) && (LoopTransType[axisnum] == LOOPTRANSMISSIONTYPE_ClosedLoop))
		{
#if !defined(_CLS_) && !defined(_FLS_) && !defined(_GCL_)
			FeedforwardPosition[axisnum] = PrimaryLoopCommand[axisnum];
#endif
			feedforwardVelocity = PrimaryLoopCommandDerivative[axisnum];
			feedforwardAccel = PrimaryLoopCommandDerivative2[axisnum];

			// CTRL-25469: Hack! I've temporarily created a jerk command value. This should be PrimaryLoopCommandDerivative3
			// once we do this the right way.
			feedforwardJerk = JerkCmdServo[axisnum];
		}
#if defined(FEATURE_INDEPENDENT_SERVO_INTERRUPT) && defined(SINGLE_AXIS_DRIVE)
		else if ((ServoMotionActive[axisnum] == ON) || (DisableServoMotion[axisnum] == true) || (ServoMotionAbortState[axisnum] != ABORT_NONE))
		{
			FeedforwardPosition[axisnum] = PrimaryLoopCommand[axisnum];
			feedforwardVelocity = PrimaryLoopCommandDerivative[axisnum];
			feedforwardAccel = PrimaryLoopCommandDerivative2[axisnum];

			// CTRL-25469: Hack! I've temporarily created a jerk command value. This should be PrimaryLoopCommandDerivative3
			// once we do this the right way.
			feedforwardJerk = JerkCmdServo[axisnum];
		}
#endif
		else
		{
			// Calculate the index into the feedforward data based on the FeedforwardAdvance parameter.
			unsigned int feedforwardIndex = 2 + ServoTimeslice + FeedforwardAdvanceSamples[axisnum];

			// Using the index determined from the FeedforwardAdvance parameter, retrieve the position
			// commands from which we will calculate the velocity/acceleration/jerk feedforward values.
			double positionPrevious = IpcIncomingCommandPacket_Read[axisnum].positionCommands[feedforwardIndex - 1];
			double positionCurrent = IpcIncomingCommandPacket_Read[axisnum].positionCommands[feedforwardIndex];
			double positionNext = IpcIncomingCommandPacket_Read[axisnum].positionCommands[feedforwardIndex + 1];
			double positionNext2 = IpcIncomingCommandPacket_Read[axisnum].positionCommands[feedforwardIndex + 2];

			// CTRL-23992: Calculate velocity using a forward difference of the position
			feedforwardVelocity = positionNext - positionCurrent;

			// CTRL-23992: Calculate acceleration using a 2nd order central difference of the position.
			feedforwardAccel = (positionNext - (2.0 * positionCurrent)) + positionPrevious;

			// CTRL-23992: Calculate jerk feedforward as a forward difference of the acceleration feedforward.
			// The calculation goes: J(t) = A(t+1) - A(t) = [V(t+2) - V(t)] - [V(t+1) - V(t-1)] which can be
			// algebraically combined/simplified to use only positions: P(t+2) - 3P(t+1) + 3P(t) - P(t-1)
			feedforwardJerk = (positionNext2 - (3.0 * positionNext)) + (3.0 * positionCurrent) - positionPrevious;

#if !defined(_CLS_) && !defined(_FLS_) && !defined(_GCL_)
			// CTRL-24049: Calculate position feedforward including the feedforward advance
			FeedforwardPosition[axisnum] = positionCurrent;
#endif
		}
	}

	// CTRL-15270: Add velocity feedforward for the Nmark CLS
	feedForwardOutput += feedforwardVelocity * GainVff[axisnum];

	// CTRL-15560: Add support for GainDff
	if ((GainDff[axisnum] > 0) && ((feedforwardAccel * feedforwardVelocity) < 0))
	{
		// If we are decelerating, use the GainDff value instead of GainAff
		AccelGain = GainDff[axisnum];
	}

	// CTRL-15270: Add acceleration feedforward for the Nmark CLS
	feedForwardOutput += feedforwardAccel * AccelGain;

	// CTRL-23992: Add jerk feedforward for the Nmark CLS
	feedForwardOutput += feedforwardJerk * GainJff[axisnum];

#else
	// CTRL-8691: Add feedforward contribution calculated by the SMC
	if ((fireWireReceivePointer_Next[fireWireReceiveIndex[axisnum]].packetInfo.commandValid) && (!HandshakeMoveActive[axisnum]))
	{
		unsigned int feedforwardIndex;

		// CTRL-5792: Calculate the index into the feedforward data from the SMC
		feedforwardIndex = FeedforwardAdvanceSamples[axisnum] + ServoTimeslice;

		if (feedforwardIndex < POSITIONS_PER_MSEC)
		{
			feedForwardOutput += fireWireReceivePointer_Active[fireWireReceiveIndex[axisnum]].feedforwardCurrentCommand[feedforwardIndex];
		}
		else
		{
			feedForwardOutput += fireWireReceivePointer_Next[fireWireReceiveIndex[axisnum]].feedforwardCurrentCommand[feedforwardIndex - POSITIONS_PER_MSEC];
		}
	}
	// CTRL-16661: Calculate feedforward while under drive control
	else if ((HandshakeMoveActive[axisnum]) && (flag_DriveGenMotionType[axisnum] != DRIVEMOTION_OSCILLATE))
	{
		feedForwardOutput += PrimaryLoopCommandDerivative[axisnum] * GainVff[axisnum] * ServoRateKhz;

		// If we are decelerating, use the GainDff value instead of GainAff
		if ((GainDff[axisnum] > 0) && ((PrimaryLoopCommandDerivative2[axisnum] * PrimaryLoopCommandDerivative[axisnum]) < 0))
		{
			AccelGain = GainDff[axisnum];
		}

		feedForwardOutput += PrimaryLoopCommandDerivative2[axisnum] * AccelGain;
	}
#endif
#else
	feedForwardOutput += (float)FeedforwardVelocity[axisnum] * GainVff[axisnum];
	feedForwardOutput += (float)FeedforwardAccel[axisnum] * GainAff[axisnum];
	 
#endif

#if !defined(_CLS_) && !defined(_FLS_) && !defined(_GCL_)
	// CTRL-10078: Add position feedforward contribution
	// CTRL-26712: Piezo drives should maintain the position component of feedforward at all times.
#if !defined(FEATURE_PIEZO)
	if (AxisIsHomed[axisnum])
#endif
	{
		feedForwardOutput += (FeedforwardPosition[axisnum] - GainPffOffset[axisnum]) * GainPff[axisnum];
	}
#endif

	// Return the feedforward contribution to the servo output
	return (feedForwardOutput);
}



#if defined(FEATURE_ENHANCED_THROUGHPUT)
#pragma CODE_SECTION(EnhancedThroughput, ".internalCode")
static inline float EnhancedThroughput(unsigned int axisnum)
{
	float EnhancedThroughputCurrent;

#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

	// Calculate the contribution from the ETM
	EnhancedThroughputCurrent = AnalogIn[EnhancedThroughputChannel[axisnum]] - EnhancedThroughputOffset[axisnum];
	EnhancedThroughputCurrent *= EnhancedThroughputGain[axisnum];

	// Clamp the current to the specified limit
	if (EnhancedThroughputCurrentClamp[axisnum] > 0)
	{
		if (EnhancedThroughputCurrent > EnhancedThroughputCurrentClamp[axisnum])
		{
			EnhancedThroughputCurrent = EnhancedThroughputCurrentClamp[axisnum];
		}
		else if (EnhancedThroughputCurrent < -EnhancedThroughputCurrentClamp[axisnum])
		{
			EnhancedThroughputCurrent = -EnhancedThroughputCurrentClamp[axisnum];
		}
	}

	// Return the ETM contribution to the servo output
	// The units of the value calculated above are Amps, so we need to re-scale
	// the value based on the maximum current of the amplifier.
	return (EnhancedThroughputCurrent * RecipAmpMaxCurrent[axisnum]);
}
#endif



#pragma CODE_SECTION(EnhancedTrackingControl, ".internalCode")
static FILTER_FP EnhancedTrackingControl(unsigned int axisnum, SERVO_FP etcFilterInput)
{
	// Temporary variables to help compiler optimizations
	FILTER_FP etcFilterOutput, etcDelayOutput;

#if (MAX_NUM_AXES == 1)
	// Overwrite the axis number parameter so the compiler will not perform array index calculations.
	axisnum = 0;
#endif

	// Calculate Enhanced Tracking Control filter output
	etcFilterOutput  = etcFilterCoeffN0[axisnum] * etcFilterInput + etcFilterCoeffN1[axisnum] * etcFilterInHist0[axisnum];
	etcFilterOutput += etcFilterCoeffN2          * etcFilterInHist1[axisnum];
	etcFilterOutput -= etcFilterCoeffD1[axisnum] * etcFilterOutHist0[axisnum];
	etcFilterOutput -= etcFilterCoeffD2[axisnum] * etcFilterOutHist1[axisnum];

	// Time shift input and output samples
	etcFilterInHist1[axisnum]  = etcFilterInHist0[axisnum];
	etcFilterInHist0[axisnum]  = etcFilterInput;
	etcFilterOutHist1[axisnum] = etcFilterOutHist0[axisnum];
	etcFilterOutHist0[axisnum] = etcFilterOutput;

	// Scale the filter output and add to ServoOutput
	ServoOutput[axisnum] += (etcFilterOutput * etcFilterGain[axisnum]);

	// Calculate Enhanced Tracking Control "delay filter" output
	etcDelayOutput  = etcDelayCoeffN0 * ServoOutput[axisnum] + etcDelayCoeffN1 * etcDelayInHist0[axisnum];
	etcDelayOutput += etcDelayCoeffN2 * etcDelayInHist1[axisnum];
	etcDelayOutput -= etcDelayCoeffD1 * etcDelayOutHist0[axisnum];
	etcDelayOutput -= etcDelayCoeffD2 * etcDelayOutHist1[axisnum];

	// Time shift input and output samples
	etcDelayInHist1[axisnum]  = etcDelayInHist0[axisnum];
	etcDelayInHist0[axisnum]  = ServoOutput[axisnum];
	etcDelayOutHist1[axisnum] = etcDelayOutHist0[axisnum];
	etcDelayOutHist0[axisnum] = etcDelayOutput;

	// Integrate the filter output unless current is clamped
	// CTRL-21111: Only stop integrating if clamped in the direction we are trying to integrate
#if !defined(COMPILING_FIRMWARE2)
	if ((ServoOutputClamped[axisnum] * etcDelayOutput) <= 0)
#else
	if (!ServoOutputClamped[axisnum])
#endif
	{
		etcIntegral[axisnum] += etcDelayOutput;
	}

	// Scale the integrator output and add to ServoOutput
	return (etcIntegral[axisnum] * etcIntegralGain[axisnum]);
}



#pragma CODE_SECTION(FilterServoOutput, ".internalCode")
static inline void FilterServoOutput(void)
{
	unsigned int axisnum = 0;
	unsigned int filterNum;

	// calculate each of the filters sequentially
	for (filterNum = 0; filterNum < NUM_SERVO_FILTERS; filterNum++)
	{
		// check to see if the filter is enabled for any axis
		if (flag_filtEnabled & (0x1 << filterNum))
		{
			// calculate the output of the second order filter
			LOOP_OVER_ALL_AXES
			{
				// temporary variable to help compiler optimizations
				FILTER_FP filtOut;

				// calculate filter output
				filtOut  = (FILTER_FP)ServoFilterCoeffN0[filterNum][axisnum] * ServoOutput[axisnum] + (FILTER_FP)ServoFilterCoeffN1[filterNum][axisnum] * FiltInHist0[filterNum][axisnum];
				filtOut += (FILTER_FP)ServoFilterCoeffN2[filterNum][axisnum] * FiltInHist1[filterNum][axisnum];
				filtOut -= (FILTER_FP)ServoFilterCoeffD1[filterNum][axisnum] * FiltOutHist0[filterNum][axisnum];
				filtOut -= (FILTER_FP)ServoFilterCoeffD2[filterNum][axisnum] * FiltOutHist1[filterNum][axisnum];

				// time shift input and output samples
				FiltInHist1[filterNum][axisnum]  = FiltInHist0[filterNum][axisnum];
				FiltInHist0[filterNum][axisnum]  = ServoOutput[axisnum];
				FiltOutHist1[filterNum][axisnum] = FiltOutHist0[filterNum][axisnum];
				FiltOutHist0[filterNum][axisnum] = filtOut;

				// only use the output if the filter is enabled on this axis
				if (ServoFilterSetup[axisnum] & (0x1 << filterNum))
				{
					ServoOutput[axisnum] = filtOut;
				}
			}
		}
	}
}



#endif

#endif
