

//  *** Excerpts from "MutualInductance_Docs" directory ***
//
//
//  The mutual inductance M of two coupled inductances L1 and L2 is equal to the mutually induced 
//  voltage in one inductance divided by the rate of change of current in the other inductance:
//  M = E2m / (di1/dt)
//  M = E1m / (di2/dt) 
//
//  If the self induced voltages of the inductances L1 and L2 are respectively E1s and E2s 
//  for the same rates of change of the current that produced the mutually induced 
//  voltages E1m and E2m, then:
//  M = (E2m / E1s)L1
//  M = (E1m / E2s)L2
//  Combining these two equations:
//  M = (E1mE2m / E1sE2s) (L1L2) = kM(L1L2)
//  where kM is the mutual coupling coefficient of the two inductances L1 and L2. 
//
//  If the coupling between the two inductances L1 and L2 is perfect, then the 
//  mutual inductance M is:
//  M = (L1L2)  
//
//
//
//  Additionally, from "Transformer Circuits.mht", pertaining to a description of transformer with
//  applied voltage "Vp" on the primary terminals and load resistance "Rload" on the secondary 
//  terminals.....
//
//     Vp = Ip*Rp + Lp*dIp/dt - M*dIs/dt   -> (viewed form primary)
//
//     M*dIp/dt = Is*(Rs + Rload) + Ls*dIs/dt -> (viewed form secondary)
//
//     So, the "group" differencial equations are
//
//
//                      Lp*dIp/dt - M*dIs/dt = Vp - Ip*Rp
//     
//                      M*dIp/dt - Ls*dIs/dt = Is*(Rs + Rload)
//
//
//     Now, lets model the three phase common-mode/differcial-mode motor output filter:                 
//
//
//
//                                                                    ----- vm_a
//                                                                   |
//     vs_a ---<CM-inductor phase a>------------<Inductor phase a>-----------------<Ra>---<Ca>---
//                      |                                                                        |
//                      |                                                                        |
//                      |                                             ----- vm_b                 |
//                      |                                            |                           |             
//     vs_b ---<CM-inductor phase b>------------<Inductor phase b>-----------------<Rb>---<Cb>---|-----
//                      |                                                                        |     |
//                      |                                                                        |    GND
//                      |                                             ----- vm_c                 |
//                      |                                            |                           |             
//     vs_c ---<CM-inductor phase b>------------<Inductor phase c>-----------------<Rc>---<Cc>---
//
//
//     vs_a = Lm_a*dia/dt  + M*dib/dt + M*dic/dt + L_a*dia/dt + Ra*ia + Integral(ia*dt)/Ca                 (1)
//
//     vs_b = M*dia/dt + Lm_b*dib/dt + M*dic/dt + L_b*dib/dt + Rb*ib + Integral(ib*dt)/Cb                  (2)
//
//     vs_c = M*dia/dt + M*dib/dt + Lm_c*dic/dt + L_c*dic/dt + Rc*ic + Integral(ic*dt)/Cc                  (3)
//
//     (Notice, unlike the AC motor model, the mutual inductance have "+" signs).
//
//                Let 
//
//                    ia = dqa/dt        (representing "i" as the "change in charge per change in time")
//  
//                    ib = dqb/dt
//
//                    ic = dqc/dt
//
//
//                Then
//  
//                    dia/dt = d2qa/dt2
// 
//                    dib/dt = d2qb/dt2
//
//                    dic/dt = d2qc/dt2
//
//
//
//      Transforming (1), (2), and (3)..
//
//
//      vs_a = Lm_a*d2qa/dt2 + Mab*d2qb/dt2 + Mac*d2qc/dt2 + L_a*d2qa/dt2 + Ra*dqa/dt + qa/Ca        (4)
//      
//      vs_b = Mab*d2qa/dt2 + Lm_b*d2qb/dt2 + Mbc*d2qc/dt2 + L_b*d2qb/dt2 + Rb*dqb/dt + qb/Cb        (5)
//
//      vs_c = Mac*d2qa/dt2 + Mbc*d2qb/dt2 + Lm_c*d2qc/dt2 + L_c*d2qc/dt2 + Rc*dqc/dt + qc/Cc        (6)
//
//
//                 Let 
//
//                   dqa/dt = qqa                                                                (7a)
//
//                   dqb/dt = qqb                                                                (7b)                                      
//
//                   dqc/dt = qqc                                                                (7c)
//
//                 Then
//
//                   d2qa/dt = dqqa/dt
//
//                   d2qb/dt = dqqb/dt
//
//                   d2qc/dt = dqqc/dt
//
//
//       Transforming (4), (5), and (6)
//
//       vs_a = Lm_a*dqqa/dt + Mab*dqqb/dt + Mac*dqqc/dt + L_a*dqqa/dt + Ra*dqa/dt + qa/Ca          (8)
//
//       vs_b = Mab*dqqa/dt + Lm_b*dqqb/dt + Mbc*dqqc/dt + L_b*dqqb/dt + Rb*dqb/dt + qb/Cb          (9)
//
//       vs_c = Mac*dqqa/dt + Mbc*dqqb/dt + Lm_c*dqqc/dt + L_c*dqqc/dt + Rc*dqc/dt + qc/Cc          (10)
//
//
//
//       The ODE equations are (from 7a,7b,7c and group equations 8,9,10)
//
//
//       dqa/dt = qqa                                                                       (11a, class Qa)                             
//
//       dqb/dt = qqb                                                                       (11b, class Qb)                                
//
//       dqc/dt = qqc                                                                       (11c, class Qc)     
//
//       (Lm_a + L_a)*dqqa/dt + Mab*dqqb/dt + Mac*dqqc/dt  = vs_a - Ra*qqa - qa/Ca          (12, class QQa_QQb_QQc_1)
//
//       Mab*dqqa/dt + (Lm_b + L_b)*dqqb/dt + Mbc*dqqc/dt = vs_b - Rb*qqb - qb/Cb           (13, class QQa_QQb_QQc_2)
//
//       Mac*dqqa/dt + Mbc*dqqb/dt + (Lm_c + L_c)*dqqc/dt = vs_c - Rc*qqc - qc/Cc           (14, class QQa_QQb_QQc_3)
//
//
//
//
//
//
//       **** Comparisons between "App_CMChoke.hpp" and "App_CMChoke.cir"  ****
//
//         
//            Analysis was done near 3.6 mSec point on plot.
//
//            We are pretty close all around. 
//
//            It is important to keep in mind that our simulation controls source
//            "slew-rate" where as the PSpice model seems to produce an infinite slope
//            on the input sources "VBus_a", "VBus_b" and "VBus_c".
//
//            C_a, C_b, C_c, VLm_a, VLm_b, and VLm_c track almost exactly.
//
//    



typedef enum {
  ODE_FUNC_NULL = -1,
  ODE_FUNC_Qa,
  ODE_FUNC_Qb,
  ODE_FUNC_Qc,
  ODE_FUNC_QQa_QQb_QQc_1,
  ODE_FUNC_QQa_QQb_QQc_2,
  ODE_FUNC_QQa_QQb_QQc_3,
  
} ODE_FUNCTION;


typedef enum {
  CTRL_FUNC_NULL = -1,
  CTRL_FUNC_RefA,
  CTRL_FUNC_RefB,
  CTRL_FUNC_RefC,


} CTRL_FUNCTION;


typedef enum {
  SRC_FUNC_NULL = -1,
  SRC_FUNC_VSrc3Phase,

} SRC_FUNCTION;


#include "Simulation.hpp" 


//Parameters....

#define Mab  .098e-3          
#define Mbc  Mab
#define Mac  Mab

#define Lm_a  .1e-3               
#define Lm_b  Lm_a
#define Lm_c  Lm_a

#define La .1e-3            
#define Lb La
#define Lc La

#define Ra 10.0
#define Rb Ra
#define Rc Ra

#define Ca .3e-6  
#define Cb Ca
#define Cc Ca

#define QUARTER_PWM_CYCLE .001         
#define PWM_GAIN 5.0

#define DC_BUS_VOLTAGE 100.0

#define SRC_SLEW_RATE 50000000.0  //volts/sec

// **** SrcObject Classes ********************************

// ---- VSrc3Phase ---------------------------------------

class VSrc3Phase : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void OdeRValueUpdate(void);
  VSrc3Phase(void);
  ~VSrc3Phase(void);
  double t_mod;
  double t_prev;
  double PwmRampDir;
  



};


VSrc3Phase Inst_VSrc3Phase;


// --------------------------------------------------------

SrcObject * SrcObjectList[] = {(SrcObject *) &Inst_VSrc3Phase,
			       0};


// ***********************************************************

// **** OdeObject Classes ********************************

// ---- Qa -----------------------------------------

class Qa : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  Qa(void);
  ~Qa(void);
  //source for this ODE
  double qqa;

};

Qa Inst_Qa;


// --------------------------------------------------------

// ---- Qb -----------------------------------------

class Qb : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  Qb(void);
  ~Qb(void);
  //source for this ODE
  double qqb;

};

Qb Inst_Qb;


// --------------------------------------------------------

// ---- Qc -----------------------------------------

class Qc : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  Qc(void);
  ~Qc(void);
  //source for this ODE
  double qqc;

};

Qc Inst_Qc;


// --------------------------------------------------------

// ---- QQa_QQb_QQc_1 -----------------------------------------

class QQa_QQb_QQc_1 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  virtual void OdeGroupMSolve(double dydt[],  double dmdt[]);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  QQa_QQb_QQc_1(void);
  ~QQa_QQb_QQc_1(void);
  //source for this ODE
  double vs_a;
  double Ref_a;
  double Pwm_sig;
  double qqa;
  double qa;
  //storage for probes...
  vector<double> Vcm_a;    //for expression  "vs_a - L_a*dqqa - Ra*qqa - qa/Ca"        
  vector<double> C_a;      //for expression "qa/Ca"
  vector<double> VBus_a;   //"vs_a" (processed DC_BUS_VOLTAGE or - DC_BUS_VOLTAGE depending on reference value)
  vector<double> VLm_a;    //for expression "L_a*dqqa"
  vector<double> VRa;      //for expression "Ra*qqa"
};

QQa_QQb_QQc_1 Inst_QQa_QQb_QQc_1;


// --------------------------------------------------------

// ---- QQa_QQb_QQc_2 -----------------------------------------

class QQa_QQb_QQc_2 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  QQa_QQb_QQc_2(void);
  ~QQa_QQb_QQc_2(void);
  //source for this ODE
  double vs_b;
  double Ref_b;
  double Pwm_sig;
  double qqb;
  double qb;
  //storage for probes...
  vector<double> Vcm_b;    //for expression  "vs_b - L_b*dqqb - Rb*qqb - qb/Cb"        
  vector<double> C_b;      //for expression "qb/Cb"  
  vector<double> VBus_b;   //"vs_b" (processed DC_BUS_VOLTAGE or - DC_BUS_VOLTAGE depending on reference value)
  vector<double> VLm_b;    //for expression "L_b*dqqb"
  vector<double> VRb;      //for expression "Rb*qqb"
  
};

QQa_QQb_QQc_2 Inst_QQa_QQb_QQc_2;


// --------------------------------------------------------


// ---- QQa_QQb_QQc_3 -----------------------------------------

class QQa_QQb_QQc_3 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  QQa_QQb_QQc_3(void);
  ~QQa_QQb_QQc_3(void);
  //source for this ODE
  double vs_c;
  double Ref_c;
  double Pwm_sig;
  double qqc;
  double qc;
  //storage for probes...
  vector<double> Vcm_c;    //for expression  "vs_c - L_c*dqqc - Rc*qqc - qc/Cc"        
  vector<double> C_c;      //for expression "qc/Cc"    
  vector<double> VBus_c;   //"vs_c" (processed DC_BUS_VOLTAGE or - DC_BUS_VOLTAGE depending on reference value)
  vector<double> VLm_c;    //for expression "L_c*dqqc"
  vector<double> VRc;      //for expression "Rc*qqc"  

};

QQa_QQb_QQc_3 Inst_QQa_QQb_QQc_3;


// --------------------------------------------------------



OdeObject * OdeObjectList[] = {(OdeObject *) &Inst_Qa,
			       (OdeObject *) &Inst_Qb,
			       (OdeObject *) &Inst_Qc,
			       (OdeObject *) &Inst_QQa_QQb_QQc_1, //the next three are in a group and must be maintained in this order.
			       (OdeObject *) &Inst_QQa_QQb_QQc_2,
			       (OdeObject *) &Inst_QQa_QQb_QQc_3,
			       0};





// ***********************************************************



// **** CtrlObject Classes (Translation) *********************

// ---- RefA  ---------------------------------------------

class RefA : public CtrlObject
{
public:
  virtual void CtrlFunction(double t);
  virtual void OdeRValueUpdate(void);
  RefA(void);
  ~RefA(void);

};



RefA Inst_RefA;

// -------------------------------------------------------------

// ---- RefB  ---------------------------------------------

class RefB : public CtrlObject
{
public:
  virtual void CtrlFunction(double t);
  virtual void OdeRValueUpdate(void);
  RefB(void);
  ~RefB(void);

};



RefB Inst_RefB;

// -------------------------------------------------------------


// ---- RefC  ---------------------------------------------

class RefC : public CtrlObject
{
public:
  virtual void CtrlFunction(double t);
  virtual void OdeRValueUpdate(void);
  RefC(void);
  ~RefC(void);

};



RefC Inst_RefC;

// -------------------------------------------------------------



CtrlObject * CtrlTranObjectList[] = {(CtrlObject *) &Inst_RefA,
				     (CtrlObject *) &Inst_RefB,
				     (CtrlObject *) &Inst_RefC,
				     0};

// **************************************************************

// **** CtrlObject Classes (Trajectory) **************************

// "CtrlTrajObjectList[]" added here.
// ***************************************************************


// **** SrcObject Functions **************************************

// ---- VSrc3Phase  ----------------------------------------------------

VSrc3Phase::VSrc3Phase(void)
{
 
  OdeObjItem * pCurOdeItem;
 

  SrcFuncName = SRC_FUNC_VSrc3Phase;


  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_1];
  pCurOdeItem->pNextOdeItem = new OdeObjItem;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_2];
  pCurOdeItem->pNextOdeItem = new OdeObjItem;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_3];



  PwmRampDir = 1.0;

  //  PlotThisOutput = TRUE;
  Plot_Tag = "VSrc";
}

VSrc3Phase::~VSrc3Phase(void)
{



}

void VSrc3Phase::SrcFunction(double t)
{
  t_mod += (t - t_prev);
  t_prev = t;

  if(t_mod > QUARTER_PWM_CYCLE){
    t_mod -=  QUARTER_PWM_CYCLE;
    PwmRampDir *= -1.0;
  }

  y = PWM_GAIN*(t_mod/QUARTER_PWM_CYCLE - .5)*PwmRampDir;

}

void VSrc3Phase::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((QQa_QQb_QQc_1 *) pCurOdeItem->pOdeObject)->Pwm_sig= y;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;    
  ((QQa_QQb_QQc_2 *) pCurOdeItem->pOdeObject)->Pwm_sig= y;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;    
  ((QQa_QQb_QQc_3 *) pCurOdeItem->pOdeObject)->Pwm_sig= y;
 

}




// -----------------------------------------------------------------

// ***************************************************************


// **** OdeObject Functions **************************************

// ---- Qa  ----------------------------------------------------

Qa::Qa(void)
{

  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_Qa;
  

  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;          
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_1];

  qqa = 0;
  
  //  PlotThisOutput = TRUE;
  Plot_Tag = "Qa";

}

Qa::~Qa(void)
{

}

double Qa::OdeFunction(double y, double t)
{
  return(qqa);
}

void Qa::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((QQa_QQb_QQc_1 *) pCurOdeItem->pOdeObject)->qa = y;
      


}

// -----------------------------------------------------------------



// ---- Qb  ----------------------------------------------------

Qb::Qb(void)
{

  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_Qb;
  

  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;          
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_2];

  qqb = 0;

}


Qb::~Qb(void)
{

}

double Qb::OdeFunction(double y, double t)
{
  return(qqb);
}

void Qb::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((QQa_QQb_QQc_2 *) pCurOdeItem->pOdeObject)->qb = y;

}

// -----------------------------------------------------------------


// ---- Qc  ----------------------------------------------------



Qc::Qc(void)
{

  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_Qc;
  

  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;          
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_3];

  qqc = 0;

}
Qc::~Qc(void)
{

}

double Qc::OdeFunction(double y, double t)
{
  return(qqc);
}

void Qc::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((QQa_QQb_QQc_3 *) pCurOdeItem->pOdeObject)->qc = y;

}

// -----------------------------------------------------------------

// ---- QQa_QQb_QQc_1  ----------------------------------------------------

QQa_QQb_QQc_1::QQa_QQb_QQc_1(void)
{
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_QQa_QQb_QQc_1;
 
  

  //this object marks the beginning of a "group solve" of
  //three objects.
  GroupSolve = 1;
  NumberOfGrpOdes = 3;

  //build the ODE Rvalue list.
 
  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_Qa];


  Ref_a = 0;
  Pwm_sig = 0;
  qqa = 0;
  qa = 0;

  PlotThisOutput = TRUE;
  Plot_Tag = "QQa";
  DoProbes = TRUE;
  
}

QQa_QQb_QQc_1::~QQa_QQb_QQc_1(void)
{

}

double QQa_QQb_QQc_1::OdeFunction(double y, double t)
{
  //Note: This is within a "group" ODE so we are really solving for "dm/dt"

          //"vs_a - Ra*qqa - qa/Ca" 
  vs_a = ShapeSquareWaveSource(((Ref_a > Pwm_sig) ? DC_BUS_VOLTAGE : - DC_BUS_VOLTAGE), SRC_SLEW_RATE ,t);
  return(vs_a - Ra*y - qa/Ca);
}

void QQa_QQb_QQc_1::RecordProbes(void)
{
  //value for "vs_a - L_a*dqqa - Ra*qqa - qa/Ca"
  Vcm_a.push_back(vs_a - La*dydt - Ra*qqa - qa/Ca);
  C_a.push_back(qa/Ca);
  VBus_a.push_back(vs_a);
  VLm_a.push_back(La*dydt);   
  VRa.push_back(Ra*qqa);      

}

void QQa_QQb_QQc_1 ::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Vcm_a, "Vcm_a");
    SimuPlot.plot_xy(Plot_t, C_a, "C_a");
    SimuPlot.plot_xy(Plot_t, VBus_a, "VBus_a");
    SimuPlot.plot_xy(Plot_t, VLm_a, "VLm_a");   
    SimuPlot.plot_xy(Plot_t, VRa, "VRa");  
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Vcm_a"){
	SimuPlot.plot_xy(Plot_t, Vcm_a, "Vcm_a");
      }
      else if(TagNamesToPlot[i] == "C_a"){
	SimuPlot.plot_xy(Plot_t, C_a, "C_a");
      }
      else if(TagNamesToPlot[i] == "VBus_a"){
	SimuPlot.plot_xy(Plot_t, VBus_a, "VBus_a");
      }
      else if(TagNamesToPlot[i] == "VLm_a"){
	SimuPlot.plot_xy(Plot_t, VLm_a, "VLm_a");
      }
      else if(TagNamesToPlot[i] == "VRa"){
	SimuPlot.plot_xy(Plot_t, VRa, "VRa");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }

  }
}


void QQa_QQb_QQc_1::OdeGroupMSolve(double dydt[], double dmdt[])
{
  //Note: This instance is the first of the group. As such, this
  //      is where the "group solve" of the individual 
  //      "dy/dt's" is performed.

  //
  //   We must solve for dqqa/dt, dqqb/dt, dqqc/t in (12), (13), and (14)
  //   where the right side of these equations is passed to us, the values
  //   in "dmdt[]". The "dydt[]" array hold the results.
  //
  //
  //   Let
  //         dqqa = dqqa/dt, dqqb = dqqb/dt and dqqc = dqqc/dt
  //         
  //
  //       (Lm_a + L_a)*dqqa + Mab*dqqb + Mac*dqqc  = dmdt[0]    (a)     
  //
  //       Mab*dqqa + (Lm_b + L_b)*dqqb + Mbc*dqqc = dmdt[1]     (b)     
  //
  //       Mac*dqqa + Mbc*dqqb + (Lm_c + L_c)*dqqc = dmdt[2]     (c)
  //
  //
  //   Solve for dqqa in (b) and (c) and subtract (c) from (b)
  //
  //       dqqa = (- (Lm_b + L_b)*dqqb - Mbc*dqqc + dmdt[1])/Mab  
  //
  //       dqqa = (- Mbc*dqqb - (Lm_c + L_c)*dqqc + dmdt[2])/Mac
  //
  //       (- (Lm_b + L_b)*dqqb - Mbc*dqqc + dmdt[1])/Mab = (- Mbc*dqqb - (Lm_c + L_c)*dqqc + dmdt[2])/Mac
  //
  //       (- Lm_b/Mab - L_b/Mab + Mbc/Mac)*dqqb + (Lm_c/Mac + L_c/Mac - Mbc/Mab)*dqqc = - dmdt[1]/Mab + dmdt[2]/Mac    (d)
  //
  //
  //
  //
  //
  //
  //   Solve for dqqa in (a) and (b) and subtract (b) from (a)
  //
  //       dqqa = (- Mab*dqqb - Mac*dqqc + dmdt[0])/(Lm_a + L_a)
  //
  //       dqqa = (- (Lm_b + L_b)*dqqb - Mbc*dqqc + dmdt[1])/Mab  
  //
  //       (- Mab*dqqb - Mac*dqqc + dmdt[0])/(Lm_a + L_a) = (- (Lm_b + L_b)*dqqb - Mbc*dqqc + dmdt[1])/Mab 
  //
  //       (- Mab/(Lm_a + L_a) + (Lm_b + L_b)/Mab)*dqqb + (- Mac/(Lm_a + L_a) + Mbc/Mab)*dqqc = - dmdt[0]/(Lm_a + L_a) + dmdt[1]/Mab  (e)
  //
  //
  //       Simplify (d) and (e)   
  //
  //       A1*dqqb + B1*dqqc = C1                 (f)           
  //
  //       A2*dqqb + B2*dqqc = C2                 (g)
  //
  //                 Where...
  //
  //                          A1 = (- Lm_b/Mab - L_b/Mab + Mbc/Mac)
  //                          B1 = (Lm_c/Mac + L_c/Mac - Mbc/Mab)
  //                          C1 = - dmdt[1]/Mab + dmdt[2]/Mac
  //                          A2 = (- Mab/(Lm_a + L_a) + (Lm_b + L_b)/Mab)
  //                          B2 = (- Mac/(Lm_a + L_a) + Mbc/Mab)
  //                          C2 = - dmdt[0]/(Lm_a + L_a) + dmdt[1]/Mab 
  //                          
  //
  //       Solve for dqqb and dqqc, (g) into (f) 
  //
  //       A1*dqqb + B1*(C2 - A2*dqqb)/B2 = C1
  //
  //       A1*dqqb - B1*A2*dqqb/B2 = C1 - B1*C2/B2
  //
  //       dqqb = (C1 - B1*C2/B2)/(A1 - B1*A2/B2) = (B2*C1 - B1*C2)/(B2*A1 - B1*A2)     (h)
  //
  //       And (h) into (g)
  //
  //       A2*(B2*C1 - B1*C2)/(B2*A1 - B1*A2) + B2*dqqc = C2
  //
  //       dggc = (C2 - A2*(B2*C1 - B1*C2)/(B2*A1 - B1*A2))/B2 = (C2 - A2*dqqb)/B2      (i)
  //
  //       (h) and (i) into (a) and solve for dqqa
  //
  //       dqqa = (- Mab*dqqb - Mac*dqqc + dmdt[0])/(Lm_a + L_a)                        (j)
  //
  //
  //
  //
  
  
#define A1 (- Lm_b/Mab - Lb/Mab + Mbc/Mac)
#define B1 (Lm_c/Mac + Lc/Mac - Mbc/Mab)
#define A2 (- Mab/(Lm_a + La) + (Lm_b + Lb)/Mab) 
#define B2 (- Mac/(Lm_a + La) + Mbc/Mab) 
 


  double dqqa;
  double dqqb;
  double dqqc;


  double C1 = - dmdt[1]/Mab + dmdt[2]/Mac;
  double C2 = - dmdt[0]/(Lm_a + La) + dmdt[1]/Mab; 


  dqqb = (B2*C1 - B1*C2)/(B2*A1 - B1*A2);
  dqqc = (C2 - A2*dqqb)/B2;
  dqqa = (- Mab*dqqb - Mac*dqqc + dmdt[0])/(Lm_a + La);

  dydt[0] = dqqa;
  dydt[1] = dqqb;
  dydt[2] = dqqc;

}

void QQa_QQb_QQc_1::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((Qa *) pCurOdeItem->pOdeObject)->qqa = y;
  
}

// -----------------------------------------------------------------


// ---- QQa_QQb_QQc_2  ----------------------------------------------------


QQa_QQb_QQc_2::QQa_QQb_QQc_2(void)
{
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_QQa_QQb_QQc_2;
 
  
  //build the ODE Rvalue list.
 
  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_Qb];


  Ref_b = 0;
  Pwm_sig = 0;
  qqb = 0;
  qb = 0;

  PlotThisOutput = TRUE;
  Plot_Tag = "QQb";
  DoProbes = TRUE;
  
}




QQa_QQb_QQc_2::~QQa_QQb_QQc_2(void)
{

}

double QQa_QQb_QQc_2::OdeFunction(double y, double t)
{
  //Note: This is within a "group" ODE so we are really solving for "dm/dt"

           //"vs_b - Rb*qqb - qb/Cb"
  vs_b = ShapeSquareWaveSource(((Ref_b > Pwm_sig) ? DC_BUS_VOLTAGE : - DC_BUS_VOLTAGE), SRC_SLEW_RATE ,t);
  return(vs_b - Rb*y - qb/Cb);
}


void QQa_QQb_QQc_2::RecordProbes(void)
{

  //value for "vs_b - L_b*dqqb - Rb*qqb - qb/Cb"
  Vcm_b.push_back(vs_b - Lb*dydt - Rb*qqb - qb/Cb);
  C_b.push_back(qb/Cb);
  VBus_b.push_back(vs_b);
  VLm_b.push_back(Lb*dydt);   
  VRb.push_back(Rb*qqb);    
}

void QQa_QQb_QQc_2 ::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Vcm_b, "Vcm_b");
    SimuPlot.plot_xy(Plot_t, C_b, "C_b");
    SimuPlot.plot_xy(Plot_t, VBus_b, "VBus_b");
    SimuPlot.plot_xy(Plot_t, VLm_b, "VLm_b");   
    SimuPlot.plot_xy(Plot_t, VRb, "VRb");  
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Vcm_b"){
	SimuPlot.plot_xy(Plot_t, Vcm_b, "Vcm_b");
      }
      else if(TagNamesToPlot[i] == "C_b"){
	SimuPlot.plot_xy(Plot_t, C_b, "C_b");
      }
      else if(TagNamesToPlot[i] == "VBus_b"){
	SimuPlot.plot_xy(Plot_t, VBus_b, "VBus_b");
      }
      else if(TagNamesToPlot[i] == "VLm_b"){
	SimuPlot.plot_xy(Plot_t, VLm_b, "VLm_b");
      }
      else if(TagNamesToPlot[i] == "VRb"){
	SimuPlot.plot_xy(Plot_t, VRb, "VRb");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }

  }
}


void QQa_QQb_QQc_2::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((Qb *) pCurOdeItem->pOdeObject)->qqb = y;

}





// -----------------------------------------------------------------


// ---- QQa_QQb_QQc_3  ----------------------------------------------------



QQa_QQb_QQc_3::QQa_QQb_QQc_3(void)
{
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_QQa_QQb_QQc_3;
 
  
  //build the ODE Rvalue list.
 
  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_Qc];


  Ref_c = 0;
  Pwm_sig = 0;
  qqc = 0;
  qc = 0;

  PlotThisOutput = TRUE;
  Plot_Tag = "QQc";
  DoProbes = TRUE;
}



QQa_QQb_QQc_3::~QQa_QQb_QQc_3(void)
{

}


double QQa_QQb_QQc_3::OdeFunction(double y, double t)
{
  //Note: This is within a "group" ODE so we are really solving for "dm/dt"

                //"vs_c - Rc*qqc - qc/Cc"
  vs_c = ShapeSquareWaveSource(((Ref_c > Pwm_sig) ? DC_BUS_VOLTAGE : - DC_BUS_VOLTAGE), SRC_SLEW_RATE ,t);
  return(vs_c - Rc*y - qc/Cc);
}

void QQa_QQb_QQc_3::RecordProbes(void)
{
  //value for "vs_c - L_c*dqqc - Rc*qqc - qc/Cc"
  Vcm_c.push_back(vs_c - Lc*dydt - Rc*qqc - qc/Cc);
  C_c.push_back(qc/Cc);
  VBus_c.push_back(vs_c);
  VLm_c.push_back(Lc*dydt);   
  VRc.push_back(Rc*qqc);   
}

void QQa_QQb_QQc_3 ::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Vcm_c, "Vcm_c");
    SimuPlot.plot_xy(Plot_t, C_c, "C_c");
    SimuPlot.plot_xy(Plot_t, VBus_c, "VBus_c");
    SimuPlot.plot_xy(Plot_t, VLm_c, "VLm_c");   
    SimuPlot.plot_xy(Plot_t, VRc, "VRc");  
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Vcm_c"){
	SimuPlot.plot_xy(Plot_t, Vcm_c, "Vcm_c");
      }
      else if(TagNamesToPlot[i] == "C_c"){
	SimuPlot.plot_xy(Plot_t, C_c, "C_c");
      }
      else if(TagNamesToPlot[i] == "VBus_c"){
	SimuPlot.plot_xy(Plot_t, VBus_c, "VBus_c");
      }
      else if(TagNamesToPlot[i] == "VLm_c"){
	SimuPlot.plot_xy(Plot_t, VLm_c, "VLm_c");
      }
      else if(TagNamesToPlot[i] == "VRc"){
	SimuPlot.plot_xy(Plot_t, VRc, "VRc");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }

  }
}

void QQa_QQb_QQc_3::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((Qc *) pCurOdeItem->pOdeObject)->qqc = y;

}





// -----------------------------------------------------------------

// ******************************************************************

// ***** CtrlObject Functions (Translation) ************************

// ---- RefA ---------------------------------------------------

RefA::RefA(void)
{
  OdeObjItem * pCurOdeItem;

  CtrlFuncName = CTRL_FUNC_RefA;

  
  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_1];  

}

RefA::~RefA(void)
{


}


void RefA::CtrlFunction(double t)
{
  //For now, just command a constant of "0"
  y =.23; // .32;  
}

void RefA::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((QQa_QQb_QQc_1 *) pCurOdeItem->pOdeObject)->Ref_a = y;

}


// -----------------------------------------------------------------

// ---- RefB ---------------------------------------------------


RefB::RefB(void)
{
  OdeObjItem * pCurOdeItem;

  CtrlFuncName = CTRL_FUNC_RefB;

  
  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_2];  

}


RefB::~RefB(void)
{


}


void RefB::CtrlFunction(double t)
{
  //For now, just command a constant of "0"
  y = 0;
}

void RefB::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((QQa_QQb_QQc_2 *) pCurOdeItem->pOdeObject)->Ref_b = y;


}

// ----------------------------------------------------------------- 

// ---- RefC ---------------------------------------------------

RefC::RefC(void)
{
  OdeObjItem * pCurOdeItem;

  CtrlFuncName = CTRL_FUNC_RefC;

  
  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_QQa_QQb_QQc_3];  

}

RefC::~RefC(void)
{


}


void RefC::CtrlFunction(double t)
{
  //For now, just command a constant of "0"
  y = 0;
}

void RefC::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((QQa_QQb_QQc_3 *) pCurOdeItem->pOdeObject)->Ref_c = y;


}

// ----------------------------------------------------------------- 





// *****************************************************************

// ***** CtrlObject Functions (Trajectory) *************************

// *****************************************************************



// **** SimuObject ************************************************




SimuObject Simulation;

SimuObject::SimuObject(void)
{
  OdeObjItem * pCurOdeItem;
  SrcObjItem * pCurSrcItem;
  CtrlObjItem * pCurCtrlItem;
  int i;

  //The simulation run time.
  SimuTime = .01;        

  //set the plot time range.
  MinimumPlotTime = 0;
  MaximumPlotTime = SimuTime;

  //For this example the setting of "RelTol" is important. If it is set to large
  //the system will become unstable. 
  RelTol =  .0000000001;
  //Setting "AbsTol" too small will make a system with a decaying oscillation 
  //take too long to finish.
  AbsTol =  .0000000000001; 
  //typical error correction setting as per "RungeKutta_java.htm" (never set higher then "1.0")
  Safety = .98;
  h_start = .000000001; 

  h = h_start;

  //set to a value other then "0" to clamp maximum "h";
  h_max = .0000002;

  //set to store and plot "h" parameter.
  //  Do_h_Plot = 1;
  
  TimeQuantum = h;
  CtrlTimeQuantum = .005;

  TranslationQuantumCount = 1;
  TranslationQuantumNum = 1;
  
  CtrlTimeAccumulator = 0;

  SrcPeriodQuantum = QUARTER_PWM_CYCLE;
  SrcPeriodAccumulator = 0;
 

  //build the SRC equation list
  pSrcEquationList = new SrcObjItem;
  pCurSrcItem = pSrcEquationList;
  i = 0;
  while(SrcObjectList[i]){
    pCurSrcItem->pSrcObject = SrcObjectList[i];
    i++;
    if(!SrcObjectList[i])
      break;
    pCurSrcItem->pNextSrcItem = new SrcObjItem;
    pCurSrcItem = pCurSrcItem->pNextSrcItem;


  }

  //build the ODE equation list.
  pOdeEquationList = new OdeObjItem;
  pCurOdeItem = pOdeEquationList;
  i = 0;
  while(1){
    pCurOdeItem->pOdeObject = OdeObjectList[i];
    i++;
    if(!OdeObjectList[i])
      break;
    pCurOdeItem->pNextOdeItem = new OdeObjItem;
    pCurOdeItem = pCurOdeItem->pNextOdeItem;
  }

  //build the CTRL (Translation) equation list.
  pTranslationList = new CtrlObjItem;
  pCurCtrlItem = pTranslationList;
  i = 0;
  while(1){
    pCurCtrlItem->pCtrlObject = CtrlTranObjectList[i];
    i++;
    if(!CtrlTranObjectList[i])
      break;
    pCurCtrlItem->pNextCtrlItem = new CtrlObjItem;
    pCurCtrlItem = pCurCtrlItem->pNextCtrlItem;
  }

 

 

}


SimuObject::~SimuObject(void)
{

  //(ideally we should delete all make allocation make in the constructor, here)

}



// ************************************************************





bool ExecuteSimulation(void)
{

   

  Simulation.OdeSimuType = ODE_SIMU_56;

  while(Simulation.t < Simulation.SimuTime){               

    if(!Simulation.DoOneInteration())
      return FALSE;
    
  }
  return TRUE;
}

void PlotResults(string TagNamesToPlot[], double ScaleFactors[], double MinPlotTimes[], double MaxPlotTimes[])
{
  Simulation.PlotSimuResults(TagNamesToPlot, ScaleFactors, MinPlotTimes, MaxPlotTimes);


}
