





//The numerical methods for solving ordinary differential equations are methods of integrating a system 
//of first order differential equations, since higher order ordinary differential equations can be reduced
// to a set of first order ODE's. For example, 
//
//     From "Numerical Solutions of Ordinary Differential Equations.mht"....
//
//     p(x)*d2y/dx2 + q(x)*dy/dx = r(x) 
//
//
//     Let y(x) = y1(x) and dy1/dx = y2(x)
//
//
//
//         dy1/dx = y2(x)
//
//
//         dy2/dx = (r(x) - q(x)*y2(x))/p(x)
//
//
//     
//
//     From "http://en.wikipedia.org/wiki/RLC_circuit"   ....
//
//
//     For an LRC series circuit with an applied voltage v(t)
//
//     vL(t) + vR(t) + vC(t) = v(t)
//
//
//     R*i(t) + L*di/dt + (1/C)*Integral(i(t))*dt = v(t)
//
//
//     To keep v(t) unchanged in transformation, use relationship
//
//
//     i(t) = dq/dt   (change of "charge" per change in time)
//
//
//     Thus,
//
//
//           L*d2q/dt2 + R*dq/dt + (1/C)q(t) = v(t)    (Equation A)  
//
//
//           d2q/dt2 + (R/L)*dq/dt + (1/LC)*q(t) = (1/L)*v(t)
//
//
//     This will be a linear system so p(x), q(x) of first equation, will be constant.
//     (don't confuse q(x) in first equation with q(t) in second equation)
//
//     Now, let
//     q(t)  = q1(t) and dq1/dt = q2(t)
//     (NOTE: q1 and q2 are NEW variables, with no relationship to charge "q" above)
//
//
//     dq1/dt = q2(t)                                  (Equation 1)
//
//     dq2/dt = (v(t) - R*q2(t) - (1/C)*q1(t))/L       (Equation 2)
//
//
//     with i = dq1/dt = q2(t)
//
//
//     Again, from "http://en.wikipedia.org/wiki/RLC_circuit" and (Equation A) above
//
//
//     v(s) = (L*s**2 + R*s + (1/C))*q(s)
//
//
//     If we go back in terms "q" (charge),
//
//
//      q(s)  =  v(s) / (L*s**2 + R*s + (1/C))
//
//         or
//
//      q(s)  =  v(s) / (L*(s**2 + (R/L)*s + (1/(L*C))))
//
//              with s = a + j*w
//
//      There is a zero at s = 0 and s = infinity
//
//
//      The poles are determined by the "quadratic formula"   s = (- b +/- (b**2 - 4*a*c)**.5)/(2*a)
//
//
//      Let 2*Y = R/L and  Wo**2 = 1/(L*C) above. This is done only to simplify the quadratic formula above
//
//
//         s = - (2*Y) +/- ((2*Y)**2 - 4*Wo**2)**.5  / 2
//
//         s = - (2*Y) +/- ((2)**2*Y**2 - (2)**2*Wo**2)**.5 / 2
//
//
//         s = - Y +/- (Y**2 - Wo**2)**.5
//
//
//
//         Critical damped is when,
//
//
//          Y**2 = (R/(2*L))**2 = 1/(L*C)
//
//          R = ((1 / (L*C))**.5) * (2*L)
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//



typedef enum {
  ODE_FUNC_NULL = -1,
  ODE_FUNC_TestRLC1,
  ODE_FUNC_TestRLC2,
  ODE_FUNC_TestRLC3,
 
  
} ODE_FUNCTION;


typedef enum {
  CTRL_FUNC_NULL = -1,
  CTRL_FUNC_VSrc,


} CTRL_FUNCTION;

typedef enum {
  SRC_FUNC_NULL = -1,


} SRC_FUNCTION;


#include "Simulation.hpp"


//Parameters....

#define L  ((double) .001)
#define C  ((double) .000001)

                //This setting for "R" by nature of the calculations above
                //gives us critical damping. (Turns out to be approx. 63.0 Ohms)
//#define R  (sqrt(1 / (L*C))*(2*L)) 

#define R  ((double) 30.0)

                //This setting for "R" will give us an under-damped system.
                //The following calculations dictates that the frequency period
                //of oscillation is:
                //
                //
                //    Wo = 2*PI*Fo = sqrt(1/(L*C)) = sqrt(1/(.001*.000001))
                //
                //          To = 1.986e-4  (approx. .0002 seconds which is very
                //                          close to what we measure here)
                //
                //

     //un-comment this definition to verify "FSFCtrl_2.m" and "FSFCtrl_3.m"  analysis and the information provided in
     //"Design of state-feedback control systems(2).mht" and "Design of state-feedback control systems(3).mht"
     //of sub-directory "./FSFCtrl-Octave-Tests"
     //( **** Don't forget to define "R = 30.0" above for these tests ****)
#define DO_FSFCtrl_2_SIMUALTION
//#define DO_FSFCtrl_3_SIMUALTION   //(NOTE: As of 12/5/08, "FSFCtrl_3.m" is incomplete.)

// **** SrcObject Classes ********************************

//"SrcObject * SrcObjectList[]" added here.

// ***********************************************************

// **** OdeObject Classes ********************************

// ---- TestRLC1 -----------------------------------------
class TestRLC1 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double h);
  virtual void OdeRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);

  TestRLC1(void);
  ~TestRLC1(void);
  //source for this ODE
  double q2;
  //storage for probes...
  vector<double> vc;   //voltage accross capacitor C

};



TestRLC1  Inst_TestRLC1;

// --------------------------------------------------------

// ---- TestRLC2 -----------------------------------------

class TestRLC2 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double h);
  virtual void OdeRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
 
  TestRLC2(void);
  ~TestRLC2(void);
      //source for this ODE
  double q1;
  double v_src;
  double u;
  //(with q2 == "y" of this ODE)
  //storage for probes...
  vector<double> Ry_L;    //for expression "R*y/L"
  vector<double> q1_CL;   //for expression "q1/(C*L)"   
 
};



TestRLC2  Inst_TestRLC2;

// ----------------------------------------------------------


// ---- TestRLC3 -----------------------------------------

class TestRLC3 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double h);
  virtual void PostOdeFunction(double t);
  virtual void OdeRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
 
  TestRLC3(void);
  ~TestRLC3(void);
      //source for this ODE
  double w;    //(here "w" is our new input)
  double q1;
  double q2;
  double v;
  double u;
  double K[2];
  double y_out;    //(here "y_out" is an actual scaler derived from "Ct*q", NOT the output of the ODE)
  double Ct[2]; //(here "Ct" is a 2-element row vector)
  //(with e == "y" of this ODE)
  //storage for probes...
  vector<double> u_sig;    //(here "u_sig" equals u = "K*q - v*e", where "K" is a 2-element row vector
                          //"q" is a 2-element column vector, "v" and "e" are scalers)
  vector<double> y_out_sig;  //(this is "y_out")
};



TestRLC3  Inst_TestRLC3;

// ----------------------------------------------------------






OdeObject * OdeObjectList[] = {(OdeObject *) &Inst_TestRLC1,
			       (OdeObject *) &Inst_TestRLC2,
			       (OdeObject *) &Inst_TestRLC3,
			       0};

// ***********************************************************


// **** CtrlObject Classes (Translation) *********************

// ---- TestCtrl ---------------------------------------------

class TestCtrl : public CtrlObject
{
public:
  virtual void CtrlFunction(double t);
  virtual void OdeRValueUpdate(void);
  TestCtrl(void);
  ~TestCtrl(void);

};



TestCtrl Inst_TestCtrl;

// -------------------------------------------------------------

CtrlObject * CtrlTranObjectList[] = {(CtrlObject *) &Inst_TestCtrl,
				     0};

// **************************************************************

// **** CtrlObject Classes (Trajectory) **************************

// "CtrlTrajObjectList[]" added here.

// ***************************************************************


// **** SrcObject Functions **************************************

// ***************************************************************


// **** OdeObject Functions **************************************

// ---- TestRLC1 (Equation 1) -------------------------------------


TestRLC1::TestRLC1(void)
{
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_TestRLC1;

  //build the ODE Rvalue list.
  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC2];
  pCurOdeItem->pNextOdeItem = new OdeObjItem;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC3];


  q2 = 0;

  DoProbes = TRUE;

}

TestRLC1::~TestRLC1(void)
{

}

double TestRLC1::OdeFunction(double y, double h)
{
  return(q2);
}

void TestRLC1::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;
 
  ((TestRLC2 *) pCurOdeItem->pOdeObject)->q1 = y;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;
  ((TestRLC3 *) pCurOdeItem->pOdeObject)->q1 = y;
 

}

void TestRLC1::RecordProbes(void)
{
 
  //store these expressions for this interation.
  //(vector "C" is infered to equal [1/C  0])
  vc.push_back(y/C);


}

void TestRLC1::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{   

  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, vc, "vc");
  
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "vc"){
	SimuPlot.plot_xy(Plot_t, vc, "vc");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }

}





// -----------------------------------------------------------------



// ----  TestRLC2 (Equation 2) -------------------------------------

TestRLC2::TestRLC2(void)
{
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_TestRLC2;

 
  //build the ODE Rvalue list.
  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC1];
  pCurOdeItem->pNextOdeItem = new OdeObjItem;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC3];

  q1 = 0;
  v_src = 0;
  u = 0;

  PlotThisOutput = TRUE;
  DoProbes = TRUE;
  Plot_Tag = "i";

}

TestRLC2::~TestRLC2(void)
{

}

double TestRLC2::OdeFunction(double y, double h)
{    //(vector "B" is infered to equal [0 1/L]' with matrix A equal [0 1; (-1/L*C) (-R/L)])
#ifdef DO_FSFCtrl_2_SIMUALTION
  return((u - R*y - q1/C)/L);
#else
#ifdef DO_FSFCtrl_3_SIMUALTION
  return((u - R*y - q1/C)/L);
#else
  return((v_src - R*y - q1/C)/L);
#endif
#endif 
}

void TestRLC2::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((TestRLC1 *) pCurOdeItem->pOdeObject)->q2 = y;
  pCurOdeItem = pCurOdeItem->pNextOdeItem;
  ((TestRLC3 *) pCurOdeItem->pOdeObject)->q2 = y;


 

}


void TestRLC2::RecordProbes(void)
{
 
  //store these expressions for this interation.
  Ry_L.push_back((R*y/L)/1000);
  q1_CL.push_back((q1/(C*L)/10000));

}

void TestRLC2::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{   

  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Ry_L, "R*y/L");
    SimuPlot.plot_xy(Plot_t, q1_CL, "q1/(C*L)");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "R*y/L"){
	SimuPlot.plot_xy(Plot_t, Ry_L, "R*y/L");
      }
      else if(TagNamesToPlot[i] == "q1/(C*L)"){
	SimuPlot.plot_xy(Plot_t, q1_CL, "q1/(C*L)");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }

}

// -----------------------------------------------------------------

// ----  TestRLC3 (Equation 3) -------------------------------------

TestRLC3::TestRLC3(void)
{
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_TestRLC3;

 

  //build the ODE Rvalue list.
  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC2];

  q1 = 0;
  q2 = 0;
  w = 0;


#ifdef DO_FSFCtrl_2_SIMUALTION
  //Form "FSFCtrl_2.m" for L = .001, C = .000001, and R = 30.0
  K[0] = 0;
  K[1] = 33.24555;  //(Ideal value should have been 33.246)
  v = 1;

#endif 


#ifdef DO_FSFCtrl_3_SIMUALTION
  //Form "FSFCtrl_3.m" for L = .001, C = .000001, and R = 30.0
  //(*** To be determined ***)
  K[0] = 0;
  K[1] = 0;  
  v = 0;
#endif 
  
  Ct[0] = 1.0/C;
  Ct[1] = 0;

  PlotThisOutput = TRUE;
  DoProbes = TRUE;
  Plot_Tag = "e";

}




TestRLC3::~TestRLC3(void)
{

}

double TestRLC3::OdeFunction(double y, double h)
{
#ifdef DO_FSFCtrl_3_SIMUALTION
  //this is "de/dt"
  return(w - y_out); 
#endif 
}

void TestRLC3::PostOdeFunction(double t)
{
#ifdef DO_FSFCtrl_2_SIMUALTION
  //this is u = - K*q + v*w ("u" now becomes "v_src" of "TestRLC2")
  u = - (K[0]*q1 + K[1]*q2) + v*w;
#else
#ifdef DO_FSFCtrl_3_SIMUALTION
  //this is y = Ct*q   ("q" is "x" in "FSFCtrl_3.m" and "Design of state-feedback control systems(3).mht"
  y_out = Ct[0]*q1 + Ct[1]*q2;
  //this is u = - K*q - v*e ("u" now becomes "v_src" of "TestRLC2")
  u = - (K[0]*q1 + K[1]*q2) - v*y;
#endif 
#endif

}

  
   

void TestRLC3::OdeRValueUpdate(void)
{


  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;
  //"u" now becomes "v_src" of "TestRLC2" for the closed loop test.
  ((TestRLC2 *) pCurOdeItem->pOdeObject)->u = u;



}

void TestRLC3::RecordProbes(void)
{

 
  //store these expressions for this interation.
  u_sig.push_back(u);
  y_out_sig.push_back(y_out);



}



void TestRLC3::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{   

  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, u_sig, "u_sig");
    SimuPlot.plot_xy(Plot_t, y_out_sig, "y_out_sig");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "u_sig"){
	SimuPlot.plot_xy(Plot_t, u_sig , "u_sig");
      }
      else if(TagNamesToPlot[i] == "y_out_sig"){
	SimuPlot.plot_xy(Plot_t, y_out_sig, "y_out_sig");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }

}



// *****************************************************************



      



// ***** CtrlObject Functions (Translation) ************************


// ---- TestCtrl ---------------------------------------------------

TestCtrl::TestCtrl(void)
{
  OdeObjItem * pCurOdeItem;

  CtrlFuncName = CTRL_FUNC_VSrc;
  
 //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

#ifdef DO_FSFCtrl_2_SIMUALTION
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC3];
#else
#ifdef DO_FSFCtrl_3_SIMUALTION
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC3];
#else
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_TestRLC2];
#endif
#endif 



 

  y = 50;

  PlotThisOutput = TRUE;
#ifdef DO_FSFCtrl_2_SIMUALTION
  Plot_Tag = "w";
#else
#ifdef DO_FSFCtrl_3_SIMUALTION
  Plot_Tag = "w";
#else
  Plot_Tag = "v_src";
#endif
#endif 


}

TestCtrl::~TestCtrl(void)
{


}


void TestCtrl::CtrlFunction(double t)
{
  if((t >= .003) && (t < .006))
    y = 0;
  else if(t >= .006)
    y = 50;
}

void TestCtrl::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

#ifdef DO_FSFCtrl_2_SIMUALTION
  ((TestRLC3 *) pCurOdeItem->pOdeObject)->w = y;
#else
#ifdef DO_FSFCtrl_3_SIMUALTION
  ((TestRLC3 *) pCurOdeItem->pOdeObject)->w = y;
#else
  ((TestRLC2 *) pCurOdeItem->pOdeObject)->v_src = y;
#endif
#endif 


}


// -----------------------------------------------------------------



// *****************************************************************

// ***** CtrlObject Functions (Trajectory) *************************

// *****************************************************************




// **** SimuObject ************************************************


SimuObject Simulation;

SimuObject::SimuObject(void)
{
  OdeObjItem * pCurOdeItem;
  SrcObjItem * pCurSrcItem;
  CtrlObjItem * pCurCtrlItem;
  int i;

  //The simulation run time.
  SimuTime = .01;   

  //set the plot time range.
  MinimumPlotTime = 0;
  MaximumPlotTime = SimuTime;  

  //For this example the setting of "RelTol" is important. If it is set to large
  //the system will become unstable. 
  RelTol =  .0000000001;

  //Setting "AbsTol" too small will make a system with a decaying oscillation 
  //take too long to finish.
  AbsTol =  .0000000000001;


  //typical error correction setting as per "RungeKutta_java.htm" (never set higher then "1.0")
  Safety = .98;
  h_start = .000000001; 

  h = h_start;

  //set to a value other then "0" to clamp maximum "h";
  h_max = .0000002;
  
  TimeQuantum = h;
  CtrlTimeQuantum = .003;

  TranslationQuantumCount = 1;
  TranslationQuantumNum = 1;
  
  CtrlTimeAccumulator = 0;

  //no source is used here.
  SrcPeriodQuantum = SimuTime;
  SrcPeriodAccumulator = 0;



  //build the ODE equation list.
  pOdeEquationList = new OdeObjItem;
  pCurOdeItem = pOdeEquationList;
  i = 0;
  while(1){
    pCurOdeItem->pOdeObject = OdeObjectList[i];
    i++;
    if(!OdeObjectList[i])
      break;
    pCurOdeItem->pNextOdeItem = new OdeObjItem;
    pCurOdeItem = pCurOdeItem->pNextOdeItem;
  }



  //build the CTRL (Translation) equation list.
  pTranslationList = new CtrlObjItem;
  pCurCtrlItem = pTranslationList;
  i = 0;
  while(1){
    pCurCtrlItem->pCtrlObject = CtrlTranObjectList[i];
    i++;
    if(!CtrlTranObjectList[i])
      break;
    pCurCtrlItem->pNextCtrlItem = new CtrlObjItem;
    pCurCtrlItem = pCurCtrlItem->pNextCtrlItem;
  } 

 

}


SimuObject::~SimuObject(void)
{

  //(ideally we should delete all make allocation make in the constructor, here)

}



// ************************************************************






bool ExecuteSimulation(void)
{

 
 

  Simulation.OdeSimuType = ODE_SIMU_56;

  while(Simulation.t < Simulation.SimuTime){ 

    if(!Simulation.DoOneInteration()){
      return FALSE;

    } 
  
    
  }
  return TRUE;
}

void PlotResults(string TagNamesToPlot[], double ScaleFactors[], double MinPlotTimes[], double MaxPlotTimes[],
		 double PlotTimeSteps[], double PlotTimeOffset)
{
  Simulation.PlotSimuResults(TagNamesToPlot, ScaleFactors, MinPlotTimes, MaxPlotTimes, PlotTimeSteps, PlotTimeOffset);


}
