

#include <math.h>
#include <stdio.h>

#define PI   3.1415926535897932384626433832795

#define FILTER_TAP_COUNT   (32 + 1)

    //Run sigma delta at 10 MHz
#define SIGMA_DELTA_TIME_SAMPLE  (0.0000001)

    //With 10 MHz, we should get 16 bit resolution with and update of 51.2 uS using a Sinc2 topology according to Table 28.3 of C2000 TRM.
#define OSR         256

#define DAC_REFERENCE 30.0       //NOTE: This is set relative to "ia" peak attaining a maximum +/- 25 amps on accel/decel.

#define Fc (10000)      //Cannot exceed .5 * Fs
#define Fs (1.0 / SIGMA_DELTA_TIME_SAMPLE)  // 10 MHz

#define CUTOFF  (Fc/Fs)

//If defined:
//      > gnuplot
//
//          gnuplot> plot './coefficients_dump.data' using 1 with lines
//
//#define DEBUG_GEN_COEFFICIENTS
#ifdef DEBUG_GEN_COEFFICIENTS
FILE * fp_debug_gen_coefficients;
#endif
void GenerateSincCoefficient(double pSincCoefficient[FILTER_TAP_COUNT], int TapCountMinusOne)
{
    int k;
    double w = 1.0;
    double coef_sum = 0;
	
#ifdef DEBUG_GEN_COEFFICIENTS
    fp_debug_gen_coefficients = fopen("coefficients_dump.data", "w");
    if(fp_debug_gen_coefficients == NULL)
    {
        printf("Can't open coefficients_dump.data\n");
        exit(-1);
    }
#endif

        //Reference:
        //
        //
        //  https://en.wikipedia.org/wiki/Sinc_function
        //  https://www.musicdsp.org/en/latest/Filters/194-windowed-sinc-fir-generator.html  (see genSinc() function and Cutoff definition)
        //  https://www.analog.com/media/en/technical-documentation/dsp-book/dsp_book_Ch16.pdf
        //  https://tomroelandts.com/articles/how-to-create-a-simple-low-pass-filter


    for(k = - TapCountMinusOne/2; k <= TapCountMinusOne/2; k++)
    {

            // Compute Blackman window. (NOTE: With Blackman window turned on, a .5 constant yields a result of 0.499991 (no dither at .000001 resolution))
        w = 0.42 - 0.5 * cos(2.0 * PI * ((double) (TapCountMinusOne / 2) + k) / (double) (TapCountMinusOne)) +
            0.08 * cos(4.0 * PI * ((double) (TapCountMinusOne / 2) + k) / (double) (TapCountMinusOne));

        if(k == 0)
        {
            pSincCoefficient[TapCountMinusOne/2] = w *  2.0 * CUTOFF;

        }
        else
        {
            pSincCoefficient[k + TapCountMinusOne/2] = w * 1.0 * sin(2.0* CUTOFF * PI * ((double) k)) /
                                                            (PI * ((double) k));

        }

        coef_sum += pSincCoefficient[k + TapCountMinusOne/2];



    }

    for(k = - TapCountMinusOne/2; k <= TapCountMinusOne/2; k++)
    {
        pSincCoefficient[k + TapCountMinusOne/2] = pSincCoefficient[k + TapCountMinusOne/2] / coef_sum;

#ifdef DEBUG_GEN_COEFFICIENTS
        fprintf(fp_debug_gen_coefficients, "%g\n", pSincCoefficient[k + TapCountMinusOne/2]);
#endif

    }

#ifdef DEBUG_GEN_COEFFICIENTS
    fclose(fp_debug_gen_coefficients);
#endif

}

//If defined:
//      > gnuplot
//
//          gnuplot> plot './sigma_delta_integrator_dump.data' using 1 with lines
//
//#define DEBUG_SIGMA_DELTA_INTEGRATOR
#ifdef DEBUG_SIGMA_DELTA_INTEGRATOR
FILE * fp_debug_sigma_delta_integrator;
double debug_sigma_delta_inegrator_t_val;
#endif


double SecondOrderSigmaDelta(double x_in, unsigned int phase)
{
    static double integrator_1[3] = {0};
    static double integrator_2[3] = {0};
    static double x_out[3] = {0};
    static double x_out_dac[3] = {0};
    double sum_1;


        //https://en.wikipedia.org/wiki/Delta-sigma_modulation  (Basic 2nd order implementation)
        //https://classes.engr.oregonstate.edu/eecs/spring2021/ece627/Lecture%20Notes/2nd%20&%20Higher-Order2.pdf  <== "2".

    sum_1 = integrator_2[phase];

        //NOTE: May use a +- deadband instead of "0"?
    if(sum_1 > 0)
    {
        x_out[phase] =  DAC_REFERENCE;
        x_out_dac[phase] =  DAC_REFERENCE;
    }
    else if(sum_1 < 0)
    {
        x_out[phase] =  - DAC_REFERENCE;
        x_out_dac[phase] =  - DAC_REFERENCE;
    }

#ifdef DEBUG_SIGMA_DELTA_INTEGRATOR
    if(debug_sigma_delta_inegrator_t_val < .02)
    {

        fprintf(fp_debug_sigma_delta_integrator, "%g\t%g\n", sum_1, x_out[phase]);


    }
    else if(fp_debug_sigma_delta_integrator)
    {

        fclose(fp_debug_sigma_delta_integrator);

        fp_debug_sigma_delta_integrator = 0;
    }
#endif



            //From "2" above, we add the the lead 'z' to make the system more stable.
    integrator_2[phase] += ((integrator_1[phase] - x_out_dac[phase] + (x_in - x_out_dac[phase])));

    integrator_1[phase] += (x_in - x_out_dac[phase]);








    return x_out[phase];



}

double ExecuteOneSincFIRIneration(double y_in, double pSincCoefficient[FILTER_TAP_COUNT], double pSincSampleState[3][FILTER_TAP_COUNT], unsigned int TapCount, unsigned int phase)
{
    int i;
    double y_out = 0;

    for(i = 0; i < TapCount; i++)
    {
        y_out += pSincSampleState[phase][i] * pSincCoefficient[i];
    }

    for(i = TapCount - 1; i > 0; i--)
    {
        pSincSampleState[phase][i] = pSincSampleState[phase][i-1];
    }

    pSincSampleState[phase][0] = y_in;




    return y_out;
}

double DoSigmaDelta2ndOrderConversion(double i_phase,
                                      double pSincCoefficient[FILTER_TAP_COUNT],
                                      double pSincSampleState_0[3][FILTER_TAP_COUNT],
                                      double pSincSampleState_1[3][FILTER_TAP_COUNT],
                                      double pSincSampleState_2[3][FILTER_TAP_COUNT],
                                      unsigned int TapCount,
                                      unsigned int OverSamplingRatio,
                                      unsigned int phase)
{
//    static unsigned int osr_count[3] = {0};
    static double sigma_delta_filtered_current_0;
    static double sigma_delta_filtered_current_1;
    static double sigma_delta_filtered_current_2;
    static double sigma_delta_current;

    double sigma_delta_filtered_current_3;

    sigma_delta_filtered_current_3 = sigma_delta_filtered_current_2;

    sigma_delta_filtered_current_2 = ExecuteOneSincFIRIneration(sigma_delta_filtered_current_1 , pSincCoefficient, pSincSampleState_2, TapCount, phase);

    sigma_delta_filtered_current_1 = ExecuteOneSincFIRIneration(sigma_delta_filtered_current_0 , pSincCoefficient, pSincSampleState_1, TapCount, phase);

    sigma_delta_filtered_current_0 = ExecuteOneSincFIRIneration(sigma_delta_current, pSincCoefficient, pSincSampleState_0, TapCount, phase);

    sigma_delta_current = SecondOrderSigmaDelta(i_phase, phase);
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

            //Comment out line 220 above and un-comment below for testing filters.

                            //OK, it looks like we have the filters working correctly. However, the Sigma Delta modulator is suspect!!!!
                            //               Find a better model to compare with?
//    static int toggle;
//
//    if(toggle == 0)
//    {
//        sigma_delta_current = .4;
//        toggle = 1;
//    }
//    else if(toggle == 1)
//    {
//        sigma_delta_current = 0;
//         toggle = 2;
//    }
//    else if(toggle == 2)
//    {
//          sigma_delta_current = 0;
//        toggle = 3;
//    }
//    else if(toggle == 3)
//    {
//        sigma_delta_current = 0;
//        toggle = 4;
//    }
//    else if(toggle == 4)
//    {
//        sigma_delta_current = 0;
//        toggle = 5;
//    }
//    else if(toggle == 5)
//    {
//        sigma_delta_current = 0;
//        toggle = 0;
//    }
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!







    return sigma_delta_filtered_current_3;
}





double SincSampleState_0[3][FILTER_TAP_COUNT];
double SincSampleState_1[3][FILTER_TAP_COUNT];
double SincSampleState_2[3][FILTER_TAP_COUNT];

double SincCoefficient[FILTER_TAP_COUNT];

double TimeSample;


//If defined:
//      > gnuplot
//
//          gnuplot> plot './sigma_delta_phase_a.data' using 0:1 with lines, '' using 0:2 with lines
//
//#define DEBUG_SIGMA_DELTA_PHASE_A
#ifdef DEBUG_SIGMA_DELTA_PHASE_A
FILE * fp_debug_sigma_delta_phase_a;
#endif

    //Called from Ia_Ib_Ic_1() constructor.
void InitializeSigmaDeltaFunction(void)
{

    GenerateSincCoefficient(SincCoefficient, FILTER_TAP_COUNT - 1);

#ifdef DEBUG_SIGMA_DELTA_PHASE_A
    fp_debug_sigma_delta_phase_a = fopen("sigma_delta_phase_a.data", "w");
    if(fp_debug_sigma_delta_phase_a == NULL)
    {
        printf("Can't open sigma_delta_phase_a.data\n");
        exit(-1);
    }
#endif




#ifdef DEBUG_SIGMA_DELTA_INTEGRATOR
    fp_debug_sigma_delta_integrator = fopen("sigma_delta_integrator_dump.data", "w");
    if(fp_debug_sigma_delta_integrator == NULL)
    {
        printf("Can't open sigma_delta_integrator_dump.data\n");
        exit(-1);
    }
#endif

}






    //Called in function Ia_Ib_Ic_1::OdeFunction() of App_Observer_Disturbance_Rejection_C2000.hpp
void ExecuteSigmaDeltaPhaseA(double ia, double * ia_sigma_delta, double t)
{
    static double t_prev = 0;
    static double t_mod = 0;
//    static unsigned int osr_count = 0;

    t_mod += (t - t_prev);
    t_prev = t;


    //Un-comment for constant test
//    ia = .1;

        //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        //!!!!! STOP:  I think we are OK!
        //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    if(t_mod > SIGMA_DELTA_TIME_SAMPLE)
    {
        * ia_sigma_delta = DoSigmaDelta2ndOrderConversion(ia, SincCoefficient, SincSampleState_0, SincSampleState_1, SincSampleState_2, FILTER_TAP_COUNT, OSR, 0);
        t_mod -= SIGMA_DELTA_TIME_SAMPLE;

#ifdef DEBUG_SIGMA_DELTA_PHASE_A
        if(t < .02)
        {
//            if(osr_count == OSR)
//            {
                fprintf(fp_debug_sigma_delta_phase_a, "%g\t%g\n", * ia_sigma_delta, ia);
//                osr_count = 0;
//            }
 //           else
 //           {
//                osr_count++;
//            }
        }
        else if(fp_debug_sigma_delta_phase_a)
        {

            fclose(fp_debug_sigma_delta_phase_a);

            fp_debug_sigma_delta_phase_a = 0;
        }
#endif
    }

#ifdef DEBUG_SIGMA_DELTA_INTEGRATOR
        //Past "t" up for debug
    debug_sigma_delta_inegrator_t_val = t;

#endif


}






