// ToyBoxFDTDbezhig, version 1.0
//      An if-I-can-do-it-you-can-do-it FDTD! 
// Copyright (C) 1998, 1999 Laurie E. Miller, Paul Hayes, Matthew O'Keefe 

// This program is free software; you can redistribute it and/or 
//     modify it under the terms of the GNU General Public License 
//     as published by the Free Software Foundation; either version 2
//     of the License, or any later version, with the following conditions
//     attached in addition to any and all conditions of the GNU
//     General Public License:
//     When reporting or displaying any results or animations created
//     using this code or modification of this code, make the appropriate
//     citation referencing ToyBoxFDTDbezhig by name and including the 
//     version number.  
//
// This program is distributed in the hope that it will be useful,
//     but WITHOUT ANY WARRANTY; without even the implied warranty 
//     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//     See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//     along with this program; if not, write to the Free Software
//     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  
//     02111-1307  USA

// Contacting the authors:
//
// Laurie E. Miller, Paul Hayes, Matthew O'Keefe
// Department of Electrical and Computer Engineering
//      200 Union Street S. E.
//      Minneapolis, MN 55455
//
// lemiller@lcse.umn.edu
// info@cemtach.org
// 
// http://cemtach.org
// http://cemtach.org/software/ToyBoxFDTD/ToyBoxFDTD.html


// MEA: This file is derived from the modified version of "ToyBoxFDTDbezhig.cpp" found in 
//      /home/maiello/Development/FDTDDevelopment/FDTDDevelopment-Linux/VolumnRendering
//
//






#include <math.h> 
#include <stdio.h>         
#include <float.h>
#include <malloc.h>
#include <stdlib.h>

// Only one selection allowed at a time...

     // The original plane-wave test
#define ORIGINAL_PLANE_WAVE_TEST

     // Un-comment if we are simulating a guassian pulsed electrical current flow through a thin conductor running in the "x" direction
//#define SIMULATE_Y_CONDUCTING_ELEMENT_TEST

     // Un-comment if we are simulating a guassian ramped (continuous) electrical current flow through a thin conductor running in the "x" direction
//#define SIMULATE_Y_CONDUCTING_ELEMENT_TEST_CONTINUOUS




#ifdef ORIGINAL_PLANE_WAVE_TEST

// constants that turn on/off the new features
//#define PMC
          // If uncommented, this line includes the PMC part of the code at 
          //     compile time.  If this line is commented out, the open sides of
          //     the waveguide revert to PEC and the guide becomes rectangular
          //     rather than parallel-plate.
//#define SINUSOIDAL_PULSE_STIMULUS
          // If uncommented, this line includes the sinusoidal pulse as the 
          //     stimulus for the simulation.  If this line is commented out,
          //     the stimulus will be a simplified continuous plane wave. 



//#define FIXED_SCALING_VALUE 100.0   //(This is my static override of the original scaling mechanisms listed above)

// program control constants

#define FREQUENCY 10.0e9
          // frequency of the stimulus in Hertz
#define GUIDE_WIDTH 0.0229
          // meters -- section of the width of the guide to be simulated
          //    -- the parallel plate waveguide is actually infinite in width
#define GUIDE_HEIGHT 0.0102
          // interior height of the guide in meters
#define LENGTH_IN_WAVELENGTHS 4.5
          // length of the waveguide in wavelengths of the stimulus


#endif 


#ifdef SIMULATE_Y_CONDUCTING_ELEMENT_TEST


#define FREQUENCY 10.0e9
          // frequency of the stimulus in Hertz
#define GUIDE_WIDTH 0.04
          // meters -- section of the width of the guide to be simulated
          //    -- the parallel plate waveguide is actually infinite in width
#define GUIDE_HEIGHT 0.04
          // interior height of the guide in meters
#define LENGTH_IN_WAVELENGTHS 2.0
          // length of the waveguide in wavelengths of the stimulus

#endif 

#ifdef SIMULATE_Y_CONDUCTING_ELEMENT_TEST_CONTINUOUS


#define FREQUENCY 10.0e9
          // frequency of the stimulus in Hertz
#define GUIDE_WIDTH 0.04
          // meters -- section of the width of the guide to be simulated
          //    -- the parallel plate waveguide is actually infinite in width
#define GUIDE_HEIGHT 0.04
          // interior height of the guide in meters
#define LENGTH_IN_WAVELENGTHS 2.0
          // length of the waveguide in wavelengths of the stimulus

#endif 



// physical constants
#define LIGHT_SPEED     299792458.0       
          // speed of light in a vacuum in meters/second
#define LIGHT_SPEED_SQUARED 89875517873681764.0        
          // m^2/s^2
#define MU_0            (M_PI*4.0e-7)         
          // permeability of free space in henry/meter
#define EPSILON_0       (1.0/(MU_0*LIGHT_SPEED_SQUARED))      
          // permittivity of free space in farad/meter






/////////////////////////////////////////////////////////////////////////////
// variable declarations (file local)
static int i,j,k;     
           // indices of the 3D array of cells
static int allocatedBytes = 0;          
           // a counter to track number of bytes allocated
static int iteration = 0;          
           // counter to track how many timesteps have been computed
static double stimulus = 0.0;       
           // value of the stimulus at a given timestep
static  double currentSimulatedTime = 0.0;
           // time in simulated seconds that the simulation has progressed
static  double totalSimulatedTime = 0.0;       
           // time in seconds that will be simulated by the program
static  double omega;       
           // angular frequency in radians/second
static  double lambda;       
           // wavelength of the stimulus in meters
static  double dx, dy, dz; 
           // space differentials (or dimensions of a single cell) in meters
static  double dt;
           // time differential (how much time between timesteps) in seconds
static  double dtmudx, dtepsdx;       
           // physical constants used in the field update equations 
static  double dtmudy, dtepsdy;       
           // physical constants used in the field update equations 
static  double dtmudz, dtepsdz;       
           // physical constants used in the field update equations 


// variable declarations (global)
double ***ex, ***ey, ***ez;  
           // pointers to the arrays of ex, ey, and ez values
double ***hx, ***hy, ***hz;
           // pointers to the arrays of hx, hy, and hz values
int nx, ny, nz;         
           // total number of cells along the x, y, and z axes, respectively

double ***Jx, ***Jy, ***Jz;           
unsigned char *** PEC;
   
// **** FEM version of original FDTD algorythm ****************************************************************************************************


// time differential (how much time between timesteps) in seconds
static  double dtmudx_cd, dtepsdx_cd;
// physical constants used in the field update equations
static  double dtmudy_cd, dtepsdy_cd;
// physical constants used in the field update equations
static  double dtmudz_cd, dtepsdz_cd;
// physical constants used in the field update equations

#define max(a,b) \
   ({ __typeof__ (a) _a = (a); \
       __typeof__ (b) _b = (b); \
     _a > _b ? _a : _b; })



	//FEM required parameters and matrixes via ./Documentation/StallcupIsaacA2018.pdf
int n_max;
double d_max;




void Init_ToyFDTDbezhig_FEM(void)
{

	  // wavelength in meters:
	  lambda = LIGHT_SPEED/FREQUENCY;
	  // angular frequency in radians/second:
	  omega = 2.0*M_PI*FREQUENCY;

	  // set ny and dy:
	  // start with a small ny:
	  ny = 3;
	  // calculate dy from the guide width and ny:
	  dy = GUIDE_WIDTH/ny;
	  // until dy is less than a twenty-fifth of a wavelength,
	  //     increment ny and recalculate dy:
	  while(dy >= lambda/25.0)
	    {
	      ny++;
	      dy = GUIDE_WIDTH/ny;
	    }

	  // set nz and dz:
	  // start with a small nz:
	  nz = 3;
	  // calculate dz from the guide height and nz:
	  dz = GUIDE_HEIGHT/nz;
	  // until dz is less than a twenty-fifth of a wavelength,
	  //     increment nz and recalculate dz:
	  while(dz >= lambda/25.0)
	    {
	      nz++;
	      dz = GUIDE_HEIGHT/nz;
	    }

	  // set dx, nx, and dt:
	  // set dx equal to dy or dz, whichever is smaller:
	  dx = (dy < dz) ? dy : dz;
	  // choose nx to make the guide LENGTH_IN_WAVELENGTHS
	  //     wavelengths long:
	  nx = (int)(LENGTH_IN_WAVELENGTHS*lambda/dx);
	  // chose dt for Courant stability:
	  dt = 1.0/(LIGHT_SPEED*sqrt(1/(dx*dx) + 1.0/(dy*dy) + 1.0/(dz*dz)));

	  //In the case of FEM, base simulation size on largest of phyical dimension calculated above.
	  n_max = max(max(nx, ny), nz);
	  d_max = max(max(dx, dy), dz);




		  // Allocate memory for the E field arrays:

		  // allocate the array of ex components:
		  ex = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      ex[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  ex[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      ex[i][j][k] = 0.0;
			    }
			}
		    }


		  // allocate the array of ey components:
		  ey = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      ey[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  ey[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      ey[i][j][k] = 0.0;
			    }
			}
		    }


		  // allocate the array of ez components:
		  ez = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      ez[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  ez[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      ez[i][j][k] = 0.0;
			    }
			}
		    }


		  // Allocate the H field arrays workspace:


		  // allocate the array of hx components:
		  hx = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      hx[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  hx[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      hx[i][j][k] = 0.0;
			    }
			}
		    }


		  // allocate the array of hy components:
		  hy = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      hy[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  hy[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      hy[i][j][k] = 0.0;
			    }
			}
		    }


		  // allocate the array of hz components:
		  hz = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      hz[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  hz[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      hz[i][j][k] = 0.0;
			    }
			}
		    }



		  // Allocate memory for the J source arrays:

		  // allocate the array of Jx components:
		  Jx = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      Jx[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  Jx[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      Jx[i][j][k] = 0.0;
			    }
			}
		    }

		  // allocate the array of Jy components:
		  Jy = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      Jy[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  Jy[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      Jy[i][j][k] = 0.0;
			    }
			}
		    }


		  // allocate the array of Jz components:
		  Jz = (double ***)malloc((n_max)*sizeof(double **));
		  for(i=0; i<(n_max); i++)
		    {
		      Jz[i] = (double **)malloc((n_max)*sizeof(double *));
		      for(j=0; j<(n_max); j++)
			{
			  Jz[i][j] = (double *)malloc((n_max)*sizeof(double));
			  for(k=0; k<(n_max); k++)
			    {
			      Jz[i][j][k] = 0.0;
			    }
			}
		    }

		  // constants used in the field update equations:
		  //NOTE: This is used only if DEBUG_MESH_CALCULATIONS_<xx>
		  //           is defined below.
		dtmudx = dt/(MU_0*d_max);
		dtepsdx = dt/(EPSILON_0*d_max);
		dtmudy = dt/(MU_0*d_max);
		dtepsdy = dt/(EPSILON_0*d_max);
		dtmudz = dt/(MU_0*d_max);
		dtepsdz = dt/(EPSILON_0*d_max);

		    //For 4nd order accurate "center differemce"  method.
        dtmudx_cd = dt/(MU_0*24.0*d_max);
        dtepsdx_cd = dt/(EPSILON_0*24.0*d_max);
        dtmudy_cd = dt/(MU_0*24.0*d_max);
        dtepsdy_cd = dt/(EPSILON_0*24.0*d_max);
        dtmudz_cd = dt/(MU_0*24.0*d_max);
        dtepsdz_cd = dt/(EPSILON_0*24.0*d_max);


}


void DoInteration_FEM(void)
{
    iteration++;

    // time in simulated seconds that the simulation has progressed:
    currentSimulatedTime = dt*(double)iteration;


#ifdef SIMULATE_Y_CONDUCTING_ELEMENT_TEST_CONTINUOUS

//#define CONDUCTOR_RUNNING_IN_X_AXIS
#ifdef CONDUCTOR_RUNNING_IN_X_AXIS

    for(i=10; i<(n_max-10); i++)
	{     //(Scaling is arbitrary in this test. What ever gives us the best visual effect)
	  // (See http://en.wikipedia.org/wiki/Gaussian_function or sub-directory "Documentation")

	  if(currentSimulatedTime <= 1.155499e-10)
	    Jx[i][n_max/2][n_max/2] =  (1.0/(.5*sqrt(2.0*M_PI)))*
	      exp(-((currentSimulatedTime - (50*2.265686119e-12))*(currentSimulatedTime - (50*2.265686119e-12)))/(2.0*(10*2.265686119e-12)*(10*2.265686119e-12)));
	  else
	    Jx[i][n_max/2][n_max/2] = .79788456;

	}
#endif

#define CONDUCTOR_RUNNING_IN_Y_AXIS
#ifdef CONDUCTOR_RUNNING_IN_Y_AXIS

    for(j=10; j<(n_max-10); j++)
    {     //(Scaling is arbitrary in this test. What ever gives us the best visual effect)
      // (See http://en.wikipedia.org/wiki/Gaussian_function or sub-directory "Documentation")

      if(currentSimulatedTime <= 1.155499e-10)
        Jy[n_max/2][j][n_max/2] =  (1.0/(.5*sqrt(2.0*M_PI)))*
          exp(-((currentSimulatedTime - (50*2.265686119e-12))*(currentSimulatedTime - (50*2.265686119e-12)))/(2.0*(10*2.265686119e-12)*(10*2.265686119e-12)));
      else
        Jy[n_max/2][j][n_max/2] = .79788456;

    }
#endif

//#define CONDUCTOR_RUNNING_IN_Z_AXIS
#ifdef CONDUCTOR_RUNNING_IN_Z_AXIS

    for(k=10; k<(n_max-10); k++)
    {     //(Scaling is arbitrary in this test. What ever gives us the best visual effect)
      // (See http://en.wikipedia.org/wiki/Gaussian_function or sub-directory "Documentation")

      if(currentSimulatedTime <= 1.155499e-10)
        Jz[n_max/2][n_max/2][k] =  (1.0/(.5*sqrt(2.0*M_PI)))*
          exp(-((currentSimulatedTime - (50*2.265686119e-12))*(currentSimulatedTime - (50*2.265686119e-12)))/(2.0*(10*2.265686119e-12)*(10*2.265686119e-12)));
      else
        Jz[n_max/2][n_max/2][k] = .79788456;

    }
#endif

#endif



//BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB



    /////////////////////////////////////////////////////////////////////////
    // Update the interior of the mesh:

    // Update all the H field vector components within the mesh:

    // Update the hx values:
//    for(i=0; i<(nx-1); i++)
//	{
//	  for(j=0; j<(ny); j++)
//	    {
//	      for(k=0; k<(nz); k++)
//		{
//		  hx[i][j][k] += (dtmudz*(ey[i+1][j][k+1] - ey[i+1][j][k]) -
//		                  dtmudy*(ez[i+1][j+1][k] - ez[i+1][j][k]));
//		}
//	    }
//	}
//#define DEBUG_MESH_CALCULATIONS_HX
#ifdef  DEBUG_MESH_CALCULATIONS_HX

    for(i=0; i<(n_max-1); i++)
   	{
   	  for(j=0; j<(n_max-1); j++)
   	    {
   	      for(k=0; k<(n_max-1); k++)
   		{
   		  hx[i][j][k] += (dtmudz*(ey[i+1][j][k+1] - ey[i+1][j][k]) -
   		                  dtmudy*(ez[i+1][j+1][k] - ez[i+1][j][k]));
   		}
   	    }
   	}



#else
        // Reference /home/maiello/Desktop/Development-Simulation/FDTD-GnuPlot-SecondOrder/Documentation/Thomson_mines_0052N_12099.pdf
        // Equation 2.35


    for(i=0; i<(n_max-1); i++)
    {
      for(j=1; j<(n_max-2); j++)
        {
          for(k=1; k<(n_max-2); k++)
        {
          hx[i][j][k] += (dtmudz_cd*(-ey[i+1][j][k+2] + 27*ey[i+1][j][k+1] - 27*ey[i+1][j][k] + ey[i+1][j][k-1]) -
                          dtmudy_cd*(-ez[i+1][j+2][k] + 27*ez[i+1][j+1][k] - 27*ez[i+1][j][k] + ez[i+1][j-1][k]));
        }
        }
    }


#endif


    // Update the hy values:
//    for(i=0; i<(nx); i++)
//	{
//	  for(j=0; j<(ny-1); j++)
//	    {
//	      for(k=0; k<(nz); k++)
//		{
//		  hy[i][j][k] +=  (dtmudx*(ez[i+1][j+1][k] - ez[i][j+1][k]) -
 //   		                   dtmudz*(ex[i][j+1][k+1] - ex[i][j+1][k]));
//		}
//	    }
//	}
//#define DEBUG_MESH_CALCULATIONS_HY
#ifdef  DEBUG_MESH_CALCULATIONS_HY

	for(i=0; i<(n_max-1); i++)
	{
	  for(j=0; j<(n_max-1); j++)
		{
		  for(k=0; k<(n_max-1); k++)
		{
		  hy[i][j][k] +=  (dtmudx*(ez[i+1][j+1][k] - ez[i][j+1][k]) -
							   dtmudz*(ex[i][j+1][k+1] - ex[i][j+1][k]));
		}
		}
	}

#else

    // Reference /home/maiello/Desktop/Development-Simulation/FDTD-GnuPlot-SecondOrder/Documentation/Thomson_mines_0052N_12099.pdf
    // Equation 2.35

    for(i=1; i<(n_max-2); i++)
    {
      for(j=0; j<(n_max-1); j++)
        {
          for(k=1; k<(n_max-2); k++)
        {
          hy[i][j][k] +=  (dtmudx_cd*(- ez[i+2][j+1][k] + 27*ez[i+1][j+1][k] - 27*ez[i][j+1][k] + ez[i-1][j+1][k]) -
                           dtmudz_cd*(- ex[i][j+1][k+2] + 27*ex[i][j+1][k+1] - 27*ex[i][j+1][k] + ex[i][j+1][k-1]));
        }
        }
    }



#endif



    // Update the hz values:
 //   for(i=0; i<(nx); i++)
//	{
//	  for(j=0; j<(ny); j++)
//	    {
//	      for(k=0; k<(nz-1); k++)
//		{
//		  hz[i][j][k] +=  (dtmudy*(ex[i][j+1][k+1] - ex[i][j][k+1]) -
//    		                   dtmudx*(ey[i+1][j][k+1] - ey[i][j][k+1]));
//		}
//	    }
//	}
//#define DEBUG_MESH_CALCULATIONS_HZ
#ifdef  DEBUG_MESH_CALCULATIONS_HZ

    for(i=0; i<(n_max-1); i++)
	{
	  for(j=0; j<(n_max-1); j++)
	    {
	      for(k=0; k<(n_max-1); k++)
		{
		  hz[i][j][k] +=  (dtmudy*(ex[i][j+1][k+1] - ex[i][j][k+1]) -
    		                   dtmudx*(ey[i+1][j][k+1] - ey[i][j][k+1]));
		}
	    }
	}

#else

    // Reference /home/maiello/Desktop/Development-Simulation/FDTD-GnuPlot-SecondOrder/Documentation/Thomson_mines_0052N_12099.pdf
    // Equation 2.35


    for(i=1; i<(n_max-2); i++)
    {
      for(j=1; j<(n_max-2); j++)
        {
          for(k=0; k<(n_max-1); k++)
        {
          hz[i][j][k] +=  (dtmudy_cd*(- ex[i][j+2][k+1] + 27*ex[i][j+1][k+1] - 27*ex[i][j][k+1] + ex[i][j-1][k+1]) -
                           dtmudx_cd*(- ey[i+2][j][k+1] + 27*ey[i+1][j][k+1] - 27*ey[i][j][k+1] + ey[i-1][j][k+1]));
        }
        }
    }





#endif



    // Update the E field vector components.


    // Update the ex values:
//    for(i=0; i<(nx); i++)
//	{
//	  for(j=1; j<(ny); j++)
//	    {
//	      for(k=1; k<(nz); k++)
//		{
//		    ex[i][j][k] = (dtepsdy*(hz[i][j][k-1] - hz[i][j-1][k-1]) -
//				   dtepsdz*(hy[i][j-1][k] - hy[i][j-1][k-1])) - Jx[i][j][k];
//		}
//	    }
//	}


//#define DEBUG_MESH_CALCULATIONS_EX
#ifdef  DEBUG_MESH_CALCULATIONS_EX

    for(i=0; i<(n_max-1); i++)
	{
	  for(j=1; j<(n_max-1); j++)
	    {
	      for(k=1; k<(n_max-1); k++)
		{
		    ex[i][j][k] = (dtepsdy*(hz[i][j][k-1] - hz[i][j-1][k-1]) -
				   dtepsdz*(hy[i][j-1][k] - hy[i][j-1][k-1])) - Jx[i][j][k];
		}
	    }
	}

#else
    // Reference /home/maiello/Desktop/Development-Simulation/FDTD-GnuPlot-SecondOrder/Documentation/Thomson_mines_0052N_12099.pdf
    // Equation 2.34

    for(i=0; i<(n_max-1); i++)
    {
      for(j=2; j<(n_max-1); j++)
        {
          for(k=2; k<(n_max-1); k++)
        {
            ex[i][j][k] = (dtepsdy_cd*(- hz[i][j+1][k-1] + 27*hz[i][j][k-1] - 27*hz[i][j-1][k-1] + hz[i][j-2][k-1]) -
                           dtepsdz_cd*(- hy[i][j-1][k+1] + 27*hy[i][j-1][k] - 27*hy[i][j-1][k-1] + hy[i][j-1][k-2])) - Jx[i][j][k];
        }
        }
    }


#endif





    // Update the ey values:
//    for(i=1; i<(nx); i++)
//	{
//	  for(j=0; j<(ny); j++)
//	    {
//	      for(k=1; k<(nz); k++)
//		{
//		    ey[i][j][k] = (dtepsdz*(hx[i-1][j][k] - hx[i-1][j][k-1]) -
//				   dtepsdx*(hz[i][j][k-1] - hz[i-1][j][k-1]))- Jy[i][j][k];
//		}
//	    }
//	}
//#define DEBUG_MESH_CALCULATIONS_EY
#ifdef  DEBUG_MESH_CALCULATIONS_EY

    for(i=1; i<(n_max-1); i++)
	{
	  for(j=0; j<(n_max-1); j++)
	    {
	      for(k=1; k<(n_max-1); k++)
		{
		    ey[i][j][k] = (dtepsdz*(hx[i-1][j][k] - hx[i-1][j][k-1]) -
				   dtepsdx*(hz[i][j][k-1] - hz[i-1][j][k-1]))- Jy[i][j][k];
		}
	    }
	}

#else

    // Reference /home/maiello/Desktop/Development-Simulation/FDTD-GnuPlot-SecondOrder/Documentation/Thomson_mines_0052N_12099.pdf
    // Equation 2.34


    for(i=2; i<(n_max-1); i++)
    {
      for(j=0; j<(n_max-1); j++)
        {
          for(k=2; k<(n_max-1); k++)
        {
            ey[i][j][k] = (dtepsdz_cd*(- hx[i-1][j][k+1] + 27*hx[i-1][j][k] - 27*hx[i-1][j][k-1] + hx[i-1][j][k-2]) -
                           dtepsdx_cd*(- hz[i+1][j][k-1] + 27*hz[i][j][k-1] - 27*hz[i-1][j][k-1] + hz[i-2][j][k-1]))- Jy[i][j][k];
        }
        }
    }



#endif


    // Update the ez values:
//    for(i=1; i<(nx); i++)
//	{
//	  for(j=1; j<(ny); j++)
//	    {
//	      for(k=0; k<(nz); k++)
//		{
//		    ez[i][j][k] = (dtepsdx*(hy[i][j-1][k] - hy[i-1][j-1][k]) -
//				   dtepsdy*(hx[i-1][j][k] - hx[i-1][j-1][k])) - Jz[i][j][k];
//		}
//	    }
//	}


//#define DEBUG_MESH_CALCULATIONS_EZ
#ifdef  DEBUG_MESH_CALCULATIONS_EZ

    for(i=1; i<(n_max-1); i++)
	{
	  for(j=1; j<(n_max-1); j++)
	    {
	      for(k=0; k<(n_max-1); k++)
		{
		    ez[i][j][k] = (dtepsdx*(hy[i][j-1][k] - hy[i-1][j-1][k]) -
				   dtepsdy*(hx[i-1][j][k] - hx[i-1][j-1][k])) - Jz[i][j][k];
		}
	    }
	}

#else

    // Reference /home/maiello/Desktop/Development-Simulation/FDTD-GnuPlot-SecondOrder/Documentation/Thomson_mines_0052N_12099.pdf
    // Equation 2.34


    for(i=2; i<(n_max-1); i++)
    {
      for(j=2; j<(n_max-1); j++)
        {
          for(k=0; k<(n_max-1); k++)
        {
            ez[i][j][k] = (dtepsdx_cd*(- hy[i+1][j-1][k] + 27*hy[i][j-1][k] - 27*hy[i-1][j-1][k] + hy[i-2][j-1][k]) -
                           dtepsdy_cd*(- hx[i-1][j+1][k] + 27*hx[i-1][j][k] - 27*hx[i-1][j-1][k] + hx[i-1][j-2][k])) - Jz[i][j][k];
        }
        }
    }




#endif





}







// *******************************************************************************************************************************************************

void Init_ToyFDTDbezhig(void)
{

  /////////////////////////////////////////////////////////////////////////////
  // setting up the problem to be modeled
  //
  // Modified from David K. Cheng, Field and Wave Electromagnetics, 2nd ed., 
  //     pages 554-555. 
  // Parallel plate waveguide, interior height = 1.02cm, infinite width of which
  //     2.29 cm is simulated.
  //
  // Choosing nx, ny, and nz:
  // There should be at least 20 cells per wavelength in each direction, 
  //     but we'll go with 25 so the animation will look prettier.    
  // The number of cells along the width of the guide and the width of 
  //    those cells should fit the simulated guide width exactly, so that ny*dy 
  //    = GUIDE_WIDTH meters.  
  //    The same should be true for nz*dz = GUIDE_HEIGHT meters.  
  // dx is chosen to be dy or dz -- whichever is smaller
  // nx is chosen to make the guide LENGTH_IN_WAVELENGTHS 
  //     wavelengths long.  
  // 
  // dt is chosen for Courant stability; the time step must be kept small 
  //     enough so that the plane wave only travels one cell length 
  //     (one dx) in a single timestep.  Otherwise FDTD cannot keep up 
  //     with the signal propagation, since FDTD computes a cell only from 
  //     it's immediate neighbors.  

  // wavelength in meters:
  lambda = LIGHT_SPEED/FREQUENCY; 
  // angular frequency in radians/second:
  omega = 2.0*M_PI*FREQUENCY; 

  // set ny and dy:
  // start with a small ny:
  ny = 3;  
  // calculate dy from the guide width and ny:
  dy = GUIDE_WIDTH/ny;
  // until dy is less than a twenty-fifth of a wavelength,
  //     increment ny and recalculate dy:
  while(dy >= lambda/25.0)
    {
      ny++;
      dy = GUIDE_WIDTH/ny;
    }

  // set nz and dz:
  // start with a small nz:
  nz = 3;  
  // calculate dz from the guide height and nz:
  dz = GUIDE_HEIGHT/nz;
  // until dz is less than a twenty-fifth of a wavelength,
  //     increment nz and recalculate dz:
  while(dz >= lambda/25.0)
    {
      nz++;
      dz = GUIDE_HEIGHT/nz;
    }

  // set dx, nx, and dt:
  // set dx equal to dy or dz, whichever is smaller:
  dx = (dy < dz) ? dy : dz;
  // choose nx to make the guide LENGTH_IN_WAVELENGTHS 
  //     wavelengths long:  
  nx = (int)(LENGTH_IN_WAVELENGTHS*lambda/dx);
  // chose dt for Courant stability:
  dt = 1.0/(LIGHT_SPEED*sqrt(1/(dx*dx) + 1.0/(dy*dy) + 1.0/(dz*dz)));
 

  // constants used in the field update equations:
  dtmudx = dt/(MU_0*dx);
  dtepsdx = dt/(EPSILON_0*dx);
  dtmudy = dt/(MU_0*dy);
  dtepsdy = dt/(EPSILON_0*dy);
  dtmudz = dt/(MU_0*dz);
  dtepsdz = dt/(EPSILON_0*dz);
  



  /////////////////////////////////////////////////////////////////////////////
  // memory allocation for the FDTD mesh:
  // There is a separate array for each of the six vector components, 
  //      ex, ey, ez, hx, hy, and hz.
  // The mesh is set up so that tangential E vectors form the outer faces of 
  //     the simulation volume.  There are nx*ny*nz cells in the mesh, but 
  //     there are nx*(ny+1)*(nz+1) ex component vectors in the mesh.
  //     There are (nx+1)*ny*(nz+1) ey component vectors in the mesh.
  //     There are (nx+1)*(ny+1)*nz ez component vectors in the mesh.
  // If you draw out a 2-dimensional slice of the mesh, you'll see why
  //     this is.  For example if you have a 3x3x3 cell mesh, and you 
  //     draw the E field components on the z=0 face, you'll see that
  //     the face has 12 ex component vectors, 3 in the x-direction
  //     and 4 in the y-direction.  That face also has 12 ey components,
  //     4 in the x-direction and 3 in the y-direction.  

  // Allocate memory for the E field arrays:

  // allocate the array of ex components:
  ex = (double ***)malloc((nx)*sizeof(double **));
  for(i=0; i<(nx); i++)
    {  
      ex[i] = (double **)malloc((ny+1)*sizeof(double *));
      for(j=0; j<(ny+1); j++)
	{
	  ex[i][j] = (double *)malloc((nz+1)*sizeof(double));
	  for(k=0; k<(nz+1); k++)
	    {
	      ex[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx)*(ny+1)*(nz+1) * sizeof(double));
  
  // allocate the array of ey components:
  ey = (double ***)malloc((nx+1)*sizeof(double **));
  for(i=0; i<(nx+1); i++)
    {  
      ey[i] = (double **)malloc((ny)*sizeof(double *));
      for(j=0; j<(ny); j++)
	{
	  ey[i][j] = (double *)malloc((nz+1)*sizeof(double));
	  for(k=0; k<(nz+1); k++)
	    {
	      ey[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx+1)*(ny)*(nz+1) * sizeof(double));

  // allocate the array of ez components:
  ez = (double ***)malloc((nx+1)*sizeof(double **));
  for(i=0; i<(nx+1); i++)
    {  
      ez[i] = (double **)malloc((ny+1)*sizeof(double *));
      for(j=0; j<(ny+1); j++)
	{
	  ez[i][j] = (double *)malloc((nz)*sizeof(double));
	  for(k=0; k<(nz); k++)
	    {
	      ez[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx+1)*(ny+1)*(nz) * sizeof(double));

  // Allocate the H field arrays:
  // Since the H arrays are staggered half a step off 
  //     from the E arrays in every direction, the H 
  //     arrays are one cell smaller in the x, y, and z 
  //     directions than the corresponding E arrays. 
  // By this arrangement, the outer faces of the mesh
  //     consist of E components only, and the H 
  //     components lie only in the interior of the mesh.  

  // allocate the array of hx components:
  hx = (double ***)malloc((nx-1)*sizeof(double **));
  for(i=0; i<(nx-1); i++)
    {  
      hx[i] = (double **)malloc((ny)*sizeof(double *));
      for(j=0; j<(ny); j++)
	{
	  hx[i][j] = (double *)malloc((nz)*sizeof(double));
	  for(k=0; k<(nz); k++)
	    {
	      hx[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx-1)*(ny)*(nz) * sizeof(double));
  
  // allocate the array of hy components:
  hy = (double ***)malloc((nx)*sizeof(double **));
  for(i=0; i<(nx); i++)
    {  
      hy[i] = (double **)malloc((ny-1)*sizeof(double *));
      for(j=0; j<(ny-1); j++)
	{
	  hy[i][j] = (double *)malloc((nz)*sizeof(double));
	  for(k=0; k<(nz); k++)
	    {
	      hy[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx)*(ny-1)*(nz) * sizeof(double));
  
  // allocate the array of hz components:
  hz = (double ***)malloc((nx)*sizeof(double **));
  for(i=0; i<(nx); i++)
    {  
      hz[i] = (double **)malloc((ny)*sizeof(double *));
      for(j=0; j<(ny); j++)
	{
	  hz[i][j] = (double *)malloc((nz-1)*sizeof(double));
	  for(k=0; k<(nz-1); k++)
	    {
	      hz[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx)*(ny)*(nz-1) * sizeof(double));




  // Allocate memory for the J source arrays:

  // allocate the array of Jx components:
  Jx = (double ***)malloc((nx)*sizeof(double **));
  for(i=0; i<(nx); i++)
    {  
      Jx[i] = (double **)malloc((ny+1)*sizeof(double *));
      for(j=0; j<(ny+1); j++)
	{
	  Jx[i][j] = (double *)malloc((nz+1)*sizeof(double));
	  for(k=0; k<(nz+1); k++)
	    {
	      Jx[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx)*(ny+1)*(nz+1) * sizeof(double));
  
  // allocate the array of Jy components:
  Jy = (double ***)malloc((nx+1)*sizeof(double **));
  for(i=0; i<(nx+1); i++)
    {  
      Jy[i] = (double **)malloc((ny)*sizeof(double *));
      for(j=0; j<(ny); j++)
	{
	  Jy[i][j] = (double *)malloc((nz+1)*sizeof(double));
	  for(k=0; k<(nz+1); k++)
	    {
	      Jy[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx+1)*(ny)*(nz+1) * sizeof(double));

  // allocate the array of Jz components:
  Jz = (double ***)malloc((nx+1)*sizeof(double **));
  for(i=0; i<(nx+1); i++)
    {  
      Jz[i] = (double **)malloc((ny+1)*sizeof(double *));
      for(j=0; j<(ny+1); j++)
	{
	  Jz[i][j] = (double *)malloc((nz)*sizeof(double));
	  for(k=0; k<(nz); k++)
	    {
	      Jz[i][j][k] = 0.0;
	    }
	}
    }
  allocatedBytes += ( (nx+1)*(ny+1)*(nz) * sizeof(double));

  // allocate the array of PEC flags:
  PEC = (unsigned char ***)malloc((nx+1)*sizeof(unsigned char **));
  for(i=0; i<(nx+1); i++)
    {  
      PEC[i] = (unsigned char **)malloc((ny+1)*sizeof(unsigned char *));
      for(j=0; j<(ny+1); j++)
	{
	  PEC[i][j] = (unsigned char *)malloc((nz)*sizeof(unsigned char));
	  for(k=0; k<(nz); k++)
	    {
	      PEC[i][j][k] = 0;
	    }
	}
    }





}


void DoInteration(void)
{
	int idx;


      iteration++;

      // time in simulated seconds that the simulation has progressed:
      currentSimulatedTime = dt*(double)iteration;  



  
#ifdef ORIGINAL_PLANE_WAVE_TEST


      /////////////////////////////////////////////////////////////////////////
      // Compute the stimulus: a sinusoidal pulse emanates from the x=0 face:
      //     The length of the guide lies in the x-direction, the width of the 
      //     guide lies in the y-direction, and the height of the guide lies
      //     in the z-direction.  So the guide is sourced by all the ez 
      //     components on the stimulus face.  
      // The pulse is one complete wavelength of a shifted sinusoid that 
      //     ranges from zero to one in intensity.  So the stimulus varies 
      //     sinusoidally from zero to one to zero again and then terminates
      //     --the x=0 face becomes a PEC thereafter.  
      //
      // compute the current stimulus value:
#ifdef SINUSOIDAL_PULSE_STIMULUS
      if (currentSimulatedTime <= 1.0/FREQUENCY)
	{
	  stimulus = .5*(1.0 + sin(omega*currentSimulatedTime - M_PI*.5));
	}
      else
	{
	  stimulus = 0.0;
	}
#else
      stimulus = sin(omega*currentSimulatedTime);
#endif
      // set all vectors on the x=0 face to the value of stimulus:
      for (i=0; i<(1); i++)
	{ 
	  for(j=0; j<(ny+1); j++)
	    {
	      for(k=0; k<nz; k++)
		{
		  ez[i][j][k] = stimulus;
		}
	    }
	}

#endif 

#ifdef SIMULATE_Y_CONDUCTING_ELEMENT_TEST



      for(i=10; i<(nx-10); i++)
	{     //(Scaling is arbitrary in this test. What ever gives us the best visual effect)
	  // (See http://en.wikipedia.org/wiki/Gaussian_function or sub-directory "Documentation")

	   Jx[i][ny/2][nz/2]   = (1.0/(.5*sqrt(2.0*M_PI)))*
	    exp(-((currentSimulatedTime - (50*2.265686119e-12))*(currentSimulatedTime - (50*2.265686119e-12)))/(2.0*(10*2.265686119e-12)*(10*2.265686119e-12)));
	  PEC[i][ny/2][nz/2] = 1;
	  
	

	}
 

#endif

#ifdef SIMULATE_Y_CONDUCTING_ELEMENT_TEST_CONTINUOUS


      for(i=10; i<(nx-10); i++)
	{     //(Scaling is arbitrary in this test. What ever gives us the best visual effect)
	  // (See http://en.wikipedia.org/wiki/Gaussian_function or sub-directory "Documentation")

	  if(currentSimulatedTime <= 1.155499e-10)
	    Jx[i][ny/2][nz/2] =  (1.0/(.5*sqrt(2.0*M_PI)))*
	      exp(-((currentSimulatedTime - (50*2.265686119e-12))*(currentSimulatedTime - (50*2.265686119e-12)))/(2.0*(10*2.265686119e-12)*(10*2.265686119e-12)));
	  else
	    Jx[i][ny/2][nz/2] = .79788456;

	    
	  PEC[i][ny/2][nz/2] = 1;
	  
	

	}
 

#endif


	
      /////////////////////////////////////////////////////////////////////////
      // Update the interior of the mesh:
      //    all vector components except those on the faces of the mesh.  
      //
      // Update all the H field vector components within the mesh:
      //     Since all H vectors are internal, all H values are updated here.  
      //     Note that the normal H vectors on the faces of the mesh are not
      //     computed here, and in fact were never allocated -- the normal
      //     H components on the faces of the mesh are never used to update
      //     any other value, so they are left out of the memory allocation 
      //     entirely.  

      // Update the hx values:
      for(i=0; i<(nx-1); i++)
	{  
	  for(j=0; j<(ny); j++)
	    {
	      for(k=0; k<(nz); k++)
		{
		  hx[i][j][k] += (dtmudz*(ey[i+1][j][k+1] - ey[i+1][j][k]) - 
		                  dtmudy*(ez[i+1][j+1][k] - ez[i+1][j][k]));
		}
	    }
	}

      // Update the hy values:
      for(i=0; i<(nx); i++)
	{  
	  for(j=0; j<(ny-1); j++)
	    {
	      for(k=0; k<(nz); k++)
		{
		  hy[i][j][k] +=  (dtmudx*(ez[i+1][j+1][k] - ez[i][j+1][k]) -
      		                   dtmudz*(ex[i][j+1][k+1] - ex[i][j+1][k]));
		}
	    }
	}

      // Update the hz values:
      for(i=0; i<(nx); i++)
	{  
	  for(j=0; j<(ny); j++)
	    {
	      for(k=0; k<(nz-1); k++)
		{
		  hz[i][j][k] +=  (dtmudy*(ex[i][j+1][k+1] - ex[i][j][k+1]) -
      		                   dtmudx*(ey[i+1][j][k+1] - ey[i][j][k+1]));
		}
	    }
	}

      // Update the E field vector components.  
      // The values on the faces of the mesh are not updated here; they 
      //      are handled by the boundary condition computation 
      //      (and stimulus computation).  

      // Update the ex values:
      for(i=0; i<(nx); i++)
	{  
	  for(j=1; j<(ny); j++)
	    {
	      for(k=1; k<(nz); k++)
		{

		  if(PEC == 0)
		    ex[i][j][k] += (dtepsdy*(hz[i][j][k-1] - hz[i][j-1][k-1]) -
				  dtepsdz*(hy[i][j-1][k] - hy[i][j-1][k-1]));
		  else
		    ex[i][j][k] = (dtepsdy*(hz[i][j][k-1] - hz[i][j-1][k-1]) -
				   dtepsdz*(hy[i][j-1][k] - hy[i][j-1][k-1])) - Jx[i][j][k];
		}
	    }
	}      

      // Update the ey values:
      for(i=1; i<(nx); i++)
	{  
	  for(j=0; j<(ny); j++)
	    {
	      for(k=1; k<(nz); k++)
		{
		  if(PEC == 0)
		    ey[i][j][k] += (dtepsdz*(hx[i-1][j][k] - hx[i-1][j][k-1]) -
				    dtepsdx*(hz[i][j][k-1] - hz[i-1][j][k-1]));
		  else
		    ey[i][j][k] = (dtepsdz*(hx[i-1][j][k] - hx[i-1][j][k-1]) -
				   dtepsdx*(hz[i][j][k-1] - hz[i-1][j][k-1]))- Jy[i][j][k];
		}
	    }
	}

      // Update the ez values:
      for(i=1; i<(nx); i++)
	{  
	  for(j=1; j<(ny); j++)
	    {
	      for(k=0; k<(nz); k++)
		{
		  if(PEC == 0)
		    ez[i][j][k] += (dtepsdx*(hy[i][j-1][k] - hy[i-1][j-1][k]) -
				    dtepsdy*(hx[i-1][j][k] - hx[i-1][j-1][k]));
		  else
		    ez[i][j][k] = (dtepsdx*(hy[i][j-1][k] - hy[i-1][j-1][k]) -
				   dtepsdy*(hx[i-1][j][k] - hx[i-1][j-1][k])) - Jz[i][j][k];
		}
	    }
	}
      

      /////////////////////////////////////////////////////////////////////////
      // Compute the boundary conditions:

#ifdef PMC
      // Compute the PMC symmetry boundaries:
      // j = 0 face, j = ny face
      for(i=0; i<nx; i++)
	{	
	  for(k=0; k<(nz+1); k++)
	    {
	      ex[i][0][k] = ex[i][ny-1][k];
	      ex[i][ny][k] = ex[i][1][k];
	    }
	}
      for(i=0; i<(nx+1); i++)
	{	
	  for(k=0; k<nz; k++)
	    {
	      ez[i][0][k] = ez[i][ny-1][k];
	      ez[i][ny][k] = ez[i][1][k];
	    }
	}
#endif

      // Compute the PEC boundaries:
      //
      // OK, so I'm yanking your chain on this one.  The PEC condition is 
      // enforced by setting the tangential E field components on the PEC
      // faces of the mesh to zero every timestep (except the stimulus 
      // face).  But the lazy/efficient way out is to initialize those 
      // vectors to zero and never compute them again, which is exactly 
      // what happens in this code.  



  
  
}	
