

//		 This simulation for a Single Phase Multilevel Converter is dervied from App_SpiceTest.hpp. Some of the documentation for how this
//       simulation is setup is retained in this file. For the full documentation, refer to App_SpiceTest.hpp.
//
//		 This file is nearly identical to the original App_SpiceTest.hpp. The changes are the replacement of the R_Load with a current source.
//       The conditional build definition INDUCTIVE_COUPLED_LOAD is retained so as to insure any changes to not break the original operation
//		 benchmarked by App_SpiceTest.cir.
//
//		 The difference is that when INDUCTIVE_COUPLED_LOAD is undefined, we connect a DC current source to "vs_a" instead of a resistor.
//       We also redefine how the SrcObject Vref operates. It "steps" the voltage command periodically in 5 steps to force the cells to
//       step up-to and down-to the DC plus and minus bus respectively.
//
//		 Also a "circulator"is added to the SrcObject's "ECtrl<x>", and controlled by statements added to SrcObject
//       "Vm0". The circulator is needed for the DC test. This is a significant change to the orginal App_SpiceTest.hpp file so we define
//		 INDUCTIVE_COUPLED_LOAD "true" to check and see if we have broken any of the functionality. Practically speeking, if everything
//       is OK, the ciculator should not effect the original operation benched-marked by App_SpiceTest.hpp
//
//
//
//
// 
//
//
//
//       Nodal currents for the diagram below are designated as follows.
//
//       "-+" for horizontally drawn currents ("------>")
//      
//       "+   for vertically drawn currents ("^
//        |"                                  |
//                                            |")
//
//                 i1 -+
//                ------------        
//               | v11  i1ct  |
//               |-------- +  |
//             |-   |D11  |-  |
//           - |- + ^   <C11> |
//          S11|-|- |i1st |   | 
//               |--------    |
//   1000 VDC ---| v01  i1cb <C1> (Cell 3p)  
//        i01 -+ |-------- +  |
//             |-   |D12  |-  |
//           - |- + ^   <C12> | 
//          S12|-|- |i1sb |   |
//               |-------     |
//        i1_l + | v12        |
//             -  ---+--------
//                   | 
//                   | 
//    ---------------
//   |
//   |             i2 -+
//   |            ------------        
//   |           | v21  i2ct  |
//   |           |-------- +  |
//   |         |-   |D21  |-  |
//   |       - |- + ^   <C21> |
//   |      S21|-|- |i2st |   | 
//   |           |--------    |
//    -----------| v12  i2cb <C2> (Cell 2p)
//        i12 -+ |-------- +  |
//             |-   |D22  |-  |
//           - |- + ^   <C22> |
//          S22|-|- |i2sb |   |
//               |--------    |
//        i2_l + | v23        |
//             -  ---+--------
//                   |
//                   |
//    ---------------   
//   |                         
//   |             i3 -+                               
//   |            ------------                      
//   |           | v31  i3ct  |                    
//   |           |-------- +  |
//   |         |-   |D31  |-  |
//   |       - |- + ^   <C31> |
//   |      S31|-|- |i3st |   |
//   |           |--------    |
//    -----------| v23  i3cb <C3> (Cell 1p)
//        i23 -+ |-------- +  |
//             |-   |D32  |-  |
//           - |- + ^   <C32> |
//          S32|-|- |i3sb |   |
//               |--------    |
//        i3_l + | v34        |
//             -  ---+--------
//                   |
//                   |
//    -- ------------
//   |   
//   |             i4 -+
//   |            ------------        
//   |           | v41  i4ct  |
//   |           |-------- +  |
//   |         |-   |D41  |-  |
//   |       - |- + ^   <C41> |
//   |      S41|-|- |i4st |   |
//   |           |--------    |
//    -----------| v34  i4cb <C4> (Cell 0p)
//        i34 -+ |-------- +  |
//             |-   |D42  |-  |
//           - |- + ^   <C42> |
//          S42|-|- |i4sb |   |
//               |--------    |
//        i4_l + | v44        |
//             -  ---+--------
//             i44 - |
//                 + |
//                   +----
//                  *|    |
//              -- <Lp>  <Rp>
//             |     |    |    ia -+
//             |     +----+------------ vs_a                                             
//             |    *|    |
//              -- <Ln>  <Rn>                          
//                   |    |                          
//    ---------------+----                           
//   |                                     
//   |             i5 -+                           
//   |            ------------                         
//   |           | v51  i5ct  |                        
//   |           |-------- +  |
//   |         |-   |D51  |-  |
//   |       - |- + ^   <C51> |
//   |      S51|-|- |i5st |   |
//   |           |--------    |
//    -----------| v45  i5cb <C5> (Cell 3n)
//        i45 -+ |-------- +  |
//             |-   |D52  |-  |
//           - |- + ^   <C52> |
//          S52|-|- |i5sb |   |
//               |--------    |
//        i5_l + | v56        |
//             -  ---+--------
//                   |
//                   |
//    ---------------
//   |   
//   |             i6 -+ 
//   |            ------------        
//   |           | v61  i6ct  |
//   |       (12)|-------- +  |
//   |         |-   |D61  |-  |
//   |       - |- + ^   <C61> |
//   |      S61|-|- |i6st |   |
//   |           |--------    |
//    -----------| v56  i6cb <C6> (Cell 2n)
//        i56 -+ |-------- +  |
//             |-   |D62  |-  |
//           - |- + ^   <C62> |
//          S62|-|- |i6sb |   |
//               |--------    |
//        i6_l + | v67        |
//             -  ---+-------
//                   |
//                   |
//    ---------------      
//   |                            
//   |             i7 -+                            
//   |            ------------                       
//   |           | v71  i7ct  |                      
//   |       (14)|-------- +  |
//   |         |-   |D71  |-  |
//   |       - |- + ^   <C71> |
//   |      S71|-|- |i7st |   |
//   |           |--------    |
//    -----------| v67  i7cb <C7> (Cell 1n)
//        i67 -+ |-------- +  |
//             |-   |D72  |-  |
//           - |- + ^   <C72> |
//          S72|-|- |i7sb |   |
//               |--------    |
//        i7_l + | v78        |
//             -  ---+--------
//                   |
//                   |
//    ---------------
//   |   
//   |             i8 -+
//   |            ------------        
//   |           | v81  i8ct  |
//   |           |-------- +  |
//   |         |-   |D81  |-  |
//   |       - |- + ^   <C81> |
//   |      S81|-|- |i8st |   |
//   |           |--------    |
//    -----------| v78  i8cb <C8> (Cell 0n)
//        i78 -+ |-------- +  |
//             |-   |D82  |-  |
//           - |- + ^   <C82> |
//          S82|-|- |i8sb |   |
//               |--------    |
//        i8_l + | v88        |
//             -  ---+--------
//                   |              
//                   |- i88
//                   |+
//                   |
//              - 1000 VDC
//
//
//
//
//    Load connection to "vs_a" if INDUCTIVE_COUPLED_LOAD is defined.
//
//    vs_a ---------<Lm_a>-----------------------------------<La>-----------------<Ra>---<Ca>---
//                     |             ia -+                                                      |
//                     |                                                                        |
//                     |                                                                        |
//                     |                                                                        |
//         ---------<lm_b>-----------------------------------<Lb>-----------------<Rb>---<Cb>---+-----
//        |                          ib -+                                                            |
//        |                                                                                          GND
//       GND                                                                                            
//                                                                                                            
//                                                                                                   
//    Load connection to "vs_a" if INDUCTIVE_COUPLED_LOAD is undefined
//
//
//
//    vs_a --------------- <CONSTANT_CURRENT_SOURCE_LOAD> -------------
//
//
//                                               
//          
//
//
//	  (NOTE: The equations below relate only to the "Backward Euler" pass. The equations for the "Trapezoidal" pass are not shown
//			 in this outline.)
//
//    For Cell 3p:
//
//            
//      v01/rS11 - v11/rS11 + v01/rD11 - v11/rD11 = i1st    		=>    - v11*(1/rS11 + 1/rD11) - i1st = - v01*(1/rS11 + 1/rD11)    (v01 is a constant)   (1)
//
//      v12/rS12 - v01/rS12 + v12/rD12 - v01/rD12 = i1sb    		=>    v12*(1/rS12 + 1/rD12) - i1sb =  v01*(1/rS12 + 1/rD12)       (v01 is a constant)   (2)
//
//      i01 + i1_l = i1					                   			=>    i01 + i1_l - i1 = 0           				                                    (3)
//
//		i1st + i1ct = i1											=>	  i1st + ilct - il = 0																(4)
//
//		i1sb + i1cb = i1_l											=> 	  i1sb + i1cb - i1_l = 0															(5)
//
//      v11*C1/h - v12*C1/h - v11_*C1/h + v12_*C1/h = i1			=>    v11*C1/h - v12*C1/h - i1 = v11_*C1/h - v12_*C1/h    		               			(6)
//
//      v01*C11/h - v11*C11/h -v01_*C11/h + v11_*C11/h = i1ct 		=>    - v11*C11/h - i1ct = - v11_*C11/h 						(v01 is a constant) (7)
//
//      v12*C12/h - v01*C12/h - v12_*C12/h + v01_*C12/h = i1cb    	=>    v12*C12/h - i1cb = v12_*C12/h  	    					(v01 is a constant) (8)
//
//
//
//    For Cell 2p:
//
//            
//      v12/rS21 - v21/rS21 + v12/rD21 - v21/rD21 = i2st                    =>    v12*(1/rS21 + 1/rD21) - v21*(1/rS21 + 1/rD21) - i2st     = 0          (9)
//
//      v23/rS22 - v12/rS22 + v23/rD22 - v12/rD22 = i2sb                    =>    v23*(1/rS22 + 1/rD22) - v12*(1/rS22 + 1/rD22) - i2sb     = 0          (10)
//
//      i12 + i2_l = i2				                    					=>    i12 + i2_l - i2 = 0                   				              	(11)
//
//		i2st + i2ct = i2													=>    i2st + i2ct - i2 = 0													(12)
//
//		i2sb + i2cb = i2_l													=>	  i2sb + i2cb - i2_l = 0												(13)
//
//      v21*C2/h - v23*C2/h - v21_*C2/h + v23_*C2/h = i2                 	=>    v21*C2/h - v23*C2/h - i2 = v21_*C2/h - v23_*C2/h                    	(14)
//
//      v12*C21/h - v21*C21/h - v12_*C21/h + v21_*C21/h = i2ct          	=>    v12*C21/h - v21*C21/h - i2ct = v12_*C21/h - v21_*C21/h      	   		(15)
//
//      v23*C22/h - v12*C22/h - v23_*C22/h + v12_*C22/h = i2cb       		=>    v23*C22/h - v12*C22/h - i2cb = v23_*C22/h - v12_*C22/h              	(16)
//
//
//
//
//    For Cell 1p:
//
//            
//      v23/rS31 - v31/rS31 + v23/rD31 - v31/rD31 = i3st                    =>    v23*(1/rS31 + 1/rD31) - v31*(1/rS31 + 1/rD31) - i3st     = 0           (17)
//
//      v34/rS32 - v23/rS32 + v34/rD32 - v23/rD32 = i3sb                    =>    v34*(1/rS32 + 1/rD32) - v23*(1/rS32 + 1/rD32) - i3sb     = 0           (18)
//
//      i23 + i3_l = i3                                                     =>    i23 + i3_l - i3 = 0                                                    (19)
//
//		i3st + i3ct = i3													=>    i3st + i3ct - i3 = 0											 		 (20)
//
//		i3sb + i3cb = i3_l													=>	  i3sb + i3cb - i3_l = 0												 (21)
//
//      v31*C3/h - v34*C3/h	- v31_*C3/h + v34_*C3/h = i3                  	=>    v31*C3/h - v34*C3/h - i3 = v31_*C3/h - v34_*C3/h                     	 (22)
//
//      v23*C31/h - v31*C31/h - v23_*C31/h + v31_*C31/h = i3ct				=>    v23*C31/h - v31*C31/h - i3ct = v23_*C31/h - v31_*C31/h         		 (23)
//
//      v34*C32/h - v23*C32/h - v34_*C32/h + v23_*C32/h = i3cb           	=>    v34*C32/h - v23*C32/h - i3cb = v34_*C32/h - v23_*C32/h              	 (24)
//
//
//
//
//    For Cell 0p:
//
//            
//      v34/rS41 - v41/rS41 + v34/rD41 - v41/rD41 = i4st                    =>    v34*(1/rS41 + 1/rD41) - v41*(1/rS41 + 1/rD41) - i4st = 0               (25)
//
//      v44/rS42 - v34/rS42 + v44/rD42 - v34/rD42 = i4sb                	=>    v44*(1/rS42 + 1/rD42) - v34*(1/rS42 + 1/rD42) - i4sb = 0               (26)
//
//      i34 + i4_l = i4                                                     =>    i34 + i4_l - i4 = 0                                                    (27)
//
//		i4st + i4ct = i4													=>    i4st + i4ct - i4 = 0											 		 (28)
//
//		i4sb + i4cb = i4_l													=>	  i4sb + i4cb - i4_l = 0												 (29)
//
//      v41*C4/h - v44*C4/h - v41_*C4/h + v44_*C4/h = i4                  	=>    v41*C4/h - v44*C4/h - i4 = v41_*C4/h - v44_*C4/h                     	 (30)
//
//      v34*C41/h - v41*C41/h - v34_*C41/h + v41_*C41/h = i4ct       		=>    v34*C41/h - v41*C41/h - i4ct = v34_*C41/h - v41_*C41/h         		 (31)
//
//      v44*C42/h - v34*C42/h - v44_*C42/h + v34_*C42/h = i4cb            	=>    v44*C42/h - v34*C42/h - i4cb = v44_*C42/h - v34_*C42/h                 (32)
//
//
//
//
//    For interphase inductance:  	 (Note the negation on the right side of Eq (36) to account for phase reversal in circuit.)
//
//
//      v44/Rp - vs_a/Rp + iLp = i44                                       =>    v44*1/Rp - vs_a*1/Rp + iLp - i44      = 0                               (33)
//      
//      vs_a/Rn - v45/Rn + iLn = i45                                       =>    vs_a*1/Rn - v45*1/Rn + iLn - i45      = 0                               (34)
//      
//      Lp*iLp/h + Mpn*iLn/h - Lp*iLp_/h - Mpn*iLn_/h = v44 - vs_a         =>    Lp*iLp/h + Mpn*iLn/h - v44 + vs_a = Lp*iLp_/h + Mpn*iLn_/h            	 (35)
//     
//      Mnp*iLp/h + Ln*iLn/h - Mnp*iLp_/h - Ln*iLn_/h = vs_a - v45  	   =>    Mnp*iLp/h + Ln*iLn/h - vs_a + v45 = Mnp*iLp_/h + Ln*iLn_/h              (36)
//
//
//    For coupled load circuit (if INDUCTIVE_COUPLED_LOAD is defined):
//
//        
//      (Note: "ib" is in a loop by itself flowing through GND. Thus, it is not part of the DC solution matrix)
//      
//      0 =  i01 - ia - i88                                                =>   i01 - i78 = ia                    (ia is considered a constant here)     (37)
// 
//                i88 = i78                    
//
//
//
//
//      ODE SOLVE
//
//      dqa/dt = ia                                                        =>   dqa/dt = ia                                                              (a)
//
//      dqb/dt = ib                                                        =>   dqb/dt = ib                                                              (b)
//
//      (Lm_a + L_a)*dia/dt + Mab*dib/dt = vs_a - Ra*ia - qa/Ca            =>   (Lm_a + L_a)*dia/dt + Mab*dib/dt = vs_a - Ra*ia - qa/Ca                  (c)
//
//      Mba*dia/dt + (Lm_b + L_b)*dib/dt =  0   - Rb*ib - qb/Cb            =>   Mba*dia/dt + (Lm_b + L_b)*dib/dt = - Rb*ib - qb/Cb                       (d)
//
//
//
//    For simple resistive  load circuit (if INDUCTIVE_COUPLED_LOAD is undefined):
//
//
//      DC SOLVE
//      
//      0 =  i01 - ia - i88                                                =>   i01 - i78 = ia                    (ia is considered a constant here)     (37)
// 
//                i88 = i78   
//
//
//
//      ODE SOLVE 
//
//      ia = CONSTANT_CURRENT_SOURCE_LOAD   (In this case the ODE defined above when INDUCTIVE_COUPLED_LOAD is undefined, simple assigns "ia")           (a)
//
//
//
//
//
//
//    For Cell 3n:
//
//            
//      v45/rS51 - v51/rS51 + v45/rD51 - v51/rD51 = i5st                   =>   v45*(1/rS51 + 1/rD51) - v51*(1/rS51 + 1/rD51) - i5st     = 0              (38)
//
//      v56/rS52 - v45/rS52 + v56/rD52 - v45/rD52 = i5sb                   =>   v56*(1/rS52 + 1/rD52) - v45*(1/rS52 + 1/rD52) - i5sb     = 0              (39)
//
//      i45 + i5_l = i5                                                    =>   i45 + i5_l - i5 = 0                                                       (40)
//
//		i5st + i5ct = i5												   =>   i5st + i5ct - i5 = 0											 		  (41)
//
//		i5sb + i5cb = i5_l												   =>	i5sb + i5cb - i5_l = 0												      (42)
//
//      v51*C5/h - v56*C5/h - v51_*C5/h + v56_*C5/h = i5                   =>   v51*C5/h - v56*C5/h - i5 = v51_*C5/h - v56_*C5/h                          (43)
//
//      v45*C51/h - v51*C51/h - v45_*C51/h + v51_*C51/h = i5ct             =>   v45*C51/h - v51*C51/h - i5ct = v45_*C51/h - v51_*C51/h            	      (44)
//
//      v56*C52/h - v45*C52/h - v56_*C52/h + v45_*C52/h = i5cb             =>   v56*C52/h - v45*C52/h - i5cb = v56_*C52/h - v45_*C52/h                    (45)
//
//
//
//
//    For Cell 2n:
//
//            
//      v56/rS61 - v61/rS61 + v56/rD61 - v61/rD61 = i6st                    =>   v56*(1/rS61 + 1/rD61) - v61*(1/rS61 + 1/rD61) - i6st     = 0            (46)
//
//      v67/rS62 - v56/rS62 + v67/rD62 - v56/rD62 = i6sb                    =>   v67*(1/rS62 + 1/rD62) - v56*(1/rS62 + 1/rD62) - i6sb     = 0            (47)
//
//      i56 + i6_l = i6                                                     =>   i56 + i6_l - i6 = 0                                                     (48)
//
//		i6st + i6ct = i6												    =>   i6st + i6ct - i6 = 0											 		 (49)
//
//		i6sb + i6cb = i6_l												    =>	 i6sb + i6cb - i6_l = 0												  	 (50)
//
//      v61*C6/h - v67*C/h - v61_*C6/h + v67_*C/h = i6                    	=>   v61*C6/h - v67*C/h - i6  = v61_*C6/h - v67_*C/h                         (51)
//
//      v56*C61/h - v61*C61/h - v56_*C61/h + v61_*C61/h = i6ct    	 		=>   v56*C61/h - v61*C61/h - i6ct = v56_*C61/h - v61_*C61/h          		 (52)
//
//      v67*C62/h - v56*C62/h - v67_*C62/h + v56_*C62/h = i6cb     	    	=>   v67*C62/h - v56*C62/h - i6cb = v67_*C62/h - v56_*C62/h                	 (53)
//
//
//
//
//    For Cell 1n:
//
//            
//      v67/rS71 - v71/rS71 + v67/rD71 - v71/rD71 = i7st                    =>   v67*(1/rS71 + 1/rD71) - v71*(1/rS71 + 1/rD71) - i7st     = 0           (54)
//
//      v78/rS72 - v67/rS72 + v78/rD72 - v67/rD72 = i7sb                    =>   v78*(1/rS72 + 1/rD72) - v67*(1/rS72 + 1/rD72) - i7sb     = 0           (55)
//
//      i67 + i7_l = i7                                                     =>   i67 + i7_l - i7 = 0                                                    (56)
//
//		i7st + i7ct = i7												    =>   i7st + i7ct - i7 = 0											  	    (57)
//
//		i7sb + i7cb = i7_l												    =>	 i7sb + i7cb - i7_l = 0												  	(58)
//
//      v71*C7/h - v78*C7/h - v71_*C7/h + v78_*C7/h = i7                    =>   v71*C7/h - v78*C7/h - i7 = v71_*C7/h - v78_*C7/h                       (59)
//
//      v67*C71/h - v71*C71/h - v67_*C71/h + v71_*C71/h = i7ct     			=>   v67*C71/h - v71*C71/h - i7ct = v67_*C71/h - v71_*C71/h          		(60)
//
//      v78*C72/h - v67*C72/h - v78_*C72/h + v67_*C72/h = i7cb            	=>   v78*C72/h - v67*C72/h - i7cb = v78_*C72/h - v67_*C72/h              	(61)
//
//
//
//    For Cell 0n:
//
//            
//      v78/rS81 - v81/rS81 + v78/rD81 - v81/rD81 = i8st                     =>   v78*(1/rS81 + 1/rD81) - v81*(1/rS81 + 1/rD81) - i8st     = 0                        (62)
//
//      v88/rS82 - v78/rS82 + v88/rD82 - v78/rD82 = i8sb                     =>   - v78*(1/rS82 + 1/rD82) - i8sb = - v88*(1/rS82 + 1/rD82) 		(v88 is a constant)   (63)
//
//      i78 + i8_l = i8                                                      =>   i78 + i8_l - i8 = 0                                                                 (64)
//
//		i8st + i8ct = i8												     =>   i8st + i8ct - i8 = 0											  	    			  (65)
//
//		i8sb + i8cb = i8_l												     =>	  i8sb + i8cb - i8_l = 0												  	          (66)
//
//      v81*C8/h - v81_*C8/h - v88*C8/h + v88_*C8/h  = i8                     =>   v81*C8/h - i8 = v81_*C8/h     					 			(v88 is a constant)   (67)
//
//      v78*C81/h - v81*C81/h - v78_*C81/h + v81_*C81/h = i8ct               =>   v78*C81/h - v81*C81/h - i8ct = v78_*C81/h - v81_*C81/h         					  (68)
//
//      v88*C82/h - v78*C82/h - v88_*C82/h + v78_*C82/h = i8cb               =>   - v78*C82/h - i8cb = - v78_*C82/h 							(v88 is a constant)   (69)
//
//
//
//      The known static quantities
//
//      v01 = 1000 VDC
//      v88 = -1000 VDC
//
//      The know quantities after the ODE pass
//
//
//
//
//      v12_, v11_
//      v23_, v21_
//		v34_, v31_
//			  v41_
//		v44_,
//		v45_
//      v56_, v51_
//		v67_, v61_
//		v78_, v71_
//		      v81_
//		iLp_, iLn_
//
//      ia, ib   (Determined from the ODE pass. Note: "ib" is not part of the DC solution)
//
//      The state vector for the DC solution (before reductions)
//
//      | v11 v12 v21 v23 v31 v34 v41 v44
//		  vs_a
//        v45 v51 v56 v61 v67 v71 v78 v81
//        i1 i1_l i01 i1st i1ct i1sb i1cb
//		  i2 i2_l i12 i2st i2ct i2sb i2cb
//        i3 i3_l i23 i3st i3ct i3sb i3cb
//        i4 i4_l i34 i4st i4ct i4sb i4cb
//		  i44
//		  iLp iLn
//        i5 i5_l i45 i5st i5ct i5sb i5cb
//        i6 i6_l i56 i6st i6ct i6sb i6cb
//        i7 i7_l i67 i7st i7ct i7sb i7cb
//        i8 i8_l i78 i8st i8ct i8sb i8cb
//		  i88 |'
//
//      The vector above can be simplified, given that
//     
//      i01 = i12 = i23 = i34 = i44 = ipl
//
//      and 
//
//      i45 = i56 = i67 = i78 = i88 = inl
//
//      The vector to solve (DC solution), is now
//
//
//      | v11 v12 v21 v23 v31 v34 v41 v44
//		  vs_a
//        v45 v51 v56 v61 v67 v71 v78 v81
//        i1 i1_l ipl i1st i1ct i1sb i1cb
//		  i2 i2_l i2st i2ct i2sb i2cb
//        i3 i3_l i3st i3ct i3sb i3cb
//        i4 i4_l i4st i4ct i4sb i4cb
//		  iLp iLn
//        i5 i5_l inl i5st i5ct i5sb i5cb
//        i6 i6_l i6st i6ct i6sb i6cb
//        i7 i7_l i7st i7ct i7sb i7cb
//        i8 i8_l i8st i8ct i8sb i8cb |'
//
//
//
//
//
//
//
//
//      Further simplification of the DC SOVE equations above, putting constants and known quantities after the ODE pass on right side.
//		(NOTE: The equations below relate only to the "Backward Euler" pass. The equations for the "Trapezoidal" pass are not shown
//			   in this outline.)
//
//
//            - v11*(1/rS11 + 1/rD11) - i1st = - v01*(1/rS11 + 1/rD11)                               (1)
//
//              v12*(1/rS12 + 1/rD12) - i1sb = v01*(1/rS12 + 1/rD12)                                 (2)
//
//              ipl + i1_l - i1 = 0                                                                  (3)
//
//				i1st + ilct - il = 0															     (4)
//
//				i1sb + i1cb - i1_l = 0															     (5)
//
//              v11*C1/h - v12*C1/h - i1 = v11_*C1/h - v12_*C1/h                                     (6)
//
//			  - v11*C11/h - i1ct = - v11_*C11/h 													 (7)
//
//			    v12*C12/h - i1cb = v12_*C12/h  														 (8)
//
//              v12*(1/rS21 + 1/rD21) - v21*(1/rS21 + 1/rD21) - i2st = 0                             (9)
//
//              v23*(1/rS22 + 1/rD22) - v12*(1/rS22 + 1/rD22) - i2sb = 0                             (10)
//
//              ipl + i2_l - i2 = 0                                                                  (11)
//
//				i2st + i2ct - i2 = 0																 (12)
//
//				i2sb + i2cb - i2_l = 0																 (13)
//
//              v21*C2/h - v23*C2/h - i2 = v21_*C2/h - v23_*C2/h                                     (14)
//
//				v12*C21/h - v21*C21/h - i2ct = v12_*C21/h - v21_*C21/h   		    				 (15)
//
//				v23*C22/h - v12*C22/h - i2cb = v23_*C22/h - v12_*C22/h               				 (16)
//
//              v23*(1/rS31 + 1/rD31) - v31*(1/rS31 + 1/rD31) - i3st = 0                             (17)
//
//              v34*(1/rS32 + 1/rD32) - v23*(1/rS32 + 1/rD32) - i3sb = 0                             (18)
//
//              ipl + i3_l - i3 = 0                                                                  (19)
//
//				i3st + i3ct - i3 = 0											 					 (20)
//
//				i3sb + i3cb - i3_l = 0																 (21)
//
//              v31*C3/h - v34*C3/h - i3 = v31_*C3/h - v34_*C3/h                                     (22)
//
//				v23*C31/h - v31*C31/h - i3ct = v23_*C31/h - v31_*C31/h          		   			 (23)
//
//				v34*C32/h - v23*C32/h - i3cb = v34_*C32/h - v23_*C32/h               				 (24)
//
//              v34*(1/rS41 + 1/rD41) - v41*(1/rS41 + 1/rD41) - i4st = 0                             (25)
//
//              v44*(1/rS42 + 1/rD42) - v34*(1/rS42 + 1/rD42) - i4sb = 0                             (26)
//
//              ipl + i4_l - i4 = 0                                                                  (27)
//
//				i4st + i4ct - i4 = 0											 					 (28)
//
//				i4sb + i4cb - i4_l = 0												 				 (29)
//
//              v41*C4/h - v44*C4/h - i4 = v41_*C4/h - v44_*C4/h                                     (30)
//
//				v34*C41/h - v41*C41/h - i4ct = v34_*C41/h - v41_*C41/h           					 (31)
//
//				v44*C42/h - v34*C42/h - i4cb = v44_*C42/h - v34_*C42/h               				 (32)
//
//              v44*1/Rp - vs_a*1/Rp + iLp - ipl = 0                                                 (33)
//
//              vs_a*1/Rn - v45*1/Rn + iLn - inl = 0                                                 (34)
//
//				Lp*iLp/h + Mpn*iLn/h - v44 + vs_a = Lp*iLp_/h + Mpn*iLn_/h						     (35)
//
//				Mnp*iLp/h + Ln*iLn/h - vs_a + v45 = Mnp*iLp_/h + Ln*iLn_/h					     	 (36)
//
//              ipl - inl = ia                                                                       (37)
//
//              v45*(1/rS51 + 1/rD51) - v51*(1/rS51 + 1/rD51) - i5st = 0                             (38)
//
//              v56*(1/rS52 + 1/rD52) - v45*(1/rS52 + 1/rD52) - i5sb = 0                             (39)
//
//              inl + i5_l - i5 = 0                                                                  (40)
//
//				i5st + i5ct - i5 = 0											 					 (41)
//
//				i5sb + i5cb - i5_l = 0												      			 (42)
//
//              v51*C5/h - v56*C5/h - i5 = v51_*C5/h - v56_*C5/h                                     (43)
//
//				v45*C51/h - v51*C51/h - i5ct = v45_*C51/h - v51_*C51/h           					 (44)
//
//				v56*C52/h - v45*C52/h - i5cb = v56_*C52/h - v45_*C52/h               	 			 (45)
//
//              v56*(1/rS61 + 1/rD61) - v61*(1/rS61 + 1/rD61) - i6st = 0                             (46)
//
//              v67*(1/rS62 + 1/rD62) - v56*(1/rS62 + 1/rD62) - i6sb = 0                             (47)
//
//              inl + i6_l - i6 = 0                                                                  (48)
//
//				i6st + i6ct - i6 = 0											 		             (49)
//
//				i6sb + i6cb - i6_l = 0												  	             (50)
//
//              v61*C6/h - v67*C6/h - i6  = v61_*C6/h - v67_*C6/h                                    (51)
//
//				v56*C61/h - v61*C61/h - i6ct = v56_*C61/h - v61_*C61/h                   			 (52)
//
//				v67*C62/h - v56*C62/h - i6cb = v67_*C62/h - v56_*C62/h               				 (53)
//
//              v67*(1/rS71 + 1/rD71) - v71*(1/rS71 + 1/rD71) - i7st = 0                             (54)
//
//              v78*(1/rS72 + 1/rD72) - v67*(1/rS72 + 1/rD72) - i7sb = 0                             (55)
//
//              inl + i7_l - i7 = 0                                                                  (56)
//
//				i7st + i7ct - i7 = 0											  	   				 (57)
//
//				i7sb + i7cb - i7_l = 0												  				 (58)
//
//              v71*C7/h - v78*C7/h - i7 = v71_*C7/h - v78_*C7/h                                     (59)
//
//				v67*C71/h - v71*C71/h - i7ct = v67_*C71/h - v71_*C71/h          					 (60)
//
//				v78*C72/h - v67*C72/h - i7cb = v78_*C72/h - v67_*C72/h               				 (61)
//
//              v78*(1/rS81 + 1/rD81) - v81*(1/rS81 + 1/rD81) - i8st = 0                             (62)
//
//            - v78*(1/rS82 + 1/rD82) - i8sb = - v88*(1/rS82 + 1/rD82)                               (63)
//
//              inl + i8_l - i8 = 0                                                                  (64)
//
//				i8st + i8ct - i8 = 0											  	    			 (65)
//
//				i8sb + i8cb - i8_l = 0												  	             (66)
//
//              v81*C8/h - i8 = v81_*C8/h                           		                         (67)
//
//				v78*C81/h - v81*C81/h - i8ct = v78_*C81/h - v81_*C81/h          					 (68)
//
//			  - v78*C82/h - i8cb = - v78_*C82/h             			               	             (69)
//
//      
//       The OLE equations (if INDUCTIVE_COUPLED_LOAD is defined) 
//
//
//             dqa/dt = ia                                                                           (a)
//
//             dqb/dt = ib                                                                           (b)
//
//             (Lm_a + L_a)*dia/dt + Mab*dib/dt = vs_a - Ra*ia - qa/Ca                               (c)
//
//             Mba*dia/dt + (Lm_b + L_b)*dib/dt = - Rb*ib - qb/Cb                                    (d)
//
//
//
//
//
//       The OLE equations (if INDUCTIVE_COUPLED_LOAD is undefined) 
//
//
//             ia = CONSTANT_CURRENT_SOURCE_LOAD                                                     (a)  (ODE used to assign "ia" directly).
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//

typedef enum {
  SRC_FUNC_NULL = -1,
  SRC_FUNC_Vm0,
  SRC_FUNC_Vm1,
  SRC_FUNC_Vm2,
  SRC_FUNC_Vm3,
  SRC_FUNC_ECtrl1,
  SRC_FUNC_ECtrl2,
  SRC_FUNC_ECtrl3,
  SRC_FUNC_ECtrl4,
  SRC_FUNC_ECtrl5,
  SRC_FUNC_ECtrl6,
  SRC_FUNC_ECtrl7,
  SRC_FUNC_ECtrl8,
  SRC_FUNC_Vref,

} SRC_FUNCTION;


typedef enum {
  ODE_FUNC_NULL = -1,
  ODE_FUNC_qa,
  ODE_FUNC_qb,
  ODE_FUNC_ia_ib_1,
  ODE_FUNC_ia_ib_2,

} ODE_FUNCTION;


typedef enum {
  CTRL_FUNC_NULL = -1,

} CTRL_FUNCTION;






typedef enum {
  COEF_FUNC_NULL = -1,
  COEF_FUNC_recp_rSrD11, 
  COEF_FUNC_recp_rSrD12, 
  COEF_FUNC_C1_h,
  COEF_FUNC_C11_h,
  COEF_FUNC_C12_h,
  COEF_FUNC_v11_v12_h_prev_C1,
  COEF_FUNC_minus_v11_h_prev_minus_PlusBus_C11,
  COEF_FUNC_v12_h_prev_plus_PlusBus_C12,
  COEF_FUNC_recp_rSrD21, 
  COEF_FUNC_recp_rSrD22, 
  COEF_FUNC_C2_h,
  COEF_FUNC_C21_h,
  COEF_FUNC_C22_h,
  COEF_FUNC_v21_v23_h_prev_C2,
  COEF_FUNC_v12_v21_h_prev_C21,
  COEF_FUNC_v23_v12_h_prev_C22,
  COEF_FUNC_recp_rSrD31, 
  COEF_FUNC_recp_rSrD32, 
  COEF_FUNC_C3_h,
  COEF_FUNC_C31_h,
  COEF_FUNC_C32_h,
  COEF_FUNC_v31_v34_h_prev_C3,
  COEF_FUNC_v23_v31_h_prev_C31,
  COEF_FUNC_v34_v23_h_prev_C32,
  COEF_FUNC_recp_rSrD41, 
  COEF_FUNC_recp_rSrD42, 
  COEF_FUNC_C4_h,
  COEF_FUNC_C41_h,
  COEF_FUNC_C42_h,
  COEF_FUNC_v41_v44_h_prev_C4,
  COEF_FUNC_v34_v41_h_prev_C41,
  COEF_FUNC_v44_v34_h_prev_C42,
  COEF_FUNC_Lp_h,
  COEF_FUNC_Mpn_h,
  COEF_FUNC_Mnp_h,
  COEF_FUNC_Ln_h,
  COEF_FUNC_iLp_iLn_h_prev_Lp_Mpn,
  COEF_FUNC_iLp_iLn_h_prev_Mnp_Ln,
  COEF_FUNC_ia,
  COEF_FUNC_recp_rSrD51, 
  COEF_FUNC_recp_rSrD52, 
  COEF_FUNC_C5_h,
  COEF_FUNC_C51_h,
  COEF_FUNC_C52_h,
  COEF_FUNC_v51_v56_h_prev_C5,
  COEF_FUNC_v45_v51_h_prev_C51,
  COEF_FUNC_v56_v45_h_prev_C52,
  COEF_FUNC_recp_rSrD61, 
  COEF_FUNC_recp_rSrD62, 
  COEF_FUNC_C6_h,
  COEF_FUNC_C61_h,
  COEF_FUNC_C62_h,
  COEF_FUNC_v61_v67_h_prev_C6,
  COEF_FUNC_v56_v61_h_prev_C61,
  COEF_FUNC_v67_v56_h_prev_C62,
  COEF_FUNC_recp_rSrD71, 
  COEF_FUNC_recp_rSrD72,
  COEF_FUNC_C7_h,
  COEF_FUNC_C71_h,
  COEF_FUNC_C72_h,
  COEF_FUNC_v71_v78_h_prev_C7,
  COEF_FUNC_v67_v71_h_prev_C71,
  COEF_FUNC_v78_v67_h_prev_C72,
  COEF_FUNC_recp_rSrD81, 
  COEF_FUNC_recp_rSrD82, 
  COEF_FUNC_C8_h,
  COEF_FUNC_C81_h,
  COEF_FUNC_C82_h,
  COEF_FUNC_v81_h_prev_plus_MinusBus_C8,
  COEF_FUNC_v78_v81_h_prev_C81,
  COEF_FUNC_minus_v78_h_prev_minus_MinusBus_C82,


} COEF_FUNCTION;



typedef enum {
  SWITCH_FUNC_NULL = -1,
  SWITCH_FUNC_rS11,
  SWITCH_FUNC_rS12,
  SWITCH_FUNC_rS21,
  SWITCH_FUNC_rS22,
  SWITCH_FUNC_rS31,
  SWITCH_FUNC_rS32,
  SWITCH_FUNC_rS41,
  SWITCH_FUNC_rS42,
  SWITCH_FUNC_rS51,
  SWITCH_FUNC_rS52,
  SWITCH_FUNC_rS61,
  SWITCH_FUNC_rS62,
  SWITCH_FUNC_rS71,
  SWITCH_FUNC_rS72,
  SWITCH_FUNC_rS81,
  SWITCH_FUNC_rS82,

} SWITCH_FUNCTION;


typedef enum {
  SPICE_FUNC_NULL = -1,
  SPICE_FUNC_MultiLevel,
} SPICE_FUNCTION;

//define this only for debugging the matrix of any declared SpiceObjects. The size must appropriately.
//#define DEBUG_SPICE_MATRIX
//#define SPICE_DEBUG_MATRIX_SIZE 69

//if "DEBUG_SPICE_MATRIX" defined above, as an option we can also elect to dump the "a_matrix" and "a_equate_vector" for
//analysis using Octave.
//#define DUMP_MATRIX_EQUATE_VECTOR_TO_FILE_FOR_EXTERNAL_ANALYSIS


#include "../Simulation/Simulation.hpp"
#include <fcntl.h>           /* For O_* constants */
#include <sys/stat.h>        /* For mode constants */
#include <semaphore.h>
#include <sys/wait.h>


//Parameters....

// Cell parameters
#define C1   .0004 
#define C2   .0004
#define C3   .0004
#define C4   .0004
#define C5   .0004
#define C6   .0004
#define C7   .0004
#define C8   .0004


// Switch coupling capacitor parameters
	//(NOTE: I increased these form .0000001 in original App_SpiceTest.hpp file to .0000005.
	//		 The dv/dt was to great causing voltage spikes (e.g.probably because diodes in SwitchObjects
	//       could not respond fast enough to change in current direction.)
#define C11	 .0000005
#define C12	 .0000005
#define C21	 .0000005
#define C22	 .0000005
#define C31	 .0000005
#define C32	 .0000005
#define C41	 .0000005
#define C42	 .0000005
#define C51	 .0000005
#define C52	 .0000005
#define C61	 .0000005
#define C62	 .0000005
#define C71	 .0000005
#define C72	 .0000005
#define C81	 .0000005
#define C82	 .0000005



//set this to the same value as the initial "OffRes" of the "SwitchObject"
//that controls switch coeficients.
#define INIT_COEF_RS_RD  1000000.0


 
// Load parameters (if INDUCTIVE_COUPLED_LOAD is defined)
#define Lm_a  1e-3               
#define Lm_b  Lm_a
#define Mab  .98e-3          
#define Mba  Mab

#define L_a .1e-3            
#define L_b L_a

#define Ra 1.0
#define Rb Ra

#define Ca 300e-6  
#define Cb Ca


// Load parameters (if INDUCTIVE_COUPLED_LOAD is undefined)
//#define CONSTANT_CURRENT_SOURCE_LOAD  	-20.0
// For the 3-phase version, we need to make current source definition a variable.
double CONSTANT_CURRENT_SOURCE_LOAD = 	-20.0;

// Interphase filter parameters

#define Lp 	.25e-3  //4.1e-3			//IMPORTANT: For RAIL_TO_RAIL_STEP_VREF, it seems is though we cannot exceed
							//  		 "tank frequecy" is a issue (W = 2*pi*f = 1/(L*C)**.5). Assume equivilent capacitance
							//			 (which is a variable) is .0002 with rail-to-rail period of .008 sec. This comes out
							//			 to approx. 8.1e-3 for interface inductance (Lp + Ln). Going greater then
							//           this causes ipl/inl current instability.
#define Ln  Lp
#define Mpn 0
#define Mnp Mpn 

#define Rp  100000.0
#define Rn  Rp

#define RECEP_Rp (1.0/Rp)
#define RECEP_Rn (1.0/Rn)

#define SWITCH_ENABLE_TIME 0  //.000750

#define HALF_PWM_CYCLE  .0005		//.00005
#define PWM_GAIN 5.0              


#define QUANTUM_PERIOD .0000125  


//#define PLUS_DC_BUS_VOLTAGE  1000.0
//For MultiLevelDCTest 3Phase test, this is a variable
double PLUS_DC_BUS_VOLTAGE = 1000.0;

#define MINUS_DC_BUS_VOLTAGE -1000.0

#define PI   3.1415926535897932384626433832795
  
        // Indices for the vector to solve.
#define Ind_v11     0
#define Ind_v12     1
#define Ind_v21     2
#define Ind_v23     3
#define Ind_v31     4
#define Ind_v34     5
#define Ind_v41     6
#define Ind_v44     7
#define Ind_vs_a    8
#define Ind_v45     9
#define Ind_v51     10
#define Ind_v56     11
#define Ind_v61     12
#define Ind_v67     13
#define Ind_v71     14
#define Ind_v78     15
#define Ind_v81     16
#define Ind_i1      17
#define Ind_i1_l    18
#define Ind_ipl     19
#define Ind_i1st	20
#define Ind_i1ct	21
#define Ind_i1sb	22
#define Ind_i1cb	23
#define Ind_i2    	24
#define Ind_i2_l  	25
#define Ind_i2st	26
#define Ind_i2ct	27
#define Ind_i2sb	28
#define Ind_i2cb	29
#define Ind_i3      30
#define Ind_i3_l    31
#define Ind_i3st	32
#define Ind_i3ct	33
#define Ind_i3sb	34
#define Ind_i3cb	35
#define Ind_i4      36
#define Ind_i4_l    37
#define Ind_i4st	38
#define Ind_i4ct	39
#define Ind_i4sb	40
#define Ind_i4cb	41
#define Ind_iLp		42
#define Ind_iLn		43
#define Ind_i5      44
#define Ind_i5_l   	45
#define Ind_inl     46
#define Ind_i5st	47
#define Ind_i5ct	48
#define Ind_i5sb	49
#define Ind_i5cb	50
#define Ind_i6      51
#define Ind_i6_l    52
#define Ind_i6st	53
#define Ind_i6ct	54
#define Ind_i6sb	55
#define Ind_i6cb	56
#define Ind_i7      57
#define Ind_i7_l    58
#define Ind_i7st	59
#define Ind_i7ct	60
#define Ind_i7sb	61
#define Ind_i7cb	62
#define Ind_i8      63
#define Ind_i8_l    64
#define Ind_i8st	65
#define Ind_i8ct	66
#define Ind_i8sb	67
#define Ind_i8cb	68






// Define if "Vref" should supply a sin() command. If undefined, "Vref" supplies a zero command..
//#define APPLY_SINEWAVE_VREF_COMMAND

// Define if  load connected to "vs_a" represents the selected inductively coupled load used for this test. 
// If undefined, the connection to "vs_a" is a constant current source load.
//#define INDUCTIVE_COUPLED_LOAD

// Define for sinusoidal varing of CONSTANT_CURRENT_SOURCE_LOAD value above. Valid only if INDUCTIVE_COUPLED_LOAD and RAIL_TO_RAIL_STEP_VREF are undefined.
// Assumes APPLY_SINEWAVE_VREF_COMMAND is defined (but does not have to be).
//#define SINUSOIDAL_CURRENT_SOURCE_LOAD

// Define if circulation compensation is to applied to each cell.
#define ENABLE_CELL_CIRCULATION

// When running cell circulation, the all of the development upto and includinng the (initial) Provisional Patent paper
// used this state un-defined. If defined, we are working on the second phase of this paper that has to do with resonance mode that
// is operating between levels +500 and -500 volts (e.g. Tests starting with "Figure_35_36_37_38_39" test below).
#define CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT


// A table to hold the "CircState" variables of SrcObject's ECtrl1 through ECtrl8. These states
// are controlled by SrcObject Vm0. A two dimensional table, the second dimension being the control
// states for ECtrl1 through ECtrl8. This sequence is modeled after the algorythm in EightCellPhase_modulator.vhd
//backed up in external hard drive "E:\DELLDUO_Machine_BACKUP\FanoutCellHubDesign\FanoutBoard\hardware"

	//Note, when running in this mode CIRCULAR_STATE_RATE  was set to "0".

unsigned int CircStateTable[][8] = {{0, 1, 2, 3, 0, 1, 2, 3},
								    {1, 2, 3, 0, 0, 1, 2, 3},
								    {1, 2, 3, 0, 1, 2, 3, 0},
								    {2, 3, 0, 1, 1, 2, 3, 0},
								    {2, 3, 0, 1, 2, 3, 0, 1},
								    {3, 0, 1, 2, 2, 3, 0, 1},
								    {3, 0, 1, 2, 3, 0, 1, 2},
								    {0, 1, 2, 3, 3, 0, 1, 2}};


	//Count of the period 2 *  HALF_PWM_CYCLE that increments the index to the first dimension of
	//"CircStateTable" above.
#define CIRCULAR_STATE_RATE 0  			//0 - 2*HALF_PWM_CYCLE, 1 - 4*HALF_PWM_CYCLE, etc...

	//Define to set Omega to a value to produce rail-to-rail switching
	////(NOTE: RAIL_TO_RAIL_STEP_REF must be un-defined below for this control to be functional).
//#define ENABLE_OMEGA_CTRL
	//Set the Omega control (1.0 PU produces no shift)
#define OMEGA_CTRL_FACTOR .2


	//Define to dump "vs_a" to either "vs_a_output_omega.dat" or "vs_a_output_no_omega.dat"
	//depending on the setting of ENABLE_OMEGA_CTRL (Usually used if APPLY_SINEWAVE_VREF_COMMAND and
	//INDUCTIVE_COUPLED_LOAD are defined).
	//
	//Run like this in octave to see results:
	//
	//		octave:1> run_filter_plots
	//
	//
	//If only interested in the filtered "vs_a" under any test condition with ENABLE_OMEGA_CTRL disabled, run like this:
	//
	//
	//		octave:1> run_filter_plots_no_omega
	//
//#define DUMP_VS_A_OUTPUT_SIGNAL




	//Define to step "vs_a" between  PLUS_DC_BUS_VOLTAGE and  MINUS_DC_BUS_VOLTAGE and a apply a DC voltage command reference.
	//(NOTE: INDUCTIVE_COUPLED_LOAD must be un-defined for this test.
	//		 ENABLE_OMEGA_CTRL should be un-defined. But if it isn't it effect is ignored.
	//       Also, this test implies that APPLY_SINEWAVE_VREF_COMMAND is undefined, and
	//       in reality like ENABLE_OMEGA_CTRL, is ignored. The state of ENABLE_CELL_CIRCULATION
	//		 can be defined or un-defined for this test
	//       SINUSOIDAL_CURRENT_SOURCE_LOAD must be undefined for this test.)
#define RAIL_TO_RAIL_STEP_VREF

//Count of the period  2 * HALF_PWM_CYCLE that increments the index to the first dimension of
//"RailToRailStateTable" above.
#define RAIL_TO_RAIL_STATE_RATE 0         //0 - 2*HALF_PWM_CYCLE, 1 - 4*HALF_PWM_CYCLE, etc...

	//Define to step "vs_a" between  PLUS_DC_BUS_VOLTAGE and  MINUS_DC_BUS_VOLTAGE and apply a sin() voltage command reference instead
	//of a DC voltage command reference.
	//(NOTE: INDUCTIVE_COUPLED_LOAD must be un-defined for this test.
	//       RAIL_TO_RAIL_STEP_VREF must be defined for this test.
	//		 ENABLE_OMEGA_CTRL should be un-defined. But if it isn't it effect is ignored.
	//       APPLY_SINEWAVE_VREF_COMMAND is ignored for this test..
	//       The state of ENABLE_CELL_CIRCULATION can be defined or un-defined for this test.
	//		 SINUSOIDAL_CURRENT_SOURCE_LOAD must be defined for this test.)
//#define RAIL_TO_RAIL_AC_TEST

#ifdef RAIL_TO_RAIL_AC_TEST

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {0, 		 		0,					PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {0,		   			0,					PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {0,		 			0,					PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {0, 				0, 					PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees


#else




//#define Figure_5_6_8
#ifdef Figure_5_6_8

	//0 biased carrier (at 0 amps) for Figure 5 and 6
	//0 biased carrier (at 20 amps) for Figure 8

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

#endif

//#define Figure_11
#ifdef Figure_11


	//125 volt biased carrier (at 20 amps)



double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,		1.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,	   -1.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-0.5), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    1.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-0.5), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	   -1.0,		0.0,		0.0}};	//315 Degrees


#endif




//#define Figure_14_15
#ifdef Figure_14_15

	//80.0 amps (same volt-second bias as "Figure_16_17")

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),		1.0,		1.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),	   -1.0,	   -1.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(-0.25),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

#endif


//#define Figure_16_17
#ifdef Figure_16_17

	//80.0 amps (same volt-second bias as "Figure_14_15")

double RailToRailStateTable[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(0.0),  	PWM_GAIN*(0),		1.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,		1.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,	   -1.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(0.0),  	PWM_GAIN*(0),	   -1.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.125), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	   -1.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(-0.25),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.125), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    1.0,		0.0,		0.0}};	//315 Degrees

#endif


//#define Figure_18_a_b_c
#ifdef Figure_18_a_b_c

//Set to 0 amps.
//Run to about 27 percent execution...
//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_a.dat"
//Set to .5 amps.
//Run to about 27 percent execution...
//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_b.dat"
//Set to 1 amps.
//Run to about 27 percent execution...
//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_c.dat"
//Run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees



#endif

//#define Figure_19_a
#ifdef Figure_19_a

	//Set to 40.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_a.dat"
	//Do the same with Figure 19_b and Figure_19_c below then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,		1.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,	   -1.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-0.5), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    1.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-0.5), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	   -1.0,		0.0,		0.0}};	//315 Degrees

#endif

//#define Figure_19_b
#ifdef Figure_19_b

	//Set to -24.0 amps  (bias is set to - 24/40 the bias of Figure_19_a)
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_b.dat"
	//Do the same with Figure_19_a above and Figure_19_c below then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(0.7),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	1.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(0.7),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   -1.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-0.3), 	PWM_GAIN*(0),	    0.0,		1.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-0.3), 	PWM_GAIN*(0),	    0.0,	   -1.0,		0.0}};	//315 Degrees

#endif


//#define Figure_19_c
#ifdef Figure_19_c

	//Set to -16.0 amps  (bias is set to - 16/40 the bias of Figure_19_a)
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_c.dat"
	//Do the same with Figure_19_a and Figure_19_b above then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(0.8),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	1.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(0.8),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   -1.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-0.2), 	PWM_GAIN*(0),	    0.0,		1.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-0.2), 	PWM_GAIN*(0),	    0.0,	   -1.0,		0.0}};	//315 Degrees

#endif


//#define Figure_20_21_22
#ifdef Figure_20_21_22


	//125 volt biased carrier (at 20 amps)

	//All other plots above,  Lp/Ln  -> .25e-3
	//Figure 20 and Figure 21  Lp/Ln -> .02e-3
    //Figure 22                Lp/Ln -> 2.0e-3



double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,		1.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.5),		PWM_GAIN*(0),	   	0.0,	   -1.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-0.5), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    1.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-0.5), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	   -1.0,		0.0,		0.0}};	//315 Degrees


#endif




//#define Figure_32
#ifdef Figure_32

	//80.0 amps  (demonstrates we need one more degree of freedom (carrier frequency) to compensate for the fact that the
	//		      tank frequency is changing).


double RailToRailStateTable[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(1.0),  	PWM_GAIN*(0),	    0.0,	   	0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),	   -1.0,	   -1.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(-0.25),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),	    1.0,	    1.0,		0.0}};	//315 Degrees


#endif


//#define Figure_33
#ifdef Figure_33

	//80.0 amps  (compensated version of "Figure_32")
	//
	//  With HALF_PWM_CYCLE set to .0003


double RailToRailStateTable[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(1.0),  	PWM_GAIN*(0),	    0.0,	   	0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),	   -1.0,	   -1.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(-0.25),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),	    1.0,	    1.0,		0.0}};	//315 Degrees


#endif




//#define Figure_34_a
#ifdef Figure_34_a

	//Set to 80.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_a.dat"
	//Do the same with Figure 34_b and Figure_34_c below then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(1.0),  	PWM_GAIN*(0),	    0.0,	   	0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),	   -1.0,	   -1.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(-0.25),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.375),	PWM_GAIN*(0.25),  	PWM_GAIN*(0),	    1.0,	    1.0,		0.0}};	//315 Degrees
#endif

//#define Figure_34_b
#ifdef Figure_34_b

	//Set to 0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_b.dat"
	//Do the same with Figure_34_a above and Figure_34_c below then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

#endif


//#define Figure_34_c
#ifdef Figure_34_c

	//Set to -80.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_c.dat"
	//Do the same with Figure_34_a and Figure_34_b above then run "plot_figure_18_19.m"
	//(Look at the shifting in this table to get alignment with the other two phases. Why did I not have to do this in "Figure_19_c"?)

double RailToRailStateTable[][6] = {{PWM_GAIN*(-1.0), 	PWM_GAIN*(-1.0), 	PWM_GAIN*(0),	    0.0,	    0.0,		0.0},   //  0 Degress
									{PWM_GAIN*(-0.375),	PWM_GAIN*(-0.25),   PWM_GAIN*(0),	   -1.0,	   -1.0,		0.0},  	// 45 Degrees
                                    {PWM_GAIN*(0.5),  	PWM_GAIN*(0.25),	PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(-0.375), PWM_GAIN*(-0.25),   PWM_GAIN*(0),	    1.0,	    1.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),	PWM_GAIN*(0),	    0.0,	    0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0),	PWM_GAIN*(-1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-1.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0}};	//315 Degrees


#endif

	//At this point this file was saved to "App_MultiLevelDCTest-(Provisinal-Patent).hpp" and we are continuing on with
	//tests that have to do with running "resonance" mode between the +500 and -500 levels.
	//Starting here, "CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT" above must be defined (except where noted below).

//#define Figure_35_36_37_38_39
#ifdef Figure_35_36_37_38_39

	//Conditions:
	//
	//   Set to 20.0 amps
	//
	//	 For Figure 35 and Figure 36, ENABLE_CELL_CIRCULATION is undefined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is undefined.
	//
	//   For Figure 37 and Figure 38 and 39, ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.
	//


double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                	{PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                	{PWM_GAIN*(1.0),   	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                	{PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                	{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                	{PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                	{PWM_GAIN*(-1.0),  	PWM_GAIN*(0.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                	{PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

#endif



//#define Figure_40_41
#ifdef Figure_40_41

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to 20.0 amps
	//
	//   MODULATE_STATE_TABLE definition will instruct "Vref::SrcFunction() to insert Modulate_StateTable[][] into
	//   RailToRailStateTable[][] at some predefined time.


#define MODULATE_STATE_TABLE

#define MODULATE_STATE_TABLE_BEGIN  .04
#define MODULATE_STATE_TABLE_END    .06


double RailToRailStateTable[8][6];

double Mod_StateTableOff[][6] =    {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                            		{PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                            		{PWM_GAIN*(1.0),   	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                            		{PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                            		{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                            		{PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                            		{PWM_GAIN*(-1.0),  	PWM_GAIN*(0.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                            		{PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees


double Mod_StateTableOn[][6] = 	   {{PWM_GAIN*(1.0),	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(0),  		PWM_GAIN*(0),	    0.0,	   	0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.25),	PWM_GAIN*(0),  		PWM_GAIN*(0),	   -1.0,	    0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.25),	PWM_GAIN*(0),  		PWM_GAIN*(0),	    1.0,	    0.0,		0.0}};	//315 Degrees
#endif



//#define Figure_42_43
#ifdef Figure_42_43

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.
	//
	//   Set to 20.0 amps
	//
	//   MODULATE_STATE_TABLE definition will instruct "Vref::SrcFunction() to insert Modulate_StateTable[][] into
	//   RailToRailStateTable[][] at some predefined time.


#define MODULATE_STATE_TABLE

#define MODULATE_STATE_TABLE_BEGIN  .04
#define MODULATE_STATE_TABLE_END    .06


double RailToRailStateTable[8][6];

double Mod_StateTableOff[][6] =    {{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                	{PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                	{PWM_GAIN*(1.0),   	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                	{PWM_GAIN*(1.0),  	PWM_GAIN*(0.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                	{PWM_GAIN*(0.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                	{PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                	{PWM_GAIN*(-1.0),  	PWM_GAIN*(0.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                	{PWM_GAIN*(-1.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees


double Mod_StateTableOn[][6] = 	   {{PWM_GAIN*(1.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(0.0),  	PWM_GAIN*(0.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees



#endif




//#define Figure_44_a
#ifdef Figure_44_a

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to 20.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_a.dat"
	//Do the same with Figure 44_b and Figure_44_c below then run "plot_figure_18_19.m"


double RailToRailStateTable[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(0.0),  	PWM_GAIN*(0.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees





#endif

//#define Figure_44_b
#ifdef Figure_44_b

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to 0.001 amps.  (NOTE: If set to "0" like Figure_18_a_b_c above, current design for cell selection circulator does not allow changes in cell selection. Must be fixed!!!)
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_b.dat"
	//Do the same with Figure_44_a above and Figure_44_c below then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

#endif


//#define Figure_44_c
#ifdef Figure_44_c

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to -20.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_c.dat"
	//Do the same with Figure_44_a and Figure_44_b above then run "plot_figure_18_19.m"
	//(Look at the shifting in this table to get alignment with the other two phases. Why did I not have to do this in "Figure_19_c"?)

double RailToRailStateTable[][6] = {{PWM_GAIN*(-1.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(0.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(0.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(0.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(-1.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-1.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-1.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees


#endif





//#define Figure_45_a
#ifdef Figure_45_a

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to 20.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_a.dat"
	//Do the same with Figure 45_b and Figure_45_c below then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(0),  		PWM_GAIN*(0),	    0.0,	   	0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.25),	PWM_GAIN*(0),  		PWM_GAIN*(0),	   -1.0,	    0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.25),	PWM_GAIN*(0),  		PWM_GAIN*(0),	    1.0,	    0.0,		0.0}};	//315 Degrees
#endif

//#define Figure_45_b
#ifdef Figure_45_b

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to 0.001 amps.  (NOTE: If set to "0" like Figure_18_a_b_c above, current design for cell selection circulator does not allow changes in cell selection. Must be fixed!!!)
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_b.dat"
	//Do the same with Figure_45_a above and Figure_45_c below then run "plot_figure_18_19.m"

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

#endif


#define Figure_45_c
#ifdef Figure_45_c

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to -20.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_c.dat"
	//Do the same with Figure_45_a and Figure_45_b above then run "plot_figure_18_19.m"
	//(Look at the shifting in this table to get alignment with the other two phases. Why did I not have to do this in "Figure_19_c"?)

double RailToRailStateTable[][6] = {{PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,	    0.0,		0.0},   //  0 Degress
									{PWM_GAIN*(-0.25),	PWM_GAIN*(0),   	PWM_GAIN*(0),	   -1.0,	    0.0,		0.0},  	// 45 Degrees
                                    {PWM_GAIN*(0.5),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(-0.25), 	PWM_GAIN*(0),   	PWM_GAIN*(0),	    1.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	    0.0,	    0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0}};	//315 Degrees


#endif





//#define Figure_46_a
#ifdef Figure_46_a

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to 20.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_a.dat"
	//Do the same with Figure 46_b and Figure_46_c below then run "plot_figure_ipl_instead_of_vs_a.m"

#define DUMP_IPL_SIGNAL_INSTEAD_OF_VS_A_SIGNAL

double RailToRailStateTable[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(1.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(1.0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(0.0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(0.0),  	PWM_GAIN*(0.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.0), 	PWM_GAIN*(0.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees
#endif

//#define Figure_46_b
#ifdef Figure_46_b

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to 0.001 amps.  (NOTE: If set to "0" like Figure_18_a_b_c above, current design for cell selection circulator does not allow changes in cell selection. Must be fixed!!!)
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_b.dat"
	//Do the same with Figure_46_a above and Figure_46_c below then run "plot_figure_ipl_instead_of_vs_a.m"

#define DUMP_IPL_SIGNAL_INSTEAD_OF_VS_A_SIGNAL

double RailToRailStateTable[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

#endif


//#define Figure_46_c
#ifdef Figure_46_c

	//ENABLE_CELL_CIRCULATION is defined and CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined.

	//Set to -20.0 amps
	//Run to about 27 percent execution...
	//copy "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_fig_c.dat"
	//Do the same with Figure_46_a and Figure_46_b above then run "plot_figure_ipl_instead_of_vs_a.m"


#define DUMP_IPL_SIGNAL_INSTEAD_OF_VS_A_SIGNAL

double RailToRailStateTable[][6] = {{PWM_GAIN*(-1.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(0.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(0.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(0.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(-1.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-1.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(-1.0), 	PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees


#endif





#endif  //RAIL_TO_RAIL_AC_TEST

	//Thes variable added for MultLevelDCTest 3 Phase.
double vs_plotdata;
double ipl_bus_current;




// Control all the ploting in this application by commenting/un-commenting  the definitions below...

  //#define PLOT_SRCOBJECT_Vm0
  //#define PLOT_SRCOBJECT_Vm1
  //#define PLOT_SRCOBJECT_Vm2
  //#define PLOT_SRCOBJECT_Vm3


//#define PLOT_SRCOBJECT_PROBES_ECtrl1
//#define PLOT_SRCOBJECT_PROBES_ECtrl2
//#define PLOT_SRCOBJECT_PROBES_ECtrl3
//#define PLOT_SRCOBJECT_PROBES_ECtrl4
//#define PLOT_SRCOBJECT_PROBES_ECtrl5
//#define PLOT_SRCOBJECT_PROBES_ECtrl6
//#define PLOT_SRCOBJECT_PROBES_ECtrl7
//#define PLOT_SRCOBJECT_PROBES_ECtrl8

	#define PLOT_SRCOBJECT_Vref


  //#define PLOT_ODEOBJECT_qa
  //#define PLOT_ODEOBJECT_qb
  //#define PLOT_ODEOBJECT_ia_ib_1
  //#define PLOT_ODEOBJECT_ia_ib_2





//#define PLOT_SWITCHOBJECT_PROBES_rS11
//#define DO_PROBE_rS11_res
//#define DO_PROBE_rS11_cur
//#define DO_PROBE_rS11_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS12
//#define DO_PROBE_rS12_res
//#define DO_PROBE_rS12_cur
//#define DO_PROBE_rS12_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS21
//#define DO_PROBE_rS21_res
//#define DO_PROBE_rS21_cur
//#define DO_PROBE_rS21_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS22
//#define DO_PROBE_rS22_res
//#define DO_PROBE_rS22_cur
//#define DO_PROBE_rS22_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS31
//#define DO_PROBE_rS31_res
//#define DO_PROBE_rS31_cur
//#define DO_PROBE_rS31_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS32
//#define DO_PROBE_rS32_res
//#define DO_PROBE_rS32_cur
//#define DO_PROBE_rS32_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS41
//#define DO_PROBE_rS41_res
//#define DO_PROBE_rS41_cur
//#define DO_PROBE_rS41_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS42
//#define DO_PROBE_rS42_res
//#define DO_PROBE_rS42_cur
//#define DO_PROBE_rS42_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS51 
//#define DO_PROBE_rS51_res
//#define DO_PROBE_rS51_cur
//#define DO_PROBE_rS51_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS52
//#define DO_PROBE_rS52_res
//#define DO_PROBE_rS52_cur
//#define DO_PROBE_rS52_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS61
//#define DO_PROBE_rS61_res
//#define DO_PROBE_rS61_cur
//#define DO_PROBE_rS61_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS62
//#define DO_PROBE_rS62_res
//#define DO_PROBE_rS62_cur
//#define DO_PROBE_rS62_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS71
//#define DO_PROBE_rS71_res
//#define DO_PROBE_rS71_cur
//#define DO_PROBE_rS71_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS72
//#define DO_PROBE_rS72_res
//#define DO_PROBE_rS72_cur
//#define DO_PROBE_rS72_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS81
//#define DO_PROBE_rS81_res
//#define DO_PROBE_rS81_cur
//#define DO_PROBE_rS81_gate

//#define PLOT_SWITCHOBJECT_PROBES_rS82
//#define DO_PROBE_rS82_res
//#define DO_PROBE_rS82_cur
//#define DO_PROBE_rS82_gate


//#define PLOT_SPICEOBJECT_PROBES_solution_vector
  //#define DO_PROBE_v11    
  //#define DO_PROBE_v12    
  //#define DO_PROBE_v21    
  //#define DO_PROBE_v23    
  //#define DO_PROBE_v31    
  //#define DO_PROBE_v34    
  //#define DO_PROBE_v41    
  //#define DO_PROBE_v44
//#define DO_PROBE_vs_a
  //#define DO_PROBE_v45
  //#define DO_PROBE_v51    
  //#define DO_PROBE_v56    
  //#define DO_PROBE_v61    
  //#define DO_PROBE_v67    
  //#define DO_PROBE_v71    
  //#define DO_PROBE_v78    
  //#define DO_PROBE_v81    
  //#define DO_PROBE_i1   
  //#define DO_PROBE_i1_l     
 //#define DO_PROBE_ipl
  //#define DO_PROBE_i1st
  //#define DO_PROBE_i1ct
  //#define DO_PROBE_i1sb
  //#define DO_PROBE_i1cb
 //#define DO_PROBE_i2
  //#define DO_PROBE_i2_l
  //#define DO_PROBE_i2st
  //#define DO_PROBE_i2ct
  //#define DO_PROBE_i2sb
  //#define DO_PROBE_i2cb
  //#define DO_PROBE_i3 
  //#define DO_PROBE_i3_l
  //#define DO_PROBE_i3st
  //#define DO_PROBE_i3ct
  //#define DO_PROBE_i3sb
  //#define DO_PROBE_i3cb
  //#define DO_PROBE_i4
  //#define DO_PROBE_i4_l
  //#define DO_PROBE_i4st
  //#define DO_PROBE_i4ct
  //#define DO_PROBE_i4sb
  //#define DO_PROBE_i4cb
  //#define DO_PROBE_iLp
  //#define DO_PROBE_iLn
  //#define DO_PROBE_i5
  //#define DO_PROBE_i5_l
 // #define DO_PROBE_inl
  //#define DO_PROBE_i5st
  //#define DO_PROBE_i5ct
  //#define DO_PROBE_i5sb
  //#define DO_PROBE_i5cb
  //#define DO_PROBE_i6 
  //#define DO_PROBE_i6_l   
  //#define DO_PROBE_i6st
  //#define DO_PROBE_i6ct
  //#define DO_PROBE_i6sb
  //#define DO_PROBE_i6cb
  //#define DO_PROBE_i7
  //#define DO_PROBE_i7_l     
  //#define DO_PROBE_i7st
  //#define DO_PROBE_i7ct
  //#define DO_PROBE_i7sb
  //#define DO_PROBE_i7cb
  //#define DO_PROBE_i8
  //#define DO_PROBE_i8_l 
  //#define DO_PROBE_i8st
  //#define DO_PROBE_i8ct
  //#define DO_PROBE_i8sb
  //#define DO_PROBE_i8cb





   
// **** SrcObject Classes ********************************

// ---- Vm0 ---------------------------------------------

class Vm0 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SrcRValueUpdate(void);
  Vm0(void);
  ~Vm0(void);
  double t_mod;
  double t_prev;
  double t_mod_fast;
  double t_prev_fast;
  double t_mod_fast_ramp;
  double t_prev_fast_ramp;
  double t_scaled_shifted;
  double t_shift;
  double t_freq_scale;
  double PwmRampDir;
  double PwmRampDirFast;
  bool SignalCirculationUpdate;
  bool SignalCirculationUpdateFast;
  double RailToRailPwmRefFast;
  double RailToRailPwmRefFastRamp;

};


Vm0 Inst_Vm0;


// --------------------------------------------------------

// ---- Vm1 ---------------------------------------

class Vm1 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SrcRValueUpdate(void);
  Vm1(void);
  ~Vm1(void);
  double t_mod;
  double t_prev;
  double t_scaled_shifted;
  double t_shift;
  double t_freq_scale;
  double PwmRampDir;
  bool OmegaCtrl;
  double t_mod_omega;
  double t_prev_omega;
  double t_scaled_shifted_omega;
  double t_shift_omega;
  double PwmRampDir_omega;



};


Vm1 Inst_Vm1;


// --------------------------------------------------------


// ---- Vm2 ----------------------------------------------

class Vm2 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SrcRValueUpdate(void);
  Vm2(void);
  ~Vm2(void);
  double t_mod;
  double t_prev;
  double t_scaled_shifted;
  double t_shift;
  double t_freq_scale;
  double PwmRampDir;
  bool OmegaCtrl;
  double t_mod_omega;
  double t_prev_omega;
  double t_scaled_shifted_omega;
  double t_shift_omega;
  double PwmRampDir_omega;

};


Vm2 Inst_Vm2;


// --------------------------------------------------------


// ---- Vm3 ----------------------------------------------

class Vm3 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SrcRValueUpdate(void);
  Vm3(void);
  ~Vm3(void);
  double t_mod;
  double t_prev;
  double t_scaled_shifted;
  double t_shift;
  double t_freq_scale;
  double PwmRampDir;
  bool OmegaCtrl;
  double t_mod_omega;
  double t_prev_omega;
  double t_scaled_shifted_omega;
  double t_shift_omega;
  double PwmRampDir_omega;

};


Vm3 Inst_Vm3;


// --------------------------------------------------------



// ---- ECtrl1 ---------------------------------------------

class ECtrl1 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl1(void);
  ~ECtrl1(void);
  double Time_0;
  double V_112;
  double V_210;
  bool   GateCtrl_111;
  bool   GateCtrl_121;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_111;
  vector<double> Gate_121;

};

ECtrl1 Inst_ECtrl1;

// --------------------------------------------------------



// ---- ECtrl2 ---------------------------------------------

class ECtrl2 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl2(void);
  ~ECtrl2(void);
  double Time_0;
  double V_112;
  double V_110;
  bool   GateCtrl_211;
  bool   GateCtrl_221;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_211;
  vector<double> Gate_221;	 

};

ECtrl2 Inst_ECtrl2;

// --------------------------------------------------------


// ---- ECtrl3 ---------------------------------------------

class ECtrl3 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl3(void);
  ~ECtrl3(void);
  double Time_0;
  double V_112;
  double V_410;
  bool   GateCtrl_311;
  bool   GateCtrl_321;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_311;
  vector<double> Gate_321;

};

ECtrl3 Inst_ECtrl3;

// --------------------------------------------------------


// ---- ECtrl4 ---------------------------------------------

class ECtrl4 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl4(void);
  ~ECtrl4(void);
  double Time_0;
  double V_112;
  double V_310;
  bool   GateCtrl_411;
  bool   GateCtrl_421;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_411;
  vector<double> Gate_421;
 

};

ECtrl4 Inst_ECtrl4;

// --------------------------------------------------------


// ---- ECtrl5 ---------------------------------------------

class ECtrl5 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl5(void);
  ~ECtrl5(void);
  double Time_0;
  double V_112;
  double V_210;
  bool   GateCtrl_511;
  bool   GateCtrl_521;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_511;
  vector<double> Gate_521;	 

};

ECtrl5 Inst_ECtrl5;

// --------------------------------------------------------


// ---- ECtrl6 ---------------------------------------------

class ECtrl6 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl6(void);
  ~ECtrl6(void);
  double Time_0;
  double V_112;
  double V_110;
  bool   GateCtrl_611;
  bool   GateCtrl_621;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_611;
  vector<double> Gate_621;
	 

};

ECtrl6 Inst_ECtrl6;

// --------------------------------------------------------


// ---- ECtrl7 ---------------------------------------------

class ECtrl7 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl7(void);
  ~ECtrl7(void);
  double Time_0;
  double V_112;
  double V_410;
  bool   GateCtrl_711;
  bool   GateCtrl_721;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_711;
  vector<double> Gate_721;
	 

};

ECtrl7 Inst_ECtrl7;

// --------------------------------------------------------


// ---- ECtrl8 ---------------------------------------------

class ECtrl8 : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SwitchRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  //we use some probes in this object to get a better understanding
  //of what is going on.
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  ECtrl8(void);
  ~ECtrl8(void);
  double Time_0;
  double V_112;
  double V_310;
  bool   GateCtrl_811;
  bool   GateCtrl_821;
  unsigned int CircState;
  //storage for probes...
  vector<double> Gate_811;
  vector<double> Gate_821;
	 

};

ECtrl8 Inst_ECtrl8;

// --------------------------------------------------------

// ---- Vref  ----------------------------------------------------

class Vref : public SrcObject
{
public:
  virtual void SrcFunction(double t);
  virtual void SrcRValueUpdate(void);
  virtual void RecordProbes(void);
  virtual void PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[]);
  Vref(void);
  ~Vref(void);
  unsigned int CircStateIndex;
  unsigned int CircStateRate;
  unsigned int RailToRailStateIndex;
  unsigned int RailToRailStateRate;

  bool OmegaCtrl;
  bool SignalCirculationUpdate;
  double RailToRailPwmRef;
  bool SignalCirculationUpdateFast;
  double RailToRailPwmRefFast;
  double RailToRailPwmRefFastRamp;

  bool   GateCtrl_111;
  bool   GateCtrl_121;
  bool   GateCtrl_211;
  bool   GateCtrl_221;
  bool   GateCtrl_311;
  bool   GateCtrl_321;
  bool   GateCtrl_411;
  bool   GateCtrl_421;
  bool   GateCtrl_511;
  bool   GateCtrl_521;
  bool   GateCtrl_611;
  bool   GateCtrl_621;
  bool   GateCtrl_711;
  bool   GateCtrl_721;
  bool   GateCtrl_811;
  bool   GateCtrl_821;

  double v11_v12;
  double v21_v23;
  double v31_v34;
  double v41_v44;
  double v51_v56;
  double v61_v67;
  double v71_v78;
  double v81_v88;

  double ia;

  //storage for probes...
  vector<double> FastRefRamp;
  vector<double> FastRef;
  vector<double> SlowRef;
  vector<double> V_Cap1;
  vector<double> V_Cap2;
  vector<double> V_Cap3;
  vector<double> V_Cap4;
  vector<double> V_Cap5;
  vector<double> V_Cap6;
  vector<double> V_Cap7;
  vector<double> V_Cap8;
};

Vref Inst_Vref;

// ---------------------------------------------------------------







SrcObject * SrcObjectList[] = {(SrcObject *) &Inst_Vm0,
			       (SrcObject *) &Inst_Vm1,
			       (SrcObject *) &Inst_Vm2,
			       (SrcObject *) &Inst_Vm3,
			       (SrcObject *) &Inst_ECtrl1,
			       (SrcObject *) &Inst_ECtrl2,
			       (SrcObject *) &Inst_ECtrl3,
			       (SrcObject *) &Inst_ECtrl4,
			       (SrcObject *) &Inst_ECtrl5,
			       (SrcObject *) &Inst_ECtrl6,
			       (SrcObject *) &Inst_ECtrl7,
			       (SrcObject *) &Inst_ECtrl8,
			       (SrcObject *) &Inst_Vref,
			       0};


// ****************************************************************




// **** OdeObject Classes *****************************************

 

// ---- qa ---------------------------------------------

class qa : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  qa(void);
  ~qa(void);
  //source for this ODE
  double ia;

};

qa Inst_qa;


// --------------------------------------------------------

// ---- qb ---------------------------------------------

class qb : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  qb(void);
  ~qb(void);
  //source for this ODE
  double ib;

};

qb Inst_qb;


// --------------------------------------------------------


// ---- ia_ib_1 ---------------------------------------------

class ia_ib_1 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  virtual void CoefRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  virtual void OdeGroupMSolve(double dydt[],  double dmdt[]);
  virtual void PostOdeFunction(double t);
  ia_ib_1(void);
  ~ia_ib_1(void);
  //source for this ODE
  double qa;
  //(ia is represented by "y")
  double vs_a;


};

ia_ib_1 Inst_ia_ib_1;

// --------------------------------------------------------

// ---- ia_ib_2 ---------------------------------------------

class ia_ib_2 : public OdeObject
{
public:
  virtual double OdeFunction(double y, double t);
  virtual void OdeRValueUpdate(void);
  ia_ib_2(void);
  ~ia_ib_2(void);
  //source for this ODE
  double qb;
 //(ib is represented by "y")



};

ia_ib_2 Inst_ia_ib_2;

// --------------------------------------------------------


 

OdeObject * OdeObjectList[] = {(OdeObject *) &Inst_qa,
			       (OdeObject *) &Inst_qb,
			       (OdeObject *) &Inst_ia_ib_1,          //the next two are in a group and must be maintained in this order. 
			       (OdeObject *) &Inst_ia_ib_2,
			       0};



// ****************************************************************




// **** CtrlObject Classes ****************************************



CtrlObject * CtrlObjectList[] = {                              
				 0};

int CtrlObjectQuantum[] = {0
                            
                          };


// ****************************************************************





// **** CoefObject Classes ****************************************

// ---- Coef_recp_rSrD11 ---------------------------------------------

class Coef_recp_rSrD11 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD11(void);
  ~Coef_recp_rSrD11(void);
  double rS11;
  double rD11;

};

Coef_recp_rSrD11 Coef_recp_rSrD11_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD12 ---------------------------------------------

class Coef_recp_rSrD12 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD12(void);
  ~Coef_recp_rSrD12(void);
  double rS12;
  double rD12;


};

Coef_recp_rSrD12 Coef_recp_rSrD12_i;

// -------------------------------------------------------------



// ---- Coef_C1_h ---------------------------------------------

class Coef_C1_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C1_h(void);
  ~Coef_C1_h(void);

  


};

Coef_C1_h Coef_C1_h_i;

// -------------------------------------------------------------


// ---- Coef_C11_h ---------------------------------------------

class Coef_C11_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C11_h(void);
  ~Coef_C11_h(void);




};

Coef_C11_h Coef_C11_h_i;

// -------------------------------------------------------------


// ---- Coef_C12_h ---------------------------------------------

class Coef_C12_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C12_h(void);
  ~Coef_C12_h(void);




};

Coef_C12_h Coef_C12_h_i;

// -------------------------------------------------------------


// ---- Coef_v11_v12_h_prev_C1 ---------------------------------

class Coef_v11_v12_h_prev_C1 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v11_v12_h_prev_C1(void);
  ~Coef_v11_v12_h_prev_C1(void);
  double v11_;
  double v12_;
  	  // When "trapezoilal" pass is run...
  double i1_;




};

Coef_v11_v12_h_prev_C1 Coef_v11_v12_h_prev_C1_i;

// -------------------------------------------------------------


// ---- Coef_minus_v11_h_prev_minus_PlusBus_C11 ----------------------

class Coef_minus_v11_h_prev_minus_PlusBus_C11 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_minus_v11_h_prev_minus_PlusBus_C11(void);
  ~Coef_minus_v11_h_prev_minus_PlusBus_C11(void);
  double v11_;
	  // When "trapezoilal" pass is run...
  double i1ct_;




};

Coef_minus_v11_h_prev_minus_PlusBus_C11 Coef_minus_v11_h_prev_minus_PlusBus_C11_i;

// -------------------------------------------------------------

// ---- Coef_v12_h_prev_plus_PlusBus_C12 -----------------------

class Coef_v12_h_prev_plus_PlusBus_C12 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v12_h_prev_plus_PlusBus_C12(void);
  ~Coef_v12_h_prev_plus_PlusBus_C12(void);
  double v12_;
	  // When "trapezoilal" pass is run...
  double i1cb_;

  


};

Coef_v12_h_prev_plus_PlusBus_C12 Coef_v12_h_prev_plus_PlusBus_C12_i;

// -------------------------------------------------------------



// ---- Coef_recp_rSrD21 ---------------------------------------------

class Coef_recp_rSrD21 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD21(void);
  ~Coef_recp_rSrD21(void);
  double rS21;
  double rD21;



};

Coef_recp_rSrD21 Coef_recp_rSrD21_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD22 ---------------------------------------------

class Coef_recp_rSrD22 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD22(void);
  ~Coef_recp_rSrD22(void);
  double rS22;
  double rD22;



};

Coef_recp_rSrD22 Coef_recp_rSrD22_i;

// -------------------------------------------------------------


// ---- Coef_C2_h ---------------------------------------------

class Coef_C2_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C2_h(void);
  ~Coef_C2_h(void);




};

Coef_C2_h Coef_C2_h_i;

// -------------------------------------------------------------


// ---- Coef_C21_h ---------------------------------------------

class Coef_C21_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C21_h(void);
  ~Coef_C21_h(void);




};

Coef_C21_h Coef_C21_h_i;

// -------------------------------------------------------------


// ---- Coef_C22_h ---------------------------------------------

class Coef_C22_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C22_h(void);
  ~Coef_C22_h(void);




};

Coef_C22_h Coef_C22_h_i;

// -------------------------------------------------------------



// ---- Coef_v21_v23_h_prev_C2 ---------------------------------

class Coef_v21_v23_h_prev_C2 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v21_v23_h_prev_C2(void);
  ~Coef_v21_v23_h_prev_C2(void);
  double v21_;
  double v23_;
	  // When "trapezoilal" pass is run...
  double i2_;

};

Coef_v21_v23_h_prev_C2 Coef_v21_v23_h_prev_C2_i;

// -------------------------------------------------------------


// ---- Coef_v12_v21_h_prev_C21 --------------------------------

class Coef_v12_v21_h_prev_C21: public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v12_v21_h_prev_C21(void);
  ~Coef_v12_v21_h_prev_C21(void);
  double v12_;
  double v21_;
	  // When "trapezoilal" pass is run...
  double i2ct_;


};

Coef_v12_v21_h_prev_C21 Coef_v12_v21_h_prev_C21_i;

// -------------------------------------------------------------


// ---- Coef_v23_v12_h_prev_C22 --------------------------------

class Coef_v23_v12_h_prev_C22 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v23_v12_h_prev_C22(void);
  ~Coef_v23_v12_h_prev_C22(void);
  double v23_;
  double v12_;
	  // When "trapezoilal" pass is run...
  double i2cb_;


};

Coef_v23_v12_h_prev_C22 Coef_v23_v12_h_prev_C22_i;

// -------------------------------------------------------------



// ---- Coef_recp_rSrD31 ---------------------------------------------

class Coef_recp_rSrD31 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD31(void);
  ~Coef_recp_rSrD31(void);
  double rS31;
  double rD31;




};

Coef_recp_rSrD31 Coef_recp_rSrD31_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD32 ---------------------------------------------

class Coef_recp_rSrD32 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD32(void);
  ~Coef_recp_rSrD32(void);
  double rS32;
  double rD32;



};

Coef_recp_rSrD32 Coef_recp_rSrD32_i;

// -------------------------------------------------------------


// ---- Coef_C3_h ---------------------------------------------

class Coef_C3_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C3_h(void);
  ~Coef_C3_h(void);




};

Coef_C3_h Coef_C3_h_i;

// -------------------------------------------------------------


// ---- Coef_C31_h ---------------------------------------------

class Coef_C31_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C31_h(void);
  ~Coef_C31_h(void);




};

Coef_C31_h Coef_C31_h_i;

// -------------------------------------------------------------


// ---- Coef_C32_h ---------------------------------------------

class Coef_C32_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C32_h(void);
  ~Coef_C32_h(void);




};

Coef_C32_h Coef_C32_h_i;

// -------------------------------------------------------------



// ---- Coef_v31_v34_h_prev_C3 ---------------------------------

class Coef_v31_v34_h_prev_C3 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v31_v34_h_prev_C3(void);
  ~Coef_v31_v34_h_prev_C3(void);
  double v31_;
  double v34_;
	  // When "trapezoilal" pass is run...
  double i3_;



};

Coef_v31_v34_h_prev_C3 Coef_v31_v34_h_prev_C3_i;

// -------------------------------------------------------------


// ---- Coef_v23_v31_h_prev_C31 --------------------------------

class Coef_v23_v31_h_prev_C31: public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v23_v31_h_prev_C31(void);
  ~Coef_v23_v31_h_prev_C31(void);
  double v23_;
  double v31_;
	  // When "trapezoilal" pass is run...
  double i3ct_;



};

Coef_v23_v31_h_prev_C31 Coef_v23_v31_h_prev_C31_i;

// -------------------------------------------------------------


// ---- Coef_v34_v23_h_prev_C32 --------------------------------

class Coef_v34_v23_h_prev_C32 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v34_v23_h_prev_C32(void);
  ~Coef_v34_v23_h_prev_C32(void);
  double v34_;
  double v23_;
	  // When "trapezoilal" pass is run...
  double i3cb_;


};

Coef_v34_v23_h_prev_C32 Coef_v34_v23_h_prev_C32_i;

// -------------------------------------------------------------



// ---- Coef_recp_rSrD41 ---------------------------------------------

class Coef_recp_rSrD41 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD41(void);
  ~Coef_recp_rSrD41(void);
  double rS41;
  double rD41;



};

Coef_recp_rSrD41 Coef_recp_rSrD41_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD42 ---------------------------------------------

class Coef_recp_rSrD42 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD42(void);
  ~Coef_recp_rSrD42(void);
  double rS42;
  double rD42;



};

Coef_recp_rSrD42 Coef_recp_rSrD42_i;

// -------------------------------------------------------------



// ---- Coef_C4_h ---------------------------------------------

class Coef_C4_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C4_h(void);
  ~Coef_C4_h(void);




};

Coef_C4_h Coef_C4_h_i;

// -------------------------------------------------------------


// ---- Coef_C41_h ---------------------------------------------

class Coef_C41_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C41_h(void);
  ~Coef_C41_h(void);




};

Coef_C41_h Coef_C41_h_i;

// -------------------------------------------------------------


// ---- Coef_C42_h ---------------------------------------------

class Coef_C42_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C42_h(void);
  ~Coef_C42_h(void);




};

Coef_C42_h Coef_C42_h_i;

// -------------------------------------------------------------



// ---- Coef_v41_v44_h_prev_C4 ---------------------------------

class Coef_v41_v44_h_prev_C4 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v41_v44_h_prev_C4(void);
  ~Coef_v41_v44_h_prev_C4(void);
  double v41_;
  double v44_;
	  // When "trapezoilal" pass is run...
  double i4_;


};

Coef_v41_v44_h_prev_C4 Coef_v41_v44_h_prev_C4_i;

// -------------------------------------------------------------


// ---- Coef_v34_v41_h_prev_C41 --------------------------------

class Coef_v34_v41_h_prev_C41: public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v34_v41_h_prev_C41(void);
  ~Coef_v34_v41_h_prev_C41(void);
  double v34_;
  double v41_;
	  // When "trapezoilal" pass is run...
  double i4ct_;



};

Coef_v34_v41_h_prev_C41 Coef_v34_v41_h_prev_C41_i;

// -------------------------------------------------------------


// ---- Coef_v44_v34_h_prev_C42 --------------------------------

class Coef_v44_v34_h_prev_C42 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v44_v34_h_prev_C42(void);
  ~Coef_v44_v34_h_prev_C42(void);
  double v44_;
  double v34_;
	  // When "trapezoilal" pass is run...
  double i4cb_;


};

Coef_v44_v34_h_prev_C42 Coef_v44_v34_h_prev_C42_i;

// -------------------------------------------------------------


// ---- Coef_Lp_h ---------------------------------------------

class Coef_Lp_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_Lp_h(void);
  ~Coef_Lp_h(void);




};

Coef_Lp_h Coef_Lp_h_i;

// -------------------------------------------------------------


// ---- Coef_Mpn_h ---------------------------------------------

class Coef_Mpn_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_Mpn_h(void);
  ~Coef_Mpn_h(void);




};

Coef_Mpn_h Coef_Mpn_h_i;

// -------------------------------------------------------------

// ---- Coef_Mnp_h ---------------------------------------------

class Coef_Mnp_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_Mnp_h(void);
  ~Coef_Mnp_h(void);




};

Coef_Mnp_h Coef_Mnp_h_i;

// ------------------------------------------------------------

// ---- Coef_Ln_h ---------------------------------------------

class Coef_Ln_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_Ln_h(void);
  ~Coef_Ln_h(void);

  


};

Coef_Ln_h Coef_Ln_h_i;

// -------------------------------------------------------------


// ---- Coef_iLp_iLn_h_prev_Lp_Mpn -------------------------------------

class Coef_iLp_iLn_h_prev_Lp_Mpn : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_iLp_iLn_h_prev_Lp_Mpn(void);
  ~Coef_iLp_iLn_h_prev_Lp_Mpn(void);
  double iLp_;
  double iLn_;
  	  // When "trapezoilal" pass is run...
  double v44_;
  double vs_a_;



};

Coef_iLp_iLn_h_prev_Lp_Mpn Coef_iLp_iLn_h_prev_Lp_Mpn_i;

// -------------------------------------------------------------



// ---- Coef_iLp_iLn_h_prev_Mnp_Ln ------------------------------------

class Coef_iLp_iLn_h_prev_Mnp_Ln : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_iLp_iLn_h_prev_Mnp_Ln(void);
  ~Coef_iLp_iLn_h_prev_Mnp_Ln(void);
  double iLp_;
  double iLn_;
  	  // When "trapezoilal" pass is run...
  double vs_a_;
  double v45_;



};

Coef_iLp_iLn_h_prev_Mnp_Ln Coef_iLp_iLn_h_prev_Mnp_Ln_i;

// -------------------------------------------------------------

 

// ---- Coef_ia ---------------------------------------------

class Coef_ia : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_ia(void);
  ~Coef_ia(void);
  double ia;
  


};

Coef_ia Coef_ia_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD51 ---------------------------------------------

class Coef_recp_rSrD51 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD51(void);
  ~Coef_recp_rSrD51(void);
  double rS51;
  double rD51;



};

Coef_recp_rSrD51 Coef_recp_rSrD51_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD52 ---------------------------------------------

class Coef_recp_rSrD52 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD52(void);
  ~Coef_recp_rSrD52(void);
  double rS52;
  double rD52;



};

Coef_recp_rSrD52 Coef_recp_rSrD52_i;

// -------------------------------------------------------------




// ---- Coef_C5_h ---------------------------------------------

class Coef_C5_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C5_h(void);
  ~Coef_C5_h(void);




};

Coef_C5_h Coef_C5_h_i;

// -------------------------------------------------------------


// ---- Coef_C51_h ---------------------------------------------

class Coef_C51_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C51_h(void);
  ~Coef_C51_h(void);




};

Coef_C51_h Coef_C51_h_i;

// -------------------------------------------------------------


// ---- Coef_C52_h ---------------------------------------------

class Coef_C52_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C52_h(void);
  ~Coef_C52_h(void);




};

Coef_C52_h Coef_C52_h_i;

// -------------------------------------------------------------



// ---- Coef_v51_v56_h_prev_C5 ---------------------------------

class Coef_v51_v56_h_prev_C5 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v51_v56_h_prev_C5(void);
  ~Coef_v51_v56_h_prev_C5(void);
  double v51_;
  double v56_;
  	  // When "trapezoilal" pass is run...
  double i5_;


};

Coef_v51_v56_h_prev_C5 Coef_v51_v56_h_prev_C5_i;

// -------------------------------------------------------------


// ---- Coef_v45_v51_h_prev_C51 --------------------------------

class Coef_v45_v51_h_prev_C51: public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v45_v51_h_prev_C51(void);
  ~Coef_v45_v51_h_prev_C51(void);
  double v45_;
  double v51_;
  	  // When "trapezoilal" pass is run...
  double i5ct_;


};

Coef_v45_v51_h_prev_C51 Coef_v45_v51_h_prev_C51_i;

// -------------------------------------------------------------


// ---- Coef_v56_v45_h_prev_C52 --------------------------------

class Coef_v56_v45_h_prev_C52 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v56_v45_h_prev_C52(void);
  ~Coef_v56_v45_h_prev_C52(void);
  double v56_;
  double v45_;
  	  // When "trapezoilal" pass is run...
  double i5cb_;


};

Coef_v56_v45_h_prev_C52 Coef_v56_v45_h_prev_C52_i;

// -------------------------------------------------------------


// ---- Coef_recp_rSrD61 ---------------------------------------------

class Coef_recp_rSrD61 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD61(void);
  ~Coef_recp_rSrD61(void);
  double rS61;
  double rD61;



};

Coef_recp_rSrD61 Coef_recp_rSrD61_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD62 ---------------------------------------------

class Coef_recp_rSrD62 : public CoefObject
{
public:

  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD62(void);
  ~Coef_recp_rSrD62(void);
  double rS62;
  double rD62;


};

Coef_recp_rSrD62 Coef_recp_rSrD62_i;

// -------------------------------------------------------------




// ---- Coef_C6_h ---------------------------------------------

class Coef_C6_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C6_h(void);
  ~Coef_C6_h(void);




};

Coef_C6_h Coef_C6_h_i;

// -------------------------------------------------------------


// ---- Coef_C61_h ---------------------------------------------

class Coef_C61_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C61_h(void);
  ~Coef_C61_h(void);




};

Coef_C61_h Coef_C61_h_i;

// -------------------------------------------------------------


// ---- Coef_C62_h ---------------------------------------------

class Coef_C62_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C62_h(void);
  ~Coef_C62_h(void);




};

Coef_C62_h Coef_C62_h_i;

// -------------------------------------------------------------



// ---- Coef_v61_v67_h_prev_C6 ---------------------------------

class Coef_v61_v67_h_prev_C6 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v61_v67_h_prev_C6(void);
  ~Coef_v61_v67_h_prev_C6(void);
  double v61_;
  double v67_;
  	  // When "trapezoilal" pass is run...
  double i6_;


};

Coef_v61_v67_h_prev_C6 Coef_v61_v67_h_prev_C6_i;

// -------------------------------------------------------------


// ---- Coef_v56_v61_h_prev_C61 --------------------------------

class Coef_v56_v61_h_prev_C61: public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v56_v61_h_prev_C61(void);
  ~Coef_v56_v61_h_prev_C61(void);
  double v56_;
  double v61_;
  	  // When "trapezoilal" pass is run...
  double i6ct_;


};

Coef_v56_v61_h_prev_C61 Coef_v56_v61_h_prev_C61_i;

// -------------------------------------------------------------


// ---- Coef_v67_v56_h_prev_C62 --------------------------------

class Coef_v67_v56_h_prev_C62 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v67_v56_h_prev_C62(void);
  ~Coef_v67_v56_h_prev_C62(void);
  double v67_;
  double v56_;
  	  // When "trapezoilal" pass is run...
  double i6cb_;


};

Coef_v67_v56_h_prev_C62 Coef_v67_v56_h_prev_C62_i;

// -------------------------------------------------------------



// ---- Coef_recp_rSrD71 ---------------------------------------------

class Coef_recp_rSrD71 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD71(void);
  ~Coef_recp_rSrD71(void);
  double rS71;
  double rD71;



};

Coef_recp_rSrD71 Coef_recp_rSrD71_i;

// -------------------------------------------------------------

// ---- Coef_recp_rSrD72 ---------------------------------------------

class Coef_recp_rSrD72 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD72(void);
  ~Coef_recp_rSrD72(void);
  double rS72;
  double rD72;



};

Coef_recp_rSrD72 Coef_recp_rSrD72_i;

// -------------------------------------------------------------


// ---- Coef_C7_h ---------------------------------------------

class Coef_C7_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C7_h(void);
  ~Coef_C7_h(void);




};

Coef_C7_h Coef_C7_h_i;

// -------------------------------------------------------------


// ---- Coef_C71_h ---------------------------------------------

class Coef_C71_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C71_h(void);
  ~Coef_C71_h(void);




};

Coef_C71_h Coef_C71_h_i;

// -------------------------------------------------------------


// ---- Coef_C72_h ---------------------------------------------

class Coef_C72_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C72_h(void);
  ~Coef_C72_h(void);




};

Coef_C72_h Coef_C72_h_i;

// -------------------------------------------------------------



// ---- Coef_v71_v78_h_prev_C7 ---------------------------------

class Coef_v71_v78_h_prev_C7 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v71_v78_h_prev_C7(void);
  ~Coef_v71_v78_h_prev_C7(void);
  double v71_;
  double v78_;
  	  // When "trapezoilal" pass is run...
  double i7_;


};

Coef_v71_v78_h_prev_C7 Coef_v71_v78_h_prev_C7_i;

// -------------------------------------------------------------


// ---- Coef_v67_v71_h_prev_C71 --------------------------------

class Coef_v67_v71_h_prev_C71: public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v67_v71_h_prev_C71(void);
  ~Coef_v67_v71_h_prev_C71(void);
  double v67_;
  double v71_;
  	  // When "trapezoilal" pass is run...
  double i7ct_;



};

Coef_v67_v71_h_prev_C71 Coef_v67_v71_h_prev_C71_i;

// -------------------------------------------------------------


// ---- Coef_v78_v67_h_prev_C72 --------------------------------

class Coef_v78_v67_h_prev_C72 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v78_v67_h_prev_C72(void);
  ~Coef_v78_v67_h_prev_C72(void);
  double v78_;
  double v67_;
  	  // When "trapezoilal" pass is run...
  double i7cb_;


};

Coef_v78_v67_h_prev_C72 Coef_v78_v67_h_prev_C72_i;

// -------------------------------------------------------------



// ---- Coef_recp_rSrD81 ---------------------------------------------

class Coef_recp_rSrD81 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD81(void);
  ~Coef_recp_rSrD81(void);
  double rS81;
  double rD81;



};

Coef_recp_rSrD81 Coef_recp_rSrD81_i;

// -------------------------------------------------------------



// ---- Coef_recp_rSrD82 ---------------------------------------------

class Coef_recp_rSrD82 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_recp_rSrD82(void);
  ~Coef_recp_rSrD82(void);
  double rS82;
  double rD82;



};

Coef_recp_rSrD82 Coef_recp_rSrD82_i;

// -------------------------------------------------------------



// ---- Coef_C8_h ---------------------------------------------

class Coef_C8_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C8_h(void);
  ~Coef_C8_h(void);




};

Coef_C8_h Coef_C8_h_i;

// -------------------------------------------------------------


// ---- Coef_C81_h ---------------------------------------------

class Coef_C81_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C81_h(void);
  ~Coef_C81_h(void);




};

Coef_C81_h Coef_C81_h_i;

// -------------------------------------------------------------


// ---- Coef_C82_h ---------------------------------------------

class Coef_C82_h : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_C82_h(void);
  ~Coef_C82_h(void);




};

Coef_C82_h Coef_C82_h_i;

// -------------------------------------------------------------



// ---- Coef_v81_h_prev_plus_MinusBus_C8 -----------------------

class Coef_v81_h_prev_plus_MinusBus_C8 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v81_h_prev_plus_MinusBus_C8(void);
  ~Coef_v81_h_prev_plus_MinusBus_C8(void);
  double v81_;
  	  // When "trapezoilal" pass is run...
  double i8_;


};

Coef_v81_h_prev_plus_MinusBus_C8 Coef_v81_h_prev_plus_MinusBus_C8_i;

// -------------------------------------------------------------


// ---- Coef_v78_v81_h_prev_C81 --------------------------------

class Coef_v78_v81_h_prev_C81: public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_v78_v81_h_prev_C81(void);
  ~Coef_v78_v81_h_prev_C81(void);
  double v78_;
  double v81_;
  	  // When "trapezoilal" pass is run...
  double i8ct_;


};

Coef_v78_v81_h_prev_C81 Coef_v78_v81_h_prev_C81_i;

// -------------------------------------------------------------


// ---- Coef_minus_v78_h_prev_minus_MinusBus_C82 ---------------------

class Coef_minus_v78_h_prev_minus_MinusBus_C82 : public CoefObject
{
public:
  virtual void CoefFunction(double recp_h, bool DoTrapezoidal);
  Coef_minus_v78_h_prev_minus_MinusBus_C82(void);
  ~Coef_minus_v78_h_prev_minus_MinusBus_C82(void);
  double v78_;
  	  // When "trapezoilal" pass is run...
  double i8cb_;

};

Coef_minus_v78_h_prev_minus_MinusBus_C82 Coef_minus_v78_h_prev_minus_MinusBus_C82_i;

// -------------------------------------------------------------










CoefObject * CoefObjectList[] = {(CoefObject *)  &Coef_recp_rSrD11_i,
				 (CoefObject *)  &Coef_recp_rSrD12_i,
				 (CoefObject *)  &Coef_C1_h_i,
				 (CoefObject *)  &Coef_C11_h_i,
				 (CoefObject *)  &Coef_C12_h_i,
				 (CoefObject *)  &Coef_v11_v12_h_prev_C1_i,
				 (CoefObject *)  &Coef_minus_v11_h_prev_minus_PlusBus_C11_i,
				 (CoefObject *)  &Coef_v12_h_prev_plus_PlusBus_C12_i,
				 (CoefObject *)  &Coef_recp_rSrD21_i,
				 (CoefObject *)  &Coef_recp_rSrD22_i,
				 (CoefObject *)  &Coef_C2_h_i,
				 (CoefObject *)  &Coef_C21_h_i,
				 (CoefObject *)  &Coef_C22_h_i,
				 (CoefObject *)  &Coef_v21_v23_h_prev_C2_i,
				 (CoefObject *)  &Coef_v12_v21_h_prev_C21_i,
				 (CoefObject *)  &Coef_v23_v12_h_prev_C22_i,
				 (CoefObject *)  &Coef_recp_rSrD31_i,
				 (CoefObject *)  &Coef_recp_rSrD32_i,
				 (CoefObject *)  &Coef_C3_h_i,
				 (CoefObject *)  &Coef_C31_h_i,
				 (CoefObject *)  &Coef_C32_h_i,
				 (CoefObject *)  &Coef_v31_v34_h_prev_C3_i,
				 (CoefObject *)  &Coef_v23_v31_h_prev_C31_i,
				 (CoefObject *)  &Coef_v34_v23_h_prev_C32_i,
				 (CoefObject *)  &Coef_recp_rSrD41_i,
				 (CoefObject *)  &Coef_recp_rSrD42_i,
				 (CoefObject *)  &Coef_C4_h_i,
				 (CoefObject *)  &Coef_C41_h_i,
				 (CoefObject *)  &Coef_C42_h_i,
				 (CoefObject *)  &Coef_v41_v44_h_prev_C4_i,
				 (CoefObject *)  &Coef_v34_v41_h_prev_C41_i,
				 (CoefObject *)  &Coef_v44_v34_h_prev_C42_i,
				 (CoefObject *)  &Coef_Lp_h_i,
				 (CoefObject *)  &Coef_Mpn_h_i,
				 (CoefObject *)  &Coef_Mnp_h_i,
				 (CoefObject *)  &Coef_Ln_h_i,
				 (CoefObject *)  &Coef_iLp_iLn_h_prev_Lp_Mpn_i,
				 (CoefObject *)  &Coef_iLp_iLn_h_prev_Mnp_Ln_i,
				 (CoefObject *)  &Coef_ia_i,
				 (CoefObject *)  &Coef_recp_rSrD51_i,
				 (CoefObject *)  &Coef_recp_rSrD52_i,
				 (CoefObject *)  &Coef_C5_h_i,
				 (CoefObject *)  &Coef_C51_h_i,
				 (CoefObject *)  &Coef_C52_h_i,
				 (CoefObject *)  &Coef_v51_v56_h_prev_C5_i,
				 (CoefObject *)  &Coef_v45_v51_h_prev_C51_i,
				 (CoefObject *)  &Coef_v56_v45_h_prev_C52_i,
				 (CoefObject *)  &Coef_recp_rSrD61_i,
				 (CoefObject *)  &Coef_recp_rSrD62_i,
				 (CoefObject *)  &Coef_C6_h_i,
				 (CoefObject *)  &Coef_C61_h_i,
				 (CoefObject *)  &Coef_C62_h_i,
				 (CoefObject *)  &Coef_v61_v67_h_prev_C6_i,
				 (CoefObject *)  &Coef_v56_v61_h_prev_C61_i,
				 (CoefObject *)  &Coef_v67_v56_h_prev_C62_i,
				 (CoefObject *)  &Coef_recp_rSrD71_i,
				 (CoefObject *)  &Coef_recp_rSrD72_i,
				 (CoefObject *)  &Coef_C7_h_i,
				 (CoefObject *)  &Coef_C71_h_i,
				 (CoefObject *)  &Coef_C72_h_i,
				 (CoefObject *)  &Coef_v71_v78_h_prev_C7_i,
				 (CoefObject *)  &Coef_v67_v71_h_prev_C71_i,
				 (CoefObject *)  &Coef_v78_v67_h_prev_C72_i,
				 (CoefObject *)  &Coef_recp_rSrD81_i,
				 (CoefObject *)  &Coef_recp_rSrD82_i,
				 (CoefObject *)  &Coef_C8_h_i,
				 (CoefObject *)  &Coef_C81_h_i,
				 (CoefObject *)  &Coef_C82_h_i,
				 (CoefObject *)  &Coef_v81_h_prev_plus_MinusBus_C8_i,
				 (CoefObject *)  &Coef_v78_v81_h_prev_C81_i,
				 (CoefObject *)  &Coef_minus_v78_h_prev_minus_MinusBus_C82_i,
				 0};



// ****************************************************************

// **** SwitchObject Classes **************************************

// ---- Switch_rS11 -----------------------------------------------

class Switch_rS11 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS11(void);
  ~Switch_rS11(void);

};

Switch_rS11 Switch_rS11_i;

// ----------------------------------------------------------------


// ---- Switch_rS12 -----------------------------------------------

class Switch_rS12 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS12(void);
  ~Switch_rS12(void);
};

Switch_rS12 Switch_rS12_i;

// ----------------------------------------------------------------


// ---- Switch_rS21 -----------------------------------------------

class Switch_rS21 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS21(void);
  ~Switch_rS21(void);

};

Switch_rS21 Switch_rS21_i;

// ----------------------------------------------------------------


// ---- Switch_rS22 -----------------------------------------------

class Switch_rS22 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS22(void);
  ~Switch_rS22(void);
};

Switch_rS22 Switch_rS22_i;

// ----------------------------------------------------------------


// ---- Switch_rS31 -----------------------------------------------

class Switch_rS31 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS31(void);
  ~Switch_rS31(void);
};

Switch_rS31 Switch_rS31_i;

// ----------------------------------------------------------------


// ---- Switch_rS32 -----------------------------------------------

class Switch_rS32 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS32(void);
  ~Switch_rS32(void);
};

Switch_rS32 Switch_rS32_i;

// ----------------------------------------------------------------


// ---- Switch_rS41 -----------------------------------------------

class Switch_rS41 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS41(void);
  ~Switch_rS41(void);
};

Switch_rS41 Switch_rS41_i;

// ----------------------------------------------------------------


// ---- Switch_rS42 -----------------------------------------------

class Switch_rS42 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS42(void);
  ~Switch_rS42(void);

};

Switch_rS42 Switch_rS42_i;

// ----------------------------------------------------------------


// ---- Switch_rS51 -----------------------------------------------

class Switch_rS51 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS51(void);
  ~Switch_rS51(void);

};

Switch_rS51 Switch_rS51_i;

// ----------------------------------------------------------------


// ---- Switch_rS52 -----------------------------------------------

class Switch_rS52 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS52(void);
  ~Switch_rS52(void);
};

Switch_rS52 Switch_rS52_i;

// ----------------------------------------------------------------


// ---- Switch_rS61 -----------------------------------------------

class Switch_rS61 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS61(void);
  ~Switch_rS61(void);
};

Switch_rS61 Switch_rS61_i;

// ----------------------------------------------------------------


// ---- Switch_rS62 -----------------------------------------------

class Switch_rS62 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS62(void);
  ~Switch_rS62(void);
};

Switch_rS62 Switch_rS62_i;

// ----------------------------------------------------------------


// ---- Switch_rS71 -----------------------------------------------

class Switch_rS71 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS71(void);
  ~Switch_rS71(void);

};

Switch_rS71 Switch_rS71_i;

// ----------------------------------------------------------------


// ---- Switch_rS72 -----------------------------------------------

class Switch_rS72 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS72(void);
  ~Switch_rS72(void);
};

Switch_rS72 Switch_rS72_i;

// ----------------------------------------------------------------

// ---- Switch_rS81 -----------------------------------------------

class Switch_rS81 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS81(void);
  ~Switch_rS81(void);

};

Switch_rS81 Switch_rS81_i;

// ----------------------------------------------------------------


// ---- Switch_rS82 -----------------------------------------------

class Switch_rS82 : public SwitchObject
{
public:
  virtual void CoefRValueUpdate(void);
  Switch_rS82(void);
  ~Switch_rS82(void);

};

Switch_rS82 Switch_rS82_i;

// ----------------------------------------------------------------



SwitchObject * SwitchObjectList[] = {(SwitchObject *) &Switch_rS11_i,
				     (SwitchObject *) &Switch_rS12_i,
				     (SwitchObject *) &Switch_rS21_i,
				     (SwitchObject *) &Switch_rS22_i,
				     (SwitchObject *) &Switch_rS31_i,
				     (SwitchObject *) &Switch_rS32_i,
				     (SwitchObject *) &Switch_rS41_i,
				     (SwitchObject *) &Switch_rS42_i,
				     (SwitchObject *) &Switch_rS51_i,
				     (SwitchObject *) &Switch_rS52_i,
				     (SwitchObject *) &Switch_rS61_i,
				     (SwitchObject *) &Switch_rS62_i,
				     (SwitchObject *) &Switch_rS71_i,
				     (SwitchObject *) &Switch_rS72_i,
				     (SwitchObject *) &Switch_rS81_i,
				     (SwitchObject *) &Switch_rS82_i,
                                     0};



// ****************************************************************



// **** SpiceObject Classes ***************************************

// ---- Spice_MultiLevel ------------------------------------------

class Spice_MultiLevel : public SpiceObject
{
public:

  Spice_MultiLevel(void);
  ~Spice_MultiLevel(void);
  virtual void OdeRValueUpdate(void);
  virtual void SwitchRValueUpdate(void);
  virtual void CoefRValueUpdate(void);
  virtual void SrcRValueUpdate(void);
  FILE * fp_vs_a_output;

};


Spice_MultiLevel Spice_MultiLevel_i;


// ---------------------------------------------------------------

SpiceObject * SpiceObjectList[] = {(SpiceObject *) &Spice_MultiLevel_i,
				   0};
				    



// **************************************************************



// **** SrcObject Functions **************************************

// ---- Vm0  ----------------------------------------------------


Vm0::Vm0(void)
{
 
  SrcObjItem * pCurSrcItem;


  SrcFuncName = SRC_FUNC_Vm0;
  LiteralName = "Vm0";

 //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;


  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl2];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl6];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];


  SignalCirculationUpdate = FALSE;
  SignalCirculationUpdateFast = FALSE;

  RailToRailPwmRefFast = 0;
  RailToRailPwmRefFastRamp = 0;

  t_freq_scale = 1.0;
  t_shift = 0;
 
  PwmRampDir = 1.0;
  PwmRampDirFast = 1.0;


#ifdef PLOT_SRCOBJECT_Vm0
  PlotThisOutput = TRUE;
#endif

  Plot_Tag = "Vm0";
}


Vm0::~Vm0(void)
{



}

void Vm0::SrcFunction(double t)
{
	SignalCirculationUpdate = FALSE;
	SignalCirculationUpdateFast = FALSE;

  t_scaled_shifted = t * t_freq_scale + t_shift;

  if (t_scaled_shifted >= t_prev){

    t_mod += (t_scaled_shifted - t_prev);
    t_prev = t_scaled_shifted;  

    if(t_mod > HALF_PWM_CYCLE){
      t_mod -=  HALF_PWM_CYCLE;
      PwmRampDir *= -1.0;

      if(PwmRampDir > 0)
    	  SignalCirculationUpdate = TRUE;


    }

 
    y = PWM_GAIN*(t_mod/HALF_PWM_CYCLE - .5)*PwmRampDir;


    t_mod_fast += (t_scaled_shifted - t_prev_fast);
    t_prev_fast = t_scaled_shifted;

    if(t_mod_fast > (HALF_PWM_CYCLE/4.0))
    {
    	t_mod_fast -= (HALF_PWM_CYCLE/4.0);
    	PwmRampDirFast *= -1.0;

    	if(PwmRampDirFast > 0)
    		SignalCirculationUpdateFast = TRUE;

    }

    RailToRailPwmRefFast = PWM_GAIN*(t_mod_fast/(HALF_PWM_CYCLE/4.0) - .5)*PwmRampDirFast;



	t_mod_fast_ramp += (t_scaled_shifted - t_prev_fast_ramp);
	t_prev_fast_ramp = t_scaled_shifted;

	if(t_mod_fast_ramp > (HALF_PWM_CYCLE/2.0))
	{
		t_mod_fast_ramp -= (HALF_PWM_CYCLE/2.0);


	}

	RailToRailPwmRefFastRamp = PWM_GAIN*(t_mod_fast_ramp/(HALF_PWM_CYCLE/2.0) - .5);




  }

}

void Vm0::SrcRValueUpdate(void)         
{
  SrcObjItem * pCurSrcItem;
 

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;


  ((ECtrl2 *) pCurSrcItem->pSrcObject)->V_110 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl6 *) pCurSrcItem->pSrcObject)->V_110 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((Vref *) pCurSrcItem->pSrcObject)->SignalCirculationUpdate = SignalCirculationUpdate;
  ((Vref *) pCurSrcItem->pSrcObject)->RailToRailPwmRef = y;
  ((Vref *) pCurSrcItem->pSrcObject)->SignalCirculationUpdateFast = SignalCirculationUpdateFast;
  ((Vref *) pCurSrcItem->pSrcObject)->RailToRailPwmRefFast = RailToRailPwmRefFast;
  ((Vref *) pCurSrcItem->pSrcObject)->RailToRailPwmRefFastRamp = RailToRailPwmRefFastRamp;

}

// -----------------------------------------------------------------

// ---- Vm1  ----------------------------------------------------

Vm1::Vm1(void)
{
 
  SrcObjItem * pCurSrcItem;


  SrcFuncName = SRC_FUNC_Vm1;
  LiteralName = "Vm1";

 //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl1];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl5];

  t_freq_scale = 1.0;
  t_shift = HALF_PWM_CYCLE/2.0 ;

 
  PwmRampDir = 1.0;
  PwmRampDir_omega = 1.0;

  OmegaCtrl = FALSE;
  t_shift_omega = (HALF_PWM_CYCLE/2.0) * OMEGA_CTRL_FACTOR;

#ifdef PLOT_SRCOBJECT_Vm1 
  PlotThisOutput = TRUE;
#endif 

  Plot_Tag = "Vm1";
}


Vm1::~Vm1(void)
{


}

void Vm1::SrcFunction(double t)
{
  t_scaled_shifted = t * t_freq_scale + t_shift;

  if (t_scaled_shifted >= t_prev){

    t_mod += (t_scaled_shifted - t_prev);
    t_prev = t_scaled_shifted;  

    if(t_mod > HALF_PWM_CYCLE){
      t_mod -=  HALF_PWM_CYCLE;
      PwmRampDir *= -1.0;
    }
 
  }

#ifdef ENABLE_OMEGA_CTRL

    t_scaled_shifted_omega = t * t_freq_scale + t_shift_omega;

    if (t_scaled_shifted_omega >= t_prev_omega){

      t_mod_omega += (t_scaled_shifted_omega - t_prev_omega);
      t_prev_omega = t_scaled_shifted_omega;

      if(t_mod_omega > HALF_PWM_CYCLE){
        t_mod_omega -=  HALF_PWM_CYCLE;
        PwmRampDir_omega *= -1.0;
      }

    }
#endif

    if(OmegaCtrl)
    	y = PWM_GAIN*(t_mod_omega/HALF_PWM_CYCLE - .5)*PwmRampDir_omega;
    else
    	y = PWM_GAIN*(t_mod/HALF_PWM_CYCLE - .5)*PwmRampDir;



}

void Vm1::SrcRValueUpdate(void)         
{
  SrcObjItem * pCurSrcItem;
 

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((ECtrl1 *) pCurSrcItem->pSrcObject)->V_210 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl5 *) pCurSrcItem->pSrcObject)->V_210 = y;



}


// -----------------------------------------------------------------

// ---- Vm2  ----------------------------------------------------

Vm2::Vm2(void)
{
 
  SrcObjItem * pCurSrcItem;


  SrcFuncName = SRC_FUNC_Vm2;
  LiteralName = "Vm2";

 //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl4];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl8];

  t_freq_scale = 1.0;
  t_shift = HALF_PWM_CYCLE;

 
  PwmRampDir = 1.0;
  PwmRampDir_omega = 1.0;

  OmegaCtrl = FALSE;
  t_shift_omega = HALF_PWM_CYCLE * OMEGA_CTRL_FACTOR;

#ifdef PLOT_SRCOBJECT_Vm2
  PlotThisOutput = TRUE;
#endif

  Plot_Tag = "Vm2";
}


Vm2::~Vm2(void)
{



}

void Vm2::SrcFunction(double t)
{
  t_scaled_shifted = t * t_freq_scale + t_shift;

  if (t_scaled_shifted >= t_prev){

    t_mod += (t_scaled_shifted - t_prev);
    t_prev = t_scaled_shifted;  

    if(t_mod > HALF_PWM_CYCLE){
      t_mod -=  HALF_PWM_CYCLE;
      PwmRampDir *= -1.0;
    }
 
  }
#ifdef ENABLE_OMEGA_CTRL

    t_scaled_shifted_omega = t * t_freq_scale + t_shift_omega;

    if (t_scaled_shifted_omega >= t_prev_omega){

      t_mod_omega += (t_scaled_shifted_omega - t_prev_omega);
      t_prev_omega = t_scaled_shifted_omega;

      if(t_mod_omega > HALF_PWM_CYCLE){
        t_mod_omega -=  HALF_PWM_CYCLE;
        PwmRampDir_omega *= -1.0;
      }

    }
#endif

    if(OmegaCtrl)
    	y = PWM_GAIN*(t_mod_omega/HALF_PWM_CYCLE - .5)*PwmRampDir_omega;
    else
    	y = PWM_GAIN*(t_mod/HALF_PWM_CYCLE - .5)*PwmRampDir;



}

void Vm2::SrcRValueUpdate(void)         
{
  SrcObjItem * pCurSrcItem;
 

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((ECtrl4 *) pCurSrcItem->pSrcObject)->V_310 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl8 *) pCurSrcItem->pSrcObject)->V_310 = y;



}


// -----------------------------------------------------------------

// ---- Vm3  ----------------------------------------------------

Vm3::Vm3(void)
{
 
  SrcObjItem * pCurSrcItem;


  SrcFuncName = SRC_FUNC_Vm3;
  LiteralName = "Vm3";

 //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl3];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl7];

  t_freq_scale = 1.0;
  t_shift = 3.0*HALF_PWM_CYCLE/2.0;

 
  PwmRampDir = 1.0;
  PwmRampDir_omega = 1.0;

  OmegaCtrl = FALSE;
  t_shift_omega = (3.0*HALF_PWM_CYCLE/2.0) * OMEGA_CTRL_FACTOR;

#ifdef PLOT_SRCOBJECT_Vm3
  PlotThisOutput = TRUE;
#endif

  Plot_Tag = "Vm3";
}


Vm3::~Vm3(void)
{



}

void Vm3::SrcFunction(double t)
{
  t_scaled_shifted = t * t_freq_scale + t_shift;

  if (t_scaled_shifted >= t_prev){

    t_mod += (t_scaled_shifted - t_prev);
    t_prev = t_scaled_shifted;  

    if(t_mod > HALF_PWM_CYCLE){
      t_mod -=  HALF_PWM_CYCLE;
      PwmRampDir *= -1.0;
    }
 
  }
#ifdef ENABLE_OMEGA_CTRL

    t_scaled_shifted_omega = t * t_freq_scale + t_shift_omega;

    if (t_scaled_shifted_omega >= t_prev_omega){

      t_mod_omega += (t_scaled_shifted_omega - t_prev_omega);
      t_prev_omega = t_scaled_shifted_omega;

      if(t_mod_omega > HALF_PWM_CYCLE){
        t_mod_omega -=  HALF_PWM_CYCLE;
        PwmRampDir_omega *= -1.0;
      }

    }
#endif

    if(OmegaCtrl)
    	y = PWM_GAIN*(t_mod_omega/HALF_PWM_CYCLE - .5)*PwmRampDir_omega;
    else
    	y = PWM_GAIN*(t_mod/HALF_PWM_CYCLE - .5)*PwmRampDir;



}

void Vm3::SrcRValueUpdate(void)         
{
  SrcObjItem * pCurSrcItem;
 

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((ECtrl3 *) pCurSrcItem->pSrcObject)->V_410 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl7 *) pCurSrcItem->pSrcObject)->V_410 = y;



}


// -----------------------------------------------------------------




// ---- ECtrl1  ----------------------------------------------------

ECtrl1::ECtrl1(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;

  SrcFuncName = SRC_FUNC_ECtrl1;
  LiteralName = "ECtrl1";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_210 = 0;
  GateCtrl_111 = 0;
  GateCtrl_121 = 0;
  	  //default circulating state for this cell.
  CircState = 0;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS11];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS12];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS21];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS22];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS31];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS32];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS41];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS42];


  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];


#ifdef PLOT_SRCOBJECT_PROBES_ECtrl1
  DoProbes = TRUE;
#endif

}

ECtrl1::~ECtrl1(void)
{



}

void ECtrl1::SrcFunction(double t) 
{

  if(t > Time_0){
    if((V_210 + V_112) > 0){
        GateCtrl_111 = 1;
    }
    else{
        GateCtrl_111 = 0;
    }

    if((-V_210 - V_112) > 0){
      GateCtrl_121 = 1;
    }
    else{
      GateCtrl_121 = 0;
    }
  }
  else{
    GateCtrl_111 = 0;
    GateCtrl_121 = 0;   

  }
     
}

void ECtrl1::RecordProbes(void)
{
  Gate_111.push_back((double) GateCtrl_111);
  Gate_121.push_back((double) GateCtrl_121);
}

void ECtrl1::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_111, "Gate_111");
    SimuPlot.plot_xy(Plot_t, Gate_121, "Gate_121");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_111"){
	SimuPlot.plot_xy(Plot_t, Gate_111, "Gate_111");
      }
      else if(TagNamesToPlot[i] == "Gate_121"){
	SimuPlot.plot_xy(Plot_t, Gate_121, "Gate_121");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}




void ECtrl1::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS11 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_111;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS12 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_121; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS21 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_111;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS22 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_121; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS31 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_111;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS32 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_121; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS41 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_111;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS42 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_121; return;}



}

void ECtrl1::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_111 = GateCtrl_111;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_121 = GateCtrl_121;
}


// -----------------------------------------------------------------

// ---- ECtrl2  ----------------------------------------------------

ECtrl2::ECtrl2(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;

  SrcFuncName = SRC_FUNC_ECtrl2;
  LiteralName = "ECtrl2";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_110 = 0;
  GateCtrl_211 = 0;
  GateCtrl_221 = 0;
	 //default circulating state for this cell.
  CircState = 1;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS11];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS12];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS21];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS22];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS31];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS32];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS41];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS42];

  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];



#ifdef PLOT_SRCOBJECT_PROBES_ECtrl2
  DoProbes = TRUE; 
#endif

}

ECtrl2::~ECtrl2(void)
{



}

void ECtrl2::SrcFunction(double t)
{

  if(t > Time_0){
    if((V_110 + V_112) > 0){
        GateCtrl_211 = 1;
    }
    else{
        GateCtrl_211 = 0;
    }

    if((-V_110 - V_112) > 0){
      GateCtrl_221 = 1;
    }
    else{
      GateCtrl_221 = 0;
    }
  }
  else{
    GateCtrl_211 = 0;
    GateCtrl_221 = 0;    

  }

}

void ECtrl2::RecordProbes(void)
{
  Gate_211.push_back((double) GateCtrl_211);
  Gate_221.push_back((double) GateCtrl_221);
}

void ECtrl2::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_211, "Gate_211");
    SimuPlot.plot_xy(Plot_t, Gate_221, "Gate_221");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_211"){
	SimuPlot.plot_xy(Plot_t, Gate_211, "Gate_211");
      }
      else if(TagNamesToPlot[i] == "Gate_221"){
	SimuPlot.plot_xy(Plot_t, Gate_221, "Gate_221");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}

void ECtrl2::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS11 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_211;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS12 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_221; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS21 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_211;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS22 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_221; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS31 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_211;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS32 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_221; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS41 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_211;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS42 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_221; return;}

}

void ECtrl2::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_211 = GateCtrl_211;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_221 = GateCtrl_221;
}


// -----------------------------------------------------------------

// ---- ECtrl3  ----------------------------------------------------

ECtrl3::ECtrl3(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;


  SrcFuncName = SRC_FUNC_ECtrl3;
  LiteralName = "ECtrl3";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_410 = 0;
  GateCtrl_311 = 0;
  GateCtrl_321 = 0;
	 //default circulating state for this cell.
  CircState = 2;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS11];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS12];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS21];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS22];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS31];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS32];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS41];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS42];

  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];



#ifdef PLOT_SRCOBJECT_PROBES_ECtrl3
  DoProbes = TRUE; 
#endif


}

ECtrl3::~ECtrl3(void)
{



}

void ECtrl3::SrcFunction(double t)
{

  if(t > Time_0){
    if((V_410 + V_112) > 0){
        GateCtrl_311 = 1;
    }
    else{
        GateCtrl_311 = 0;
    }

    if((-V_410 - V_112) > 0){
      GateCtrl_321 = 1;
    }
    else{
      GateCtrl_321 = 0;
    }
  }
  else{
    GateCtrl_311 = 0;
    GateCtrl_321 = 0;    

  }

}

void ECtrl3::RecordProbes(void)
{
  Gate_311.push_back((double) GateCtrl_311);
  Gate_321.push_back((double) GateCtrl_321);
}

void ECtrl3::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_311, "Gate_311");
    SimuPlot.plot_xy(Plot_t, Gate_321, "Gate_321");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_311"){
	SimuPlot.plot_xy(Plot_t, Gate_311, "Gate_311");
      }
      else if(TagNamesToPlot[i] == "Gate_321"){
	SimuPlot.plot_xy(Plot_t, Gate_321, "Gate_321");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}

void ECtrl3::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS11 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_311;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS12 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_321; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS21 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_311;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS22 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_321; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS31 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_311;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS32 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_321; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS41 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_311;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS42 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_321; return;}

}


void ECtrl3::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_311 = GateCtrl_311;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_321 = GateCtrl_321;
}


// -----------------------------------------------------------------

// ---- ECtrl4  ----------------------------------------------------

ECtrl4::ECtrl4(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;

  SrcFuncName = SRC_FUNC_ECtrl4;
  LiteralName = "ECtrl4";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_310 = 0;
  GateCtrl_411 = 0;
  GateCtrl_421 = 0;
	 //default circulating state for this cell.
  CircState = 3;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS11];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS12];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS21];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS22];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS31];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS32];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS41];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS42];


  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];


#ifdef PLOT_SRCOBJECT_PROBES_ECtrl4
  DoProbes = TRUE; 
#endif


}

ECtrl4::~ECtrl4(void)
{



}

void ECtrl4::SrcFunction(double t)
{

  if(t > Time_0){
    if((V_310 + V_112) > 0){
        GateCtrl_411 = 1;
    }
    else{
        GateCtrl_411 = 0;
    }

    if((-V_310 - V_112) > 0){
      GateCtrl_421 = 1;
    }
    else{
      GateCtrl_421 = 0;
    }
  }
  else{
    GateCtrl_411 = 0;
    GateCtrl_421 = 0;    

  }

}

void ECtrl4::RecordProbes(void)
{
  Gate_411.push_back((double) GateCtrl_411);
  Gate_421.push_back((double) GateCtrl_421);
}

void ECtrl4::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_411, "Gate_411");
    SimuPlot.plot_xy(Plot_t, Gate_421, "Gate_421");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_411"){
	SimuPlot.plot_xy(Plot_t, Gate_411, "Gate_411");
      }
      else if(TagNamesToPlot[i] == "Gate_421"){
	SimuPlot.plot_xy(Plot_t, Gate_421, "Gate_421");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}

void ECtrl4::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS11 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_411;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS12 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_421; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS21 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_411;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS22 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_421; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS31 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_411;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS32 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_421; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS41 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_411;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS42 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_421; return;}

}

void ECtrl4::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_411 = GateCtrl_411;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_421 = GateCtrl_421;
}

// -----------------------------------------------------------------


// ---- ECtrl5  ----------------------------------------------------

ECtrl5::ECtrl5(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;

  SrcFuncName = SRC_FUNC_ECtrl5;
  LiteralName = "ECtrl5";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_210 = 0;
  GateCtrl_511 = 0;
  GateCtrl_521 = 0;
	 //default circulating state for this cell.
  CircState = 0;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS51];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS52];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS61];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS62];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS71];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS72];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS81];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS82];


  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];




#ifdef PLOT_SRCOBJECT_PROBES_ECtrl5
  DoProbes = TRUE; 
#endif


}

ECtrl5::~ECtrl5(void)
{



}

void ECtrl5::SrcFunction(double t)
{

  if(t > Time_0){
    if((-V_210 - V_112) > 0){
        GateCtrl_511 = 1;
    }
    else{
        GateCtrl_511 = 0;
    }

    if((V_210 + V_112) > 0){
      GateCtrl_521 = 1;
    }
    else{
      GateCtrl_521 = 0;
    }
  }
  else{
    GateCtrl_511 = 0;
    GateCtrl_521 = 0;    

  }

}

void ECtrl5::RecordProbes(void)
{
  Gate_511.push_back((double) GateCtrl_511);
  Gate_521.push_back((double) GateCtrl_521);
}

void ECtrl5::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_511, "Gate_511");
    SimuPlot.plot_xy(Plot_t, Gate_521, "Gate_521");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_511"){
	SimuPlot.plot_xy(Plot_t, Gate_511, "Gate_511");
      }
      else if(TagNamesToPlot[i] == "Gate_521"){
	SimuPlot.plot_xy(Plot_t, Gate_521, "Gate_521");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}


void ECtrl5::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS51 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_511;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS52 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_521; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS61 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_511;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS62 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_521; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS71 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_511;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS72 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_521; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS81 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_511;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS82 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_521; return;}



}

void ECtrl5::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_511 = GateCtrl_511;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_521 = GateCtrl_521;
}


// -----------------------------------------------------------------


// ---- ECtrl6  ----------------------------------------------------

ECtrl6::ECtrl6(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;

  SrcFuncName = SRC_FUNC_ECtrl6;
  LiteralName = "ECtrl6";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_110 = 0;
  GateCtrl_611 = 0;
  GateCtrl_621 = 0;
	 //default circulating state for this cell.
  CircState = 1;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS51];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS52];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS61];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS62];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS71];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS72];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS81];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS82];

  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];



#ifdef PLOT_SRCOBJECT_PROBES_ECtrl6
  DoProbes = TRUE; 
#endif


}

ECtrl6::~ECtrl6(void)
{



}

void ECtrl6::SrcFunction(double t)
{

  if(t > Time_0){
    if((-V_110 - V_112) > 0){
        GateCtrl_611 = 1;
    }
    else{
        GateCtrl_611 = 0;
    }

    if((V_110 + V_112) > 0){
      GateCtrl_621 = 1;
    }
    else{
      GateCtrl_621 = 0;
    }
  }
  else{
    GateCtrl_611 = 0;
    GateCtrl_621 = 0;    

  }

}

void ECtrl6::RecordProbes(void)
{
  Gate_611.push_back((double) GateCtrl_611);
  Gate_621.push_back((double) GateCtrl_621);
}

void ECtrl6::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_611, "Gate_611");
    SimuPlot.plot_xy(Plot_t, Gate_621, "Gate_621");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_611"){
	SimuPlot.plot_xy(Plot_t, Gate_611, "Gate_611");
      }
      else if(TagNamesToPlot[i] == "Gate_621"){
	SimuPlot.plot_xy(Plot_t, Gate_621, "Gate_621");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}


void ECtrl6::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS51 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_611;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS52 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_621; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS61 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_611;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS62 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_621; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS71 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_611;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS72 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_621; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS81 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_611;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS82 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_621; return;}

}

void ECtrl6::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_611 = GateCtrl_611;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_621 = GateCtrl_621;
}


// -----------------------------------------------------------------


// ---- ECtrl7  ----------------------------------------------------

ECtrl7::ECtrl7(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;


  SrcFuncName = SRC_FUNC_ECtrl7;
  LiteralName = "ECtrl7";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_410 = 0;
  GateCtrl_711 = 0;
  GateCtrl_721 = 0;
	 //default circulating state for this cell.
  CircState = 2;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS51];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS52];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS61];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS62];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS71];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS72];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS81];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS82];

  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];



#ifdef PLOT_SRCOBJECT_PROBES_ECtrl7
  DoProbes = TRUE; 
#endif


}

ECtrl7::~ECtrl7(void)
{



}

void ECtrl7::SrcFunction(double t)
{

  if(t > Time_0){
    if((-V_410 - V_112) > 0){
        GateCtrl_711 = 1;
    }
    else{
        GateCtrl_711 = 0;
    }

    if((V_410 + V_112) > 0){
      GateCtrl_721 = 1;
    }
    else{
      GateCtrl_721 = 0;
    }
  }
  else{
    GateCtrl_711 = 0;
    GateCtrl_721 = 0;    

  }

}

void ECtrl7::RecordProbes(void)
{
  Gate_711.push_back((double) GateCtrl_711);
  Gate_721.push_back((double) GateCtrl_721);
}

void ECtrl7::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_711, "Gate_711");
    SimuPlot.plot_xy(Plot_t, Gate_721, "Gate_721");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_711"){
	SimuPlot.plot_xy(Plot_t, Gate_711, "Gate_711");
      }
      else if(TagNamesToPlot[i] == "Gate_721"){
	SimuPlot.plot_xy(Plot_t, Gate_721, "Gate_721");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}




void ECtrl7::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS51 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_711;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS52 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_721; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS61 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_711;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS62 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_721; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS71 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_711;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS72 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_721; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS81 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_711;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS82 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_721; return;}

}

void ECtrl7::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_711 = GateCtrl_711;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_721 = GateCtrl_721;
}




// -----------------------------------------------------------------


// ---- ECtrl8  ----------------------------------------------------

ECtrl8::ECtrl8(void)
{
  SwitchObjItem * pCurSwitchItem;
  SrcObjItem * pCurSrcItem;

  SrcFuncName = SRC_FUNC_ECtrl8;
  LiteralName = "ECtrl8";

  Time_0 = SWITCH_ENABLE_TIME;
  V_112 = 0;
  V_310 = 0;
  GateCtrl_811 = 0;
  GateCtrl_821 = 0;
	 //default circulating state for this cell.
  CircState = 3;

 //build the Switch Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS51];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS52];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS61];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS62];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS71];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS72];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS81];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS82];

  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];



#ifdef PLOT_SRCOBJECT_PROBES_ECtrl8
  DoProbes = TRUE; 
#endif


}

ECtrl8::~ECtrl8(void)
{



}

void ECtrl8::SrcFunction(double t)
{

  if(t > Time_0){
    if((-V_310 - V_112) > 0){
        GateCtrl_811 = 1;
    }
    else{
        GateCtrl_811 = 0;
    }

    if((V_310 + V_112) > 0){
      GateCtrl_821 = 1;
    }
    else{
      GateCtrl_821 = 0;
    }
  }
  else{
    GateCtrl_811 = 0;
    GateCtrl_821 = 0;    

  }

}

void ECtrl8::RecordProbes(void)
{
  Gate_811.push_back((double) GateCtrl_811);
  Gate_821.push_back((double) GateCtrl_821);
}

void ECtrl8::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
    SimuPlot.plot_xy(Plot_t, Gate_811, "Gate_811");
    SimuPlot.plot_xy(Plot_t, Gate_821, "Gate_821");
  }
  else{
    for(i = 0; i < 20; i++){
      if(TagNamesToPlot[i] == "Gate_811"){
	SimuPlot.plot_xy(Plot_t, Gate_811, "Gate_811");
      }
      else if(TagNamesToPlot[i] == "Gate_821"){
	SimuPlot.plot_xy(Plot_t, Gate_821, "Gate_821");
      }
      else if(TagNamesToPlot[i] == ""){
	break;
      }
    }
  }   
}





void ECtrl8::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update RValues
  pCurSwitchItem = pSwitchObjRValList;

  if(CircState == 0){((Switch_rS51 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_811;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 0){((Switch_rS52 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_821; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS61 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_811;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 1){((Switch_rS62 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_821; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS71 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_811;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 2){((Switch_rS72 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_821; return;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS81 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_811;}
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  if(CircState == 3){((Switch_rS82 *) pCurSwitchItem->pSwitchObject)->DevGate = GateCtrl_821; return;}

}

void ECtrl8::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_811 = GateCtrl_811;
  ((Vref *) pCurSrcItem->pSrcObject)->GateCtrl_821 = GateCtrl_821;
}



// -----------------------------------------------------------------






// ---- Vref  ----------------------------------------------------

Vref::Vref(void)
{
 
  SrcObjItem * pCurSrcItem;


  SrcFuncName = SRC_FUNC_Vref;
  LiteralName = "Vref";


  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl1];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl2];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl3];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl4];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl5];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl6];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl7];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_ECtrl8];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vm1];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vm2];
  pCurSrcItem->pNextSrcItem = new SrcObjItem;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vm3];



  CircStateIndex = 0;
  CircStateRate = 0;
  RailToRailStateIndex = 0;
  RailToRailStateRate = 0;

  SignalCirculationUpdate = FALSE;
  RailToRailPwmRef = 0;

  SignalCirculationUpdateFast = FALSE;
  RailToRailPwmRefFast = 0;
  RailToRailPwmRefFastRamp = 0;

  OmegaCtrl = FALSE;


#ifdef PLOT_SRCOBJECT_Vref
  PlotThisOutput = TRUE;
  DoProbes = TRUE;
#endif

  Plot_Tag = "Vref";
}

Vref::~Vref(void)
{


}

void Vref::SrcFunction(double t)
{
   double SubModCommand_0;
   double SubModCommand_1;
   double SubModCommand_2;
   double PwmRef;
   double CmdRef;
   bool SignalCellSelectionUpdate = FALSE;
   unsigned char GateVector = 0;
   double VoltageVal[8];
   unsigned int VoltageIdx[8];
   double TempVoltageVal;
   unsigned int TempVoltageIdx;

#ifdef ENABLE_CELL_CIRCULATION

	  //Define fast update  for circulation.
//#define CICULATOR_USES_FAST_REF

#ifdef CICULATOR_USES_FAST_REF
  if(SignalCirculationUpdateFast == TRUE)
#else
  if(SignalCirculationUpdate == TRUE)
#endif
  {

	  if(CircStateRate == CIRCULAR_STATE_RATE)
	  {
		  CircStateIndex = (CircStateIndex + 1) & 0x7; 	//(used if CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is not defined)
		  CircStateRate = 0;
		  SignalCellSelectionUpdate = TRUE;				//(used if CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT is defined)
	  }
	  else
	  {
		  CircStateRate++;
	  }

  }

#ifdef CELL_SELECTION_MODE_FOR_PROVISIONAL_PATENT_AMENDMENT

  if(SignalCellSelectionUpdate)
  {
	  	  //Table index is always "0" for this mode.
	  CircStateIndex = 0;


	  //Update only only if we are not in a dead-time region.
	  if((GateCtrl_111 || GateCtrl_121) &&
	     (GateCtrl_211 || GateCtrl_221) &&
	     (GateCtrl_311 || GateCtrl_321) &&
	     (GateCtrl_411 || GateCtrl_421) &&
	     (GateCtrl_511 || GateCtrl_521) &&
	     (GateCtrl_611 || GateCtrl_621) &&
	     (GateCtrl_711 || GateCtrl_721) &&
	     (GateCtrl_811 || GateCtrl_821))
	  {
		  	  //Our definition here is that "GateVector" represents the states where cell voltage is impressed into the circuit. (In other
		  	  //words, the bottom IGBT of a given cell is turned on.)
		  if(GateCtrl_121)
			  GateVector |= 0x80;
		  if(GateCtrl_221)
			  GateVector |= 0x40;
		  if(GateCtrl_321)
			  GateVector |= 0x20;
		  if(GateCtrl_421)
			  GateVector |= 0x10;
		  if(GateCtrl_521)
			  GateVector |= 0x8;
		  if(GateCtrl_621)
			  GateVector |= 0x4;
		  if(GateCtrl_721)
			  GateVector |= 0x2;
		  if(GateCtrl_821)
			  GateVector |= 0x1;


		   //because this is only 5 level, we will sort long-hand...

		  VoltageVal[7] = v11_v12;		//Cell 3p
		  VoltageIdx[7] = 0;
		  VoltageVal[6] = v21_v23;		//Cell 2p
		  VoltageIdx[6] = 1;
		  VoltageVal[5] = v31_v34;		//Cell 1p
		  VoltageIdx[5] = 2;
		  VoltageVal[4] = v41_v44;		//Cell 0p
		  VoltageIdx[4] = 3;


		  if(VoltageVal[7] < VoltageVal[6])
		  {		//swap 7 and 6
			  TempVoltageVal = VoltageVal[7];
			  TempVoltageIdx = VoltageIdx[7];
			  VoltageVal[7] = VoltageVal[6];
			  VoltageIdx[7] = VoltageIdx[6];
			  VoltageVal[6] = TempVoltageVal;
			  VoltageIdx[6] = TempVoltageIdx;
		  }

		  if(VoltageVal[7] < VoltageVal[5])
		  {
			  	  //5 goes to the top.
			  TempVoltageVal = VoltageVal[7];
			  TempVoltageIdx = VoltageIdx[7];
			  VoltageVal[7] = VoltageVal[5];
			  VoltageIdx[7] = VoltageIdx[5];
			  VoltageVal[5] = VoltageVal[6];
			  VoltageIdx[5] = VoltageIdx[6];
			  VoltageVal[6] = TempVoltageVal;
			  VoltageIdx[6] = TempVoltageIdx;
		  }
		  else if(VoltageVal[6] < VoltageVal[5])
		  {
			  	  //5 goes between 7 and 6
			  TempVoltageVal = VoltageVal[6];
			  TempVoltageIdx = VoltageIdx[6];
			  VoltageVal[6] = VoltageVal[5];
			  VoltageIdx[6] = VoltageIdx[5];
			  VoltageVal[5] = TempVoltageVal;
			  VoltageIdx[5] = TempVoltageIdx;

		  } //else 5 in its proper place.

		  if(VoltageVal[7] < VoltageVal[4])
		  {
			  	  //4 goes to the top.
			  TempVoltageVal = VoltageVal[7];
			  TempVoltageIdx = VoltageIdx[7];
			  VoltageVal[7] = VoltageVal[4];
			  VoltageIdx[7] = VoltageIdx[4];
			  VoltageVal[4] = VoltageVal[5];
			  VoltageIdx[4] = VoltageIdx[5];
			  VoltageVal[5] = VoltageVal[6];
			  VoltageIdx[5] = VoltageIdx[6];
			  VoltageVal[6] = TempVoltageVal;
			  VoltageIdx[6] = TempVoltageIdx;
		  }
		  else if(VoltageVal[6] < VoltageVal[4])
		  {
			  	  //4 goes between 7 and 6
			  TempVoltageVal = VoltageVal[6];
			  TempVoltageIdx = VoltageIdx[6];
			  VoltageVal[6] = VoltageVal[4];
			  VoltageIdx[6] = VoltageIdx[4];
			  VoltageVal[4] = VoltageVal[5];
			  VoltageIdx[4] = VoltageIdx[5];
			  VoltageVal[5] = TempVoltageVal;
			  VoltageIdx[5] = TempVoltageIdx;
		  }
		  else if(VoltageVal[5] < VoltageVal[4])
		  {
			  	  //4 goes between 6 and 5
			  TempVoltageVal = VoltageVal[5];
			  TempVoltageIdx = VoltageIdx[5];
			  VoltageVal[5] = VoltageVal[4];
			  VoltageIdx[5] = VoltageIdx[4];
			  VoltageVal[4] = TempVoltageVal;
			  VoltageIdx[4] = TempVoltageIdx;
		  }  //else 4 in its proper place.



		  VoltageVal[3] = v51_v56;		//Cell 3n
		  VoltageIdx[3] = 0;
		  VoltageVal[2] = v61_v67;		//Cell 2n
		  VoltageIdx[2] = 1;
		  VoltageVal[1] = v71_v78;		//Cell 1n
		  VoltageIdx[1] = 2;
		  VoltageVal[0] = v81_v88;		//Cell 0n
		  VoltageIdx[0] = 3;


		  if(VoltageVal[3] < VoltageVal[2])
		  {		//swap 3 and 2
			  TempVoltageVal = VoltageVal[3];
			  TempVoltageIdx = VoltageIdx[3];
			  VoltageVal[3] = VoltageVal[2];
			  VoltageIdx[3] = VoltageIdx[2];
			  VoltageVal[2] = TempVoltageVal;
			  VoltageIdx[2] = TempVoltageIdx;
		  }

		  if(VoltageVal[3] < VoltageVal[1])
		  {
			  	  //1 goes to the top.
			  TempVoltageVal = VoltageVal[3];
			  TempVoltageIdx = VoltageIdx[3];
			  VoltageVal[3] = VoltageVal[1];
			  VoltageIdx[3] = VoltageIdx[1];
			  VoltageVal[1] = VoltageVal[2];
			  VoltageIdx[1] = VoltageIdx[2];
			  VoltageVal[2] = TempVoltageVal;
			  VoltageIdx[2] = TempVoltageIdx;
		  }
		  else if(VoltageVal[2] < VoltageVal[1])
		  {
			  	  //1 goes between 3 and 2
			  TempVoltageVal = VoltageVal[2];
			  TempVoltageIdx = VoltageIdx[2];
			  VoltageVal[2] = VoltageVal[1];
			  VoltageIdx[2] = VoltageIdx[1];
			  VoltageVal[1] = TempVoltageVal;
			  VoltageIdx[1] = TempVoltageIdx;

		  } //else 1 in its proper place.

		  if(VoltageVal[3] < VoltageVal[0])
		  {
			  	  //0 goes to the top.
			  TempVoltageVal = VoltageVal[3];
			  TempVoltageIdx = VoltageIdx[3];
			  VoltageVal[3] = VoltageVal[0];
			  VoltageIdx[3] = VoltageIdx[0];
			  VoltageVal[0] = VoltageVal[1];
			  VoltageIdx[0] = VoltageIdx[1];
			  VoltageVal[1] = VoltageVal[2];
			  VoltageIdx[1] = VoltageIdx[2];
			  VoltageVal[2] = TempVoltageVal;
			  VoltageIdx[2] = TempVoltageIdx;
		  }
		  else if(VoltageVal[2] < VoltageVal[0])
		  {
			  	  //0 goes between 3 and 2
			  TempVoltageVal = VoltageVal[2];
			  TempVoltageIdx = VoltageIdx[2];
			  VoltageVal[2] = VoltageVal[0];
			  VoltageIdx[2] = VoltageIdx[0];
			  VoltageVal[0] = VoltageVal[1];
			  VoltageIdx[0] = VoltageIdx[1];
			  VoltageVal[1] = TempVoltageVal;
			  VoltageIdx[1] = TempVoltageIdx;
		  }
		  else if(VoltageVal[1] < VoltageVal[0])
		  {
			  	  //0 goes between 2 and 1
			  TempVoltageVal = VoltageVal[1];
			  TempVoltageIdx = VoltageIdx[1];
			  VoltageVal[1] = VoltageVal[0];
			  VoltageIdx[1] = VoltageIdx[0];
			  VoltageVal[0] = TempVoltageVal;
			  VoltageIdx[0] = TempVoltageIdx;
		  }  //else 0 in its proper place.

		  	  //For "p" leg, if "ia" > 0, we choose lowest voltage(s), and for "n" leg, we choose the highest voltages(s).
		      //Reverse the logic for "ia" < 0". If "ia" = 0, do nothing (later, we may choose an absolute minimum level
		      //instead of "0").

		  if(ia == 0)
			  return;


		  	  //Long-hand and no optimization on logic for now (we may need to keep track of states later)...

		  if(GateVector == 0xf0)  				// 1 state of output at +BUS
		  {
			  //Do nothing.
		  }

		  else if(GateVector == 0x71)          // 16 states of output at 1/2 +BUS
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];    //Minimum voltage should be placed on Cell 3p position
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[3];    //Maximum voltage should be placed on Cell's 3n, 2n and 1n positions
				  CircStateTable[0][5] = VoltageIdx[2];	   //(Notice how these fields for GateVector states 0x72, 0x74 and 0x78 below are shifted.
				  CircStateTable[0][6] = VoltageIdx[1];    // This is for order only because the states are for cells that are "off". Thus, there effect
				  CircStateTable[0][7] = VoltageIdx[0];    // is redundant.)
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];	   //Maximum voltage should be placed on Cell 3p positions
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[0];	   //Minimum voltage should be placed on Cell's 3n, 2n and 1n positions
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
		  }
		  else if(GateVector == 0x72)
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];     //And so forth...
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
		  }
		  else if(GateVector == 0x74)
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
		  }
		  else if(GateVector == 0x78)
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
		  }
		  else if(GateVector == 0xb1)
		  {
			  	  //continue by copying 0x71 here and change only indexes 0,1,2 and 3 for each direction of current

				  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0xb2)
		  {
					  //continue by copying 0x72 here and change only indexes 0,1,2 and 3 to reflect 0xb1
				  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
		  }
		  else if(GateVector == 0xb4)
		  {
			  	  	  //continue by copying 0x74 here and change only indexes 0,1,2 and 3 to reflect 0xb1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0xb8)
		  {
			  	  	  //continue by copying 0x78 here and change only indexes 0,1,2 and 3 to reflect 0xb1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
		  }
		  else if(GateVector == 0xd1)
		  {
			  	  //continue by copying 0xb1 here and change only indexes 0,1,2 and 3 for each direction of current
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0xd2)
		  {
			  //continue by copying 0xb2 here and change only indexes 0,1,2 and 3 to reflect 0xd1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
		  }
		  else if(GateVector == 0xd4)
		  {
			  	  //continue by copying 0xb4 here and change only indexes 0,1,2 and 3 to reflect 0xd1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0xd8)
		  {
			  	  //continue by copying 0xb8 here and change only indexes 0,1,2 and 3 to reflect 0xd1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
		  }
		  else if(GateVector == 0xe1)
		  {
			  	  //continue by copying 0xd1 here and change only indexes 0,1,2 and 3 for each direction of current
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0xe2)
		  {
			  	  //continue by copying 0xd2 here and change only indexes 0,1,2 and 3 to reflect 0xe1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
		  }
		  else if(GateVector == 0xe4)
		  {
			  	  //continue by copying 0xd4 here and change only indexes 0,1,2 and 3 to reflect 0xe1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0xe8)
		  {
			  	  //continue by copying 0xd8 here and change only indexes 0,1,2 and 3 to reflect 0xe1
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
		  }

		  else if(GateVector == 0x33)		// 16 states of output at 0
		  {
			  	  	  //For the next 16 states, 2 of the lowest voltages and 2 of the highest voltages are selected.
				  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0x3c)
		  {
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0x3a)
		  {
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0x35)
		  {
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[4];
					  CircStateTable[0][1] = VoltageIdx[5];
					  CircStateTable[0][2] = VoltageIdx[6];
					  CircStateTable[0][3] = VoltageIdx[7];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[7];
					  CircStateTable[0][1] = VoltageIdx[6];
					  CircStateTable[0][2] = VoltageIdx[5];
					  CircStateTable[0][3] = VoltageIdx[4];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0xc3)
		  {
			  	  //continue by copying 0x33 here and change only indexes 0,1,2 and 3 for each direction of current
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0xcc)
		  {
			  	  //continue by copying 0x3c here and change only indexes 0,1,2 and 3 to reflect 0xc3
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0xca)
		  {
			  //continue by copying 0x3a here and change only indexes 0,1,2 and 3 to reflect 0xc3
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0xc5)
		  {
			  //continue by copying 0x35 here and change only indexes 0,1,2 and 3 to reflect 0xc3
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0xa3)
		  {
			  	  //continue by copying 0xc3 here and change only indexes 0,1,2 and 3 for each direction of current
			 	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0xac)
		  {
			  //continue by copying 0xcc here and change only indexes 0,1,2 and 3 to reflect 0xa3
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0xaa)
		  {
			  //continue by copying 0xca here and change only indexes 0,1,2 and 3 to reflect 0xa3
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[1];
					  CircStateTable[0][5] = VoltageIdx[3];
					  CircStateTable[0][6] = VoltageIdx[0];
					  CircStateTable[0][7] = VoltageIdx[2];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[2];
					  CircStateTable[0][5] = VoltageIdx[0];
					  CircStateTable[0][6] = VoltageIdx[3];
					  CircStateTable[0][7] = VoltageIdx[1];
				  }
		  }
		  else if(GateVector == 0xa5)
		  {
			  //continue by copying 0xc5 here and change only indexes 0,1,2 and 3 to reflect 0xa3
			  	  if(ia > 0)
				  {
					  CircStateTable[0][0] = VoltageIdx[6];
					  CircStateTable[0][1] = VoltageIdx[4];
					  CircStateTable[0][2] = VoltageIdx[7];
					  CircStateTable[0][3] = VoltageIdx[5];

					  CircStateTable[0][4] = VoltageIdx[3];
					  CircStateTable[0][5] = VoltageIdx[1];
					  CircStateTable[0][6] = VoltageIdx[2];
					  CircStateTable[0][7] = VoltageIdx[0];
				  }
				  else
				  {
					  CircStateTable[0][0] = VoltageIdx[5];
					  CircStateTable[0][1] = VoltageIdx[7];
					  CircStateTable[0][2] = VoltageIdx[4];
					  CircStateTable[0][3] = VoltageIdx[6];

					  CircStateTable[0][4] = VoltageIdx[0];
					  CircStateTable[0][5] = VoltageIdx[2];
					  CircStateTable[0][6] = VoltageIdx[1];
					  CircStateTable[0][7] = VoltageIdx[3];
				  }
		  }
		  else if(GateVector == 0x53)
		  {
			  //continue by copying 0xa3 here and change only indexes 0,1,2 and 3 for each direction of current
		 	  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
		  }
		  else if(GateVector == 0x5c)
		  {
			  //continue by copying 0xac here and change only indexes 0,1,2 and 3 to reflect 0x53
		  	  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
		  }
		  else if(GateVector == 0x5a)
		  {
			  //continue by copying 0xaa here and change only indexes 0,1,2 and 3 to reflect 0x53
		  	  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
		  }
		  else if(GateVector == 0x55)
		  {
			  //continue by copying 0xa5 here and change only indexes 0,1,2 and 3 to reflect 0x53
		  	  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
		  }


		  else if(GateVector == 0x17)          // 16 states of output at 1/2 -BUS
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];     //Minimum voltage should be placed on Cell's 3p, 2p and 1p positions
				  CircStateTable[0][1] = VoltageIdx[5];		//(Notice how these fields for GateVector states 0x27, 0x47 and 0x87 below are shifted.
				  CircStateTable[0][2] = VoltageIdx[6];		// This is for order only because the states are for cells that are "off". Thus, there effect
				  CircStateTable[0][3] = VoltageIdx[7];		// is redundant.)

				  CircStateTable[0][4] = VoltageIdx[3];   	//Maximum voltage should be placed on Cell 3n position
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];	    //Maximum voltage should be placed on Cell's 3p, 2p and 1p positions
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[0];	    //Minimum voltage should be placed on Cell 3n position
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
		  }
		  else if(GateVector == 0x27)
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];     //And so forth...
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
		  }
		  else if(GateVector == 0x47)
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
		  }
		  else if(GateVector == 0x87)
		  {
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
		  }
		  else if(GateVector == 0x1b)
		  {
			  //continue by copying 0x17 here and change only indexes 4,5,6 and 7 for each direction of current
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
		  }
		  else if(GateVector == 0x2b)
		  {
			  //continue by copying 0x27 here and change only indexes 4,5,6 and 7 to reflect 0x1b
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
		  }
		  else if(GateVector == 0x4b)
		  {
			  //continue by copying 0x47 here and change only indexes 4,5,6 and 7 to reflect 0x1b
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
		  }
		  else if(GateVector == 0x8b)
		  {
			  //continue by copying 0x87 here and change only indexes 4,5,6 and 7 to reflect 0x1b
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[0];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[2];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[3];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[1];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
		  }
		  else if(GateVector == 0x1d)
		  {
			  //continue by copying 0x1b here and change only indexes 4,5,6 and 7 for each direction of current
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }

		  }
		  else if(GateVector == 0x2d)
		  {
			  //continue by copying 0x2b here and change only indexes 4,5,6 and 7 to reflect 0x1d
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }

		  }
		  else if(GateVector == 0x4d)
		  {
			  //continue by copying 0x4b here and change only indexes 4,5,6 and 7 to reflect 0x1d
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }
		  }
		  else if(GateVector == 0x8d)
		  {
			  //continue by copying 0x8b here and change only indexes 4,5,6 and 7 to reflect 0x1d
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[0];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[2];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[3];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[1];
			  }

		  }
		  else if(GateVector == 0x1e)
		  {
			  //continue by copying 0x1d here and change only indexes 4,5,6 and 7 for each direction of current
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
		  }
		  else if(GateVector == 0x2e)
		  {
			  //continue by copying 0x2d here and change only indexes 4,5,6 and 7 to reflect 0x1e
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[6];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[4];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[5];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[7];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
		  }
		  else if(GateVector == 0x4e)
		  {
			  //continue by copying 0x4d here and change only indexes 4,5,6 and 7 to reflect 0x1e
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[6];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[4];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[5];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[7];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
		  }
		  else if(GateVector == 0x8e)
		  {
			  //continue by copying 0x8d here and change only indexes 4,5,6 and 7 to reflect 0x1e
			  if(ia > 0)
			  {
				  CircStateTable[0][0] = VoltageIdx[7];
				  CircStateTable[0][1] = VoltageIdx[4];
				  CircStateTable[0][2] = VoltageIdx[5];
				  CircStateTable[0][3] = VoltageIdx[6];

				  CircStateTable[0][4] = VoltageIdx[2];
				  CircStateTable[0][5] = VoltageIdx[1];
				  CircStateTable[0][6] = VoltageIdx[0];
				  CircStateTable[0][7] = VoltageIdx[3];
			  }
			  else
			  {
				  CircStateTable[0][0] = VoltageIdx[4];
				  CircStateTable[0][1] = VoltageIdx[7];
				  CircStateTable[0][2] = VoltageIdx[6];
				  CircStateTable[0][3] = VoltageIdx[5];

				  CircStateTable[0][4] = VoltageIdx[1];
				  CircStateTable[0][5] = VoltageIdx[2];
				  CircStateTable[0][6] = VoltageIdx[3];
				  CircStateTable[0][7] = VoltageIdx[0];
			  }
		  }

		  else if(GateVector == 0x0f)  				// 1 state of output at -BUS
		  {
			  //Do nothing.
		  }


	  }
  }

#endif

#endif


#ifdef RAIL_TO_RAIL_STEP_VREF

#ifdef RAIL_TO_RAIL_AC_TEST

 // To generate the plots for this test, do the following:
 //
 //     - Set CONSTANT_CURRENT_SOURCE_LOAD to "20.0"
 //     - Define SINUSOIDAL_CURRENT_SOURCE_LOAD (eg., enable it).
 //		- Set RAIL_TO_RAIL_CMD_CTRL to "1" below.
 //		- Run simulation to completion (.2 sec).
 //     - Plot "MLvl_vs_a" and "ia" and label it as Figure 27
 //		- Rename "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_with_sig.dat"
 //		- Set RAIL_TO_RAIL_CMD_CTRL to "0" below.
 //     - Comment out SINUSOIDAL_CURRENT_SOURCE_LOAD (eg., disable it).
 //     - Set CONSTANT_CURRENT_SOURCE_LOAD to "0"
 //		- Run simulation to completion (.2 sec).
 //     - Plot "MLvl_vs_a" and "ia" and label it as Figure 28
 //		- Rename "vs_a_output_no_omega.dat" to "vs_a_output_no_omega_with_no_sig.dat"
 //     - Run "plot_figure_29.m" to generate vs_a_diff.dat and plot "Figure 29"
 //     - Run "plot_figure_30_detail.m" to plot "Figure 30"
 //     - Run "plot_vs_a_diff_filtered.m" to plot "Figure "31"
 //
 //

//#define RAIL_TO_RAIL_CMD_CTRL  1
#define RAIL_TO_RAIL_CMD_CTRL  0


     //(always "fast" update for this test)
  if(SignalCirculationUpdateFast == TRUE)
  {

	  if(RailToRailStateRate == RAIL_TO_RAIL_STATE_RATE)
	  {
		  RailToRailStateIndex = (RailToRailStateIndex + 1) & 0x7;
		  RailToRailStateRate = 0;
	  }
	  else
	  {
		  RailToRailStateRate++;
	  }

  }


  	  	  // - Generate reference  (At 20 amps, 125 - .5*20*.001/.0004 = 125 – 25 = 100 zero to peak.
  	  	  //   See "Compensated_Command_Analysis.pdf")
  	  CmdRef = - .5 * sin(2*PI*60*t) * RAIL_TO_RAIL_CMD_CTRL;


  	  // - Based on polarity, initialize RailToRailStateTable[1][4], RailToRailStateTable[3][4], RailToRailStateTable[5][3], RailToRailStateTable[7][3]
  	  //   or	RailToRailStateTable[1][3], RailToRailStateTable[3][3], RailToRailStateTable[5][4], RailToRailStateTable[7][4]

  	  // - Based on magnitude and polarity, scale  RailToRailStateTable[1][1], RailToRailStateTable[3][1], RailToRailStateTable[5][0], RailToRailStateTable[7][0]
  	  //   or  RailToRailStateTable[1][0], RailToRailStateTable[3][0], RailToRailStateTable[5][1], RailToRailStateTable[7][1]

  	  if(CmdRef >= 0)
  	  {
  		RailToRailStateTable[1][4] = 1.0;
  		RailToRailStateTable[3][4] = -1.0;
  		RailToRailStateTable[5][3] = 1.0;
  		RailToRailStateTable[7][3] = -1.0;
  		RailToRailStateTable[1][3] = 0;
  		RailToRailStateTable[3][3] = 0;
  		RailToRailStateTable[5][4] = 0;
  		RailToRailStateTable[7][4] = 0;

  		RailToRailStateTable[1][1] = PWM_GAIN * CmdRef;
  		RailToRailStateTable[3][1] = PWM_GAIN * CmdRef;
  		RailToRailStateTable[5][0] = PWM_GAIN * (CmdRef - 1.0);
  		RailToRailStateTable[7][0] = PWM_GAIN * (CmdRef - 1.0);
  		RailToRailStateTable[1][0] = PWM_GAIN;
  		RailToRailStateTable[3][0] = PWM_GAIN;
  		RailToRailStateTable[5][1] = 0;
  		RailToRailStateTable[7][1] = 0;

  	  }
  	  else
  	  {

    	RailToRailStateTable[1][4] = 0;
    	RailToRailStateTable[3][4] = 0;
    	RailToRailStateTable[5][3] = 0;
    	RailToRailStateTable[7][3] = 0;
    	RailToRailStateTable[1][3] = 1.0;
    	RailToRailStateTable[3][3] = -1.0;
    	RailToRailStateTable[5][4] = 1.0;
    	RailToRailStateTable[7][4] = -1.0;

  		RailToRailStateTable[1][1] = 0;
  		RailToRailStateTable[3][1] = 0;
  		RailToRailStateTable[5][0] = - PWM_GAIN;
  		RailToRailStateTable[7][0] = - PWM_GAIN;
  		RailToRailStateTable[1][0] = PWM_GAIN * (CmdRef + 1.0);
  		RailToRailStateTable[3][0] = PWM_GAIN * (CmdRef + 1.0);
  		RailToRailStateTable[5][1] = PWM_GAIN * CmdRef;
  		RailToRailStateTable[7][1] = PWM_GAIN * CmdRef;

  	  }




	  //Zero level modifications....

	  if(RailToRailStateTable[RailToRailStateIndex][3] != 0)
	  {
		 PwmRef = RailToRailPwmRefFastRamp * RailToRailStateTable[RailToRailStateIndex][3];
	  }
	  else
	  {
		PwmRef = RailToRailPwmRefFast;
	  }

	  if(RailToRailStateTable[RailToRailStateIndex][0] > 0)
	  {
		 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN - RailToRailStateTable[RailToRailStateIndex][0]))
			 SubModCommand_0 = - PWM_GAIN*.25;
		 else
			 SubModCommand_0 = 0;



	   }
	   else if(RailToRailStateTable[RailToRailStateIndex][0] < 0)
	   {
		 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN + RailToRailStateTable[RailToRailStateIndex][0]))
			 SubModCommand_0 = PWM_GAIN*.25;
		 else
			 SubModCommand_0 = 0;

	   }
	   else  //RailToRailStateTable[RailToRailStateIndex][0] == 0
	   {
		  SubModCommand_0 = 0;

	   }

	  //First level modifications...



	if(RailToRailStateTable[RailToRailStateIndex][4] != 0)
	{
		PwmRef = RailToRailPwmRefFastRamp * RailToRailStateTable[RailToRailStateIndex][4];
	}
	else
	{
		PwmRef = RailToRailPwmRefFast;
	}



	if(RailToRailStateTable[RailToRailStateIndex][1] > 0)
	{
		if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN - RailToRailStateTable[RailToRailStateIndex][1]))
			SubModCommand_1 = - PWM_GAIN*.25;
		else
			SubModCommand_1 = 0;



	}
	else if(RailToRailStateTable[RailToRailStateIndex][1] < 0)
	{
		if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN + RailToRailStateTable[RailToRailStateIndex][1]))
			SubModCommand_1 = PWM_GAIN*.25;
		else
			SubModCommand_1 = 0;

	}
	else  //RailToRailStateTable[RailToRailStateIndex][1] == 0
	{
		SubModCommand_1 = 0;

	}

	if(t < .1)
	{
		y = SubModCommand_0 + SubModCommand_1;
	}
	else
	{					//(some kind of offset here, cancel it)
		 y = PWM_GAIN * ((100.0 / 2000.0) * sin(2*PI*60*t) + (20 / 2000.0)) * RAIL_TO_RAIL_CMD_CTRL;
	}




#else


#ifdef MODULATE_STATE_TABLE

  	  if((t >= MODULATE_STATE_TABLE_BEGIN) && (t < MODULATE_STATE_TABLE_END))
  		 memcpy(RailToRailStateTable, Mod_StateTableOn, sizeof(RailToRailStateTable));
  	  else
  		 memcpy(RailToRailStateTable, Mod_StateTableOff, sizeof(RailToRailStateTable));


#endif


  	  	  //Define fast update  for fast "Rail-to-Rail" step mode.
#define RAIL_TO_RAIL_USES_FAST_REF

#ifdef RAIL_TO_RAIL_USES_FAST_REF
  if(SignalCirculationUpdateFast == TRUE)
#else
  if(SignalCirculationUpdate == TRUE)
#endif
  {

	  if(RailToRailStateRate == RAIL_TO_RAIL_STATE_RATE)
	  {
		  RailToRailStateIndex = (RailToRailStateIndex + 1) & 0x7;
		  RailToRailStateRate = 0;
	  }
	  else
	  {
		  RailToRailStateRate++;
	  }

  }


  	  //Signal PwmRef  is this (from "Vm0"):    "PWM_GAIN*(t_mod/HALF_PWM_CYCLE - .5)*PwmRampDir"
  	  //The following simulates an implementation of Space Vector PWM applied (conditionally) on
  	  //each of the 5 levels

      //Zero level modifications....

#ifdef RAIL_TO_RAIL_USES_FAST_REF

 if(RailToRailStateTable[RailToRailStateIndex][3] != 0)
 {
	 PwmRef = RailToRailPwmRefFastRamp * RailToRailStateTable[RailToRailStateIndex][3];
 }
 else
 {
	PwmRef = RailToRailPwmRefFast;
 }
#else
 PwmRef = RailToRailPwmRef;
#endif


 if(RailToRailStateTable[RailToRailStateIndex][0] > 0)
 {
	 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN - RailToRailStateTable[RailToRailStateIndex][0]))
		 SubModCommand_0 = - PWM_GAIN*.25;
	 else
		 SubModCommand_0 = 0;



 }
 else if(RailToRailStateTable[RailToRailStateIndex][0] < 0)
 {
	 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN + RailToRailStateTable[RailToRailStateIndex][0]))
		 SubModCommand_0 = PWM_GAIN*.25;
	 else
		 SubModCommand_0 = 0;

 }
 else  //RailToRailStateTable[RailToRailStateIndex][0] == 0
 {
	  SubModCommand_0 = 0;

 }




  	  //First level modifications...

#ifdef RAIL_TO_RAIL_USES_FAST_REF

  if(RailToRailStateTable[RailToRailStateIndex][4] != 0)
  {
	  PwmRef = RailToRailPwmRefFastRamp * RailToRailStateTable[RailToRailStateIndex][4];
  }
  else
  {
	  PwmRef = RailToRailPwmRefFast;
  }
#else
  PwmRef = RailToRailPwmRef;
#endif


  if(RailToRailStateTable[RailToRailStateIndex][1] > 0)
  {
	 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN - RailToRailStateTable[RailToRailStateIndex][1]))
		 SubModCommand_1 = - PWM_GAIN*.25;
	 else
		 SubModCommand_1 = 0;



  }
  else if(RailToRailStateTable[RailToRailStateIndex][1] < 0)
  {
	 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN + RailToRailStateTable[RailToRailStateIndex][1]))
		 SubModCommand_1 = PWM_GAIN*.25;
	 else
		 SubModCommand_1 = 0;

  }
  else  //RailToRailStateTable[RailToRailStateIndex][1] == 0
  {
	  SubModCommand_1 = 0;

  }


  	  //Second level modifications....


#ifdef RAIL_TO_RAIL_USES_FAST_REF

  if(RailToRailStateTable[RailToRailStateIndex][5] != 0)
  {
	  PwmRef = RailToRailPwmRefFastRamp * RailToRailStateTable[RailToRailStateIndex][5];
  }
  else
  {
	  PwmRef = RailToRailPwmRefFast;
  }
#else
  PwmRef = RailToRailPwmRef;
#endif




  if(RailToRailStateTable[RailToRailStateIndex][2] > 0)
  {
	 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN - RailToRailStateTable[RailToRailStateIndex][2]))
		 SubModCommand_2 = - PWM_GAIN*.25;
	 else
		 SubModCommand_2 = 0;



  }
  else if(RailToRailStateTable[RailToRailStateIndex][2] < 0)
  {
	 if((PwmRef + (PWM_GAIN*.5)) > (PWM_GAIN + RailToRailStateTable[RailToRailStateIndex][2]))
		 SubModCommand_2 = PWM_GAIN*.25;
	 else
		 SubModCommand_2 = 0;

  }
  else  //RailToRailStateTable[RailToRailStateIndex][2] == 0
  {
	  SubModCommand_2 = 0;

  }


  y = SubModCommand_0 + SubModCommand_1 + SubModCommand_2;


#endif  //RAIL_TO_RAIL_AC_TEST



#else


#ifdef ENABLE_OMEGA_CTRL

  	  OmegaCtrl = TRUE;

#endif




#ifdef APPLY_SINEWAVE_VREF_COMMAND
  y = 2 * sin(2*PI*60*t);
#else


  y = 0;


#endif 

#endif
 

}


void Vref::RecordProbes(void)
{
	FastRefRamp.push_back((double) RailToRailPwmRefFastRamp);
	FastRef.push_back((double) RailToRailPwmRefFast);
	SlowRef.push_back((double) RailToRailPwmRef);
	V_Cap1.push_back((double) v11_v12);
	V_Cap2.push_back((double) v21_v23);
	V_Cap3.push_back((double) v31_v34);
	V_Cap4.push_back((double) v41_v44);
	V_Cap5.push_back((double) v51_v56);
	V_Cap6.push_back((double) v61_v67);
	V_Cap7.push_back((double) v71_v78);
	V_Cap8.push_back((double) v81_v88);

}

void Vref::PlotProbes(Gnuplot & SimuPlot, vector<double> & Plot_t, string TagNamesToPlot[])
{
  int i;
  if(TagNamesToPlot[0] == ""){
	SimuPlot.plot_xy(Plot_t, FastRefRamp, "FastRefRamp");
	SimuPlot.plot_xy(Plot_t, FastRef, "FastRef");
	SimuPlot.plot_xy(Plot_t, SlowRef, "SlowRef");
  }
  else{
	for(i = 0; i < 20; i++){
	  if(TagNamesToPlot[i] == "FastRefRamp"){
		  SimuPlot.plot_xy(Plot_t, FastRefRamp, "FastRefRamp");
	  }
	  else if(TagNamesToPlot[i] == "FastRef"){
		  SimuPlot.plot_xy(Plot_t, FastRef, "FastRef");
	  }
	  else if(TagNamesToPlot[i] == "SlowRef"){
		  SimuPlot.plot_xy(Plot_t, SlowRef, "SlowRef");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap1"){
		  SimuPlot.plot_xy(Plot_t, V_Cap1, "V_Cap1");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap2"){
		  SimuPlot.plot_xy(Plot_t, V_Cap2, "V_Cap2");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap3"){
		  SimuPlot.plot_xy(Plot_t, V_Cap3, "V_Cap3");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap4"){
		  SimuPlot.plot_xy(Plot_t, V_Cap4, "V_Cap4");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap5"){
		  SimuPlot.plot_xy(Plot_t, V_Cap5, "V_Cap5");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap6"){
		  SimuPlot.plot_xy(Plot_t, V_Cap6, "V_Cap6");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap7"){
		  SimuPlot.plot_xy(Plot_t, V_Cap7, "V_Cap7");
	  }
	  else if(TagNamesToPlot[i] == "V_Cap8"){
		  SimuPlot.plot_xy(Plot_t, V_Cap8, "V_Cap8");
	  }
	  else if(TagNamesToPlot[i] == ""){
		  break;
	  }
	}
  }

}



void Vref::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

#ifndef ENABLE_CELL_CIRCULATION
  ((ECtrl1 *) pCurSrcItem->pSrcObject)->V_112 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl2 *) pCurSrcItem->pSrcObject)->V_112 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl3 *) pCurSrcItem->pSrcObject)->V_112 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl4 *) pCurSrcItem->pSrcObject)->V_112 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl5 *) pCurSrcItem->pSrcObject)->V_112 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl6 *) pCurSrcItem->pSrcObject)->V_112 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl7 *) pCurSrcItem->pSrcObject)->V_112 = y;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;    
  ((ECtrl8 *) pCurSrcItem->pSrcObject)->V_112 = y;
#ifdef ENABLE_OMEGA_CTRL
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((Vm1 *) pCurSrcItem->pSrcObject)->OmegaCtrl = OmegaCtrl;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((Vm2 *) pCurSrcItem->pSrcObject)->OmegaCtrl = OmegaCtrl;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((Vm3 *) pCurSrcItem->pSrcObject)->OmegaCtrl = OmegaCtrl;
#endif

#else

  ((ECtrl1 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl1 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][0];
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl2 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl2 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][1];
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl3 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl3 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][2];
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl4 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl4 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][3];
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl5 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl5 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][4];
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl6 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl6 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][5];
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl7 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl7 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][6];
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((ECtrl8 *) pCurSrcItem->pSrcObject)->V_112 = y;
  ((ECtrl8 *) pCurSrcItem->pSrcObject)->CircState = CircStateTable[CircStateIndex][7];
#ifdef ENABLE_OMEGA_CTRL
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((Vm1 *) pCurSrcItem->pSrcObject)->OmegaCtrl = OmegaCtrl;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((Vm2 *) pCurSrcItem->pSrcObject)->OmegaCtrl = OmegaCtrl;
  pCurSrcItem = pCurSrcItem->pNextSrcItem;
  ((Vm3 *) pCurSrcItem->pSrcObject)->OmegaCtrl = OmegaCtrl;
#endif

#endif

}




// -----------------------------------------------------------------

// ******************************************************************





// **** OdeObject Functions ****************************************



// ---- qa --------------------------------------------------------

qa::qa(void)
{
  
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_qa;
  LiteralName = "qa";

  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;
 
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_ia_ib_1];

  ia = 0;

#ifdef PLOT_ODEOBJECT_qa
  PlotThisOutput = TRUE;
#endif

  Plot_Tag = "qa";

}

qa::~qa(void)
{

}

double qa::OdeFunction(double y, double t)
{

#ifdef INDUCTIVE_COUPLED_LOAD
     //ia
  return(ia);
#else
  return(0);
#endif

}

void qa::OdeRValueUpdate(void)
{
#ifdef INDUCTIVE_COUPLED_LOAD

  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((ia_ib_1 *) pCurOdeItem->pOdeObject)->qa = y;

#endif
}

// -----------------------------------------------------------------


// ---- qb --------------------------------------------------------

qb::qb(void)
{
  
  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_qb;
  LiteralName = "qb";

  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;
 
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_ia_ib_2];

  ib = 0;

#ifdef PLOT_ODEOBJECT_qb
  PlotThisOutput = TRUE;
#endif

  Plot_Tag = "qb";

}

qb::~qb(void)
{

}

double qb::OdeFunction(double y, double t)
{
#ifdef INDUCTIVE_COUPLED_LOAD
     //ib
  return(ib);
#else
  return(0);
#endif

}

void qb::OdeRValueUpdate(void)
{
#ifdef INDUCTIVE_COUPLED_LOAD

  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((ia_ib_2 *) pCurOdeItem->pOdeObject)->qb = y;

#endif

}

// -----------------------------------------------------------------

// ---- ia_ib_1 --------------------------------------------------

ia_ib_1::ia_ib_1(void)
{

  CoefObjItem * pCurCoefItem;
  OdeObjItem * pCurOdeItem;
  SrcObjItem * pCurSrcItem;

  OdeFuncName = ODE_FUNC_ia_ib_1;
  LiteralName = "ia_ib_1";

#ifdef INDUCTIVE_COUPLED_LOAD

  //this object marks the beginning of a "group solve" of
  //two objects.
  GroupSolve = 1;
  NumberOfGrpOdes = 2;

#endif

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;
  pCurCoefItem = pCoefObjRValList;

  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_ia];

  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_qa];

  //build the SRC Rvalue list.

  pSrcObjRValList = new SrcObjItem;
  pCurSrcItem = pSrcObjRValList;

  pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];


  qa = 0;
  //(ia is represented by "y")
  vs_a = 0;

#ifdef PLOT_ODEOBJECT_ia_ib_1
  PlotThisOutput = TRUE;
#endif

  Plot_Tag = "ia";


}


ia_ib_1::~ia_ib_1(void)
{

}

double ia_ib_1::OdeFunction(double y, double t)
{
#ifdef INDUCTIVE_COUPLED_LOAD
  //Note: This is within a "group" ODE so we are really returning for "dm/dt"

  // vs_a - Ra*ia - qa/Ca
  return(vs_a - Ra*y - qa/Ca);
#else
  return(0);
#endif

}

void ia_ib_1::PostOdeFunction(double t)
{

#ifndef INDUCTIVE_COUPLED_LOAD

#ifndef SINUSOIDAL_CURRENT_SOURCE_LOAD
  //(Assign the value for "ia" directly)
  y = CONSTANT_CURRENT_SOURCE_LOAD;
#else


  y = - CONSTANT_CURRENT_SOURCE_LOAD * sin(2*PI*60*t - (PI/12.0));

#endif

#endif


}

void ia_ib_1::OdeGroupMSolve(double dydt[], double dmdt[])
{
  //Note: This instance is the first of the group. As such, this
  //      is where the "group solve" of the individual
  //      "dy/dt's" is performed.
  //
  //
  //      Reference OLE equations (i) and (j) in the description at the beginning of this file.
  //
  //
  //
  //      Let
  //         dia = dia/dt and dib = dib/dt  //
  //
  //
  //
  //             (Lm_a + L_a)*dia + Mab*dib = dmdt[0]                                                      (1)
  //
  //             Mba*dia + (Lm_b + L_b)*dib = dmdt[1]                                                      (2)
  //
  //      Solve for dia in (1) and (2) and subtract (2) from (1)
  //
  //             dia = dmdt[0]/(Lm_a + L_a) - Mab*dib/(Lm_a + L_a)
  //
  //             dia = dmdt[1]/Mba - (Lm_b + L_b)*dib/Mba
  //
  //             dmdt[0]/(Lm_a + L_a) - Mab*dib/(Lm_a + L_a) = dmdt[1]/Mba - (Lm_b + L_b)*dib/Mba          (3)
  //
  //      Solve for dib
  //
  //             dib((Lm_b + L_b)/Mba - Mab/(Lm_a + L_a)) = dmdt[1]/Mba - dmdt[0]/(Lm_a + L_a)
  //
  //
  //      Let
  //
  //            B_coef = (Lm_b + L_b)
  //            C_coef = (Lm_a + L_a)
  //
  //
  //            dib = (dmdt[1]/Mba - dmdt[0]/C_coef)/(B_coef/Mba - Mab/C_coef)                            (4)
  //
  //
  //      Solve for diLp using (4) on (1)
  //
  //
  //           dia  = (dmdt[0] - Mab*dib)/(Lm_a + L_a)
  //
  //           dia  = (dmdt[0] - Mab*(dmdt[1]/Mba - dmdt[0]/C_coef)/(B_coef/Mba - Mab/C_coef))/C_coef     (5)
  //
  //
  //
  //

#define B_coef (Lm_b + L_b)
#define C_coef (Lm_a + L_a)

  double dia;
  double dib;

  dib = (dmdt[1]/Mba - dmdt[0]/C_coef)/(B_coef/Mba - Mab/C_coef);
  dia = (dmdt[0] - Mab*(dmdt[1]/Mba - dmdt[0]/C_coef)/(B_coef/Mba - Mab/C_coef))/C_coef;


  dydt[0] = dia;
  dydt[1] = dib;


}

void ia_ib_1::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_ia *) pCurCoefItem->pCoefObject)->ia = y;

}

void ia_ib_1::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((class qa *) pCurOdeItem->pOdeObject)->ia = y;

}

void ia_ib_1::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;

  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->ia = y;

}


// -----------------------------------------------------------------

// ---- ia_ib_2 --------------------------------------------------

ia_ib_2::ia_ib_2(void)
{

  OdeObjItem * pCurOdeItem;

  OdeFuncName = ODE_FUNC_ia_ib_2;
  LiteralName = "ia_ib_2";


  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;
  pCurOdeItem = pOdeObjRValList;

  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_qb];

  qb = 0;
 //(ib is represented by "y")

#ifdef PLOT_ODEOBJECT_ia_ib_2
  PlotThisOutput = TRUE;
#endif

  Plot_Tag = "ib";


}

ia_ib_2::~ia_ib_2(void)
{

}

double ia_ib_2::OdeFunction(double y, double t)
{

#ifdef INDUCTIVE_COUPLED_LOAD
  //Note: This is within a "group" ODE so we are really returning for "dm/dt"

  // - Rb*ib - qb/Cb
  return(- Rb*y - qb/Cb);
#else
  return(0);
#endif

}

void ia_ib_2::OdeRValueUpdate(void)
{
  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((class qb *) pCurOdeItem->pOdeObject)->ib = y;

}

// -----------------------------------------------------------------



// ******************************************************************


// **** CtrlObject Functions ****************************************

// ******************************************************************



// **** CoefObject Functions ****************************************

// ---- Coef_recp_rSrD11 ---------------------------------------------

Coef_recp_rSrD11::Coef_recp_rSrD11(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD11;
  LiteralName = "recp_rSrD11"; 
  rS11 = INIT_COEF_RS_RD;
  rD11 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD11::~Coef_recp_rSrD11(void)
{


}

void Coef_recp_rSrD11::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS11 + 1/rD11;

}



// ------------------------------------------------------------------

// ---- Coef_recp_rSrD12 ---------------------------------------------

Coef_recp_rSrD12::Coef_recp_rSrD12(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD12;
  LiteralName = "recp_rSrD12"; 
  rS12 = INIT_COEF_RS_RD;
  rD12 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD12::~Coef_recp_rSrD12(void)
{


}

void Coef_recp_rSrD12::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS12 + 1/rD12;

}



// ------------------------------------------------------------------        


// ---- Coef_C1_h ----------------------------------------------------

Coef_C1_h::Coef_C1_h(void)
{
  CoefFuncName = COEF_FUNC_C1_h;
  LiteralName = "C1_h";

}

Coef_C1_h::~Coef_C1_h(void)
{


}

void Coef_C1_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C1 * recp_h;
	else
		y = C1 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C11_h ----------------------------------------------------

Coef_C11_h::Coef_C11_h(void)
{
  CoefFuncName = COEF_FUNC_C11_h;
  LiteralName = "C11_h";

}

Coef_C11_h::~Coef_C11_h(void)
{


}

void Coef_C11_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C11 * recp_h;
	else
		y = C11 * recp_h;

}

// ------------------------------------------------------------------

// ---- Coef_C12_h ----------------------------------------------------

Coef_C12_h::Coef_C12_h(void)
{
  CoefFuncName = COEF_FUNC_C12_h;
  LiteralName = "C12_h";

}

Coef_C12_h::~Coef_C12_h(void)
{


}

void Coef_C12_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C12 * recp_h;
	else
		y = C12 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v11_v12_h_prev_C1 ----------------------------------------------------

Coef_v11_v12_h_prev_C1::Coef_v11_v12_h_prev_C1(void)
{
  CoefFuncName = COEF_FUNC_v11_v12_h_prev_C1;
  LiteralName = "v11_v12_h_prev_C1";

	//Assign initial states...
  v11_ = (6.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+1500)
  v12_ = (6.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+1000)

}

Coef_v11_v12_h_prev_C1::~Coef_v11_v12_h_prev_C1(void)
{


}

void Coef_v11_v12_h_prev_C1::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v11_ - v12_) * C1 * recp_h - i1_;
	else
		y = (v11_ - v12_) * C1 * recp_h;

}

// ------------------------------------------------------------------    


// ---- Coef_minus_v11_h_prev_minus_PlusBus_C11 ----------------------------------------------------

Coef_minus_v11_h_prev_minus_PlusBus_C11::Coef_minus_v11_h_prev_minus_PlusBus_C11(void)
{
  CoefFuncName = COEF_FUNC_minus_v11_h_prev_minus_PlusBus_C11;
  LiteralName = "minus_v11_h_prev_minus_PlusBus_C11";

	//Assign initial states...
  v11_ = (6.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+1500)

}

Coef_minus_v11_h_prev_minus_PlusBus_C11::~Coef_minus_v11_h_prev_minus_PlusBus_C11(void)
{



}

void Coef_minus_v11_h_prev_minus_PlusBus_C11::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = - 2 * v11_ * C11 * recp_h - i1ct_;
	else
		y = - v11_ * C11 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v12_h_prev_plus_PlusBus_C12 ----------------------------------------------------

Coef_v12_h_prev_plus_PlusBus_C12::Coef_v12_h_prev_plus_PlusBus_C12(void)
{
  CoefFuncName = COEF_FUNC_v12_h_prev_plus_PlusBus_C12;
  LiteralName = "v12_h_prev_plus_PlusBus_C12";

	//Assign initial states...
  v12_ = (6.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+1000)

}

Coef_v12_h_prev_plus_PlusBus_C12::~Coef_v12_h_prev_plus_PlusBus_C12(void)
{

}

void Coef_v12_h_prev_plus_PlusBus_C12::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y =  2 * v12_ * C12 * recp_h - i1cb_;
	else
		y =  v12_ * C12 * recp_h;

}

// ------------------------------------------------------------------





// ---- Coef_recp_rSrD21 ---------------------------------------------

Coef_recp_rSrD21::Coef_recp_rSrD21(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD21;
  LiteralName = "recp_rSrD21"; 
  rS21 = INIT_COEF_RS_RD;
  rD21 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD21::~Coef_recp_rSrD21(void)
{


}

void Coef_recp_rSrD21::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS21 + 1/rD21;

}



// ------------------------------------------------------------------

// ---- Coef_recp_rSrD22 ---------------------------------------------

Coef_recp_rSrD22::Coef_recp_rSrD22(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD22;
  LiteralName = "recp_rSrD22"; 
  rS22 = INIT_COEF_RS_RD;
  rD22 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD22::~Coef_recp_rSrD22(void)
{


}

void Coef_recp_rSrD22::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS22 + 1/rD22;

}

// ------------------------------------------------------------------  




// ---- Coef_C2_h ----------------------------------------------------

Coef_C2_h::Coef_C2_h(void)
{
  CoefFuncName = COEF_FUNC_C2_h;
  LiteralName = "C2_h";

}

Coef_C2_h::~Coef_C2_h(void)
{


}

void Coef_C2_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C2 * recp_h;
	else
		y = C2 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C21_h ----------------------------------------------------

Coef_C21_h::Coef_C21_h(void)
{
  CoefFuncName = COEF_FUNC_C21_h;
  LiteralName = "C21_h";

}

Coef_C21_h::~Coef_C21_h(void)
{


}

void Coef_C21_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C21 * recp_h;
	else
		y = C21 * recp_h;

}

// ------------------------------------------------------------------

// ---- Coef_C22_h ----------------------------------------------------

Coef_C22_h::Coef_C22_h(void)
{
  CoefFuncName = COEF_FUNC_C22_h;
  LiteralName = "C22_h";

}

Coef_C22_h::~Coef_C22_h(void)
{


}

void Coef_C22_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C22 * recp_h;
	else
		y = C22 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v21_v23_h_prev_C2 ----------------------------------------------------

Coef_v21_v23_h_prev_C2::Coef_v21_v23_h_prev_C2(void)
{
  CoefFuncName = COEF_FUNC_v21_v23_h_prev_C2;
  LiteralName = "v21_v23_h_prev_C2";

	//Assign initial states...
  v21_ = (5.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+1166.66)
  v23_ = (5.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+666.66)

}

Coef_v21_v23_h_prev_C2::~Coef_v21_v23_h_prev_C2(void)
{



}

void Coef_v21_v23_h_prev_C2::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v21_ - v23_) * C2 * recp_h - i2_;
	else
		y = (v21_ - v23_) * C2 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v12_v21_h_prev_C21 ----------------------------------------------------

Coef_v12_v21_h_prev_C21::Coef_v12_v21_h_prev_C21(void)
{
  CoefFuncName = COEF_FUNC_v12_v21_h_prev_C21;
  LiteralName = "v12_v21_h_prev_C21";

	//Assign initial states...
  v12_ = (6.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+1000)
  v21_ = (5.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+1166.66)

}

Coef_v12_v21_h_prev_C21::~Coef_v12_v21_h_prev_C21(void)
{


}

void Coef_v12_v21_h_prev_C21::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v12_ - v21_) * C21 * recp_h - i2ct_;
	else
		y = (v12_ - v21_) * C21 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v23_v12_h_prev_C22 ----------------------------------------------------

Coef_v23_v12_h_prev_C22::Coef_v23_v12_h_prev_C22(void)
{
  CoefFuncName = COEF_FUNC_v23_v12_h_prev_C22;
  LiteralName = "v23_v12_h_prev_C22";

	//Assign initial states...
  v12_ = (6.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+1000)
  v23_ = (5.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+666.66)


}

Coef_v23_v12_h_prev_C22::~Coef_v23_v12_h_prev_C22(void)
{

}

void Coef_v23_v12_h_prev_C22::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v23_ - v12_) * C22 * recp_h - i2cb_;
	else
		y = (v23_ - v12_) * C22 * recp_h;

}

// ------------------------------------------------------------------    



// ---- Coef_recp_rSrD31 ---------------------------------------------

Coef_recp_rSrD31::Coef_recp_rSrD31(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD31;
  LiteralName = "recp_rSrD31";
  rS31 = INIT_COEF_RS_RD;
  rD31 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD31::~Coef_recp_rSrD31(void)
{


}

void Coef_recp_rSrD31::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS31 + 1/rD31;

}

// ------------------------------------------------------------------

// ---- Coef_recp_rSrD32 ---------------------------------------------

Coef_recp_rSrD32::Coef_recp_rSrD32(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD32;
  LiteralName = "recp_rSrD32";
  rS32 = INIT_COEF_RS_RD;
  rD32 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD32::~Coef_recp_rSrD32(void)
{


}

void Coef_recp_rSrD32::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS32 + 1/rD32;

}
// ------------------------------------------------------------------




// ---- Coef_C3_h ----------------------------------------------------

Coef_C3_h::Coef_C3_h(void)
{
  CoefFuncName = COEF_FUNC_C3_h;
  LiteralName = "C3_h";

}

Coef_C3_h::~Coef_C3_h(void)
{


}

void Coef_C3_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C3 *  recp_h;
	else
		y = C3 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C31_h ----------------------------------------------------

Coef_C31_h::Coef_C31_h(void)
{
  CoefFuncName = COEF_FUNC_C31_h;
  LiteralName = "C31_h";

}

Coef_C31_h::~Coef_C31_h(void)
{


}

void Coef_C31_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C31 * recp_h;
	else
		y = C31 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C32_h ----------------------------------------------------

Coef_C32_h::Coef_C32_h(void)
{
  CoefFuncName = COEF_FUNC_C32_h;
  LiteralName = "C32_h";

}

Coef_C32_h::~Coef_C32_h(void)
{


}

void Coef_C32_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C32 * recp_h;
	else
		y = C32 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v31_v34_h_prev_C3 ----------------------------------------------------

Coef_v31_v34_h_prev_C3::Coef_v31_v34_h_prev_C3(void)
{
  CoefFuncName = COEF_FUNC_v31_v34_h_prev_C3;
  LiteralName = "v31_v34_h_prev_C3";

	//Assign initial states...
  v31_ = (4.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+833.33)
  v34_ = (4.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+333.33)

}

Coef_v31_v34_h_prev_C3::~Coef_v31_v34_h_prev_C3(void)
{

}

void Coef_v31_v34_h_prev_C3::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v31_ - v34_) * C3 * recp_h - i3_;
	else
		y = (v31_ - v34_) * C3 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v23_v31_h_prev_C31 ----------------------------------------------------

Coef_v23_v31_h_prev_C31::Coef_v23_v31_h_prev_C31(void)
{
  CoefFuncName = COEF_FUNC_v23_v31_h_prev_C31;
  LiteralName = "v23_v31_h_prev_C31";

	//Assign initial states...
  v31_ = (4.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+833.33)
  v23_ = (5.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+666.66)

}

Coef_v23_v31_h_prev_C31::~Coef_v23_v31_h_prev_C31(void)
{

}

void Coef_v23_v31_h_prev_C31::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v23_ - v31_) * C31 * recp_h - i3ct_;
	else
		y = (v23_ - v31_) * C31 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v34_v23_h_prev_C32 ----------------------------------------------------

Coef_v34_v23_h_prev_C32::Coef_v34_v23_h_prev_C32(void)
{
  CoefFuncName = COEF_FUNC_v34_v23_h_prev_C32;
  LiteralName = "v34_v23_h_prev_C32";

	//Assign initial states...
  v23_ = (5.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+666.66)
  v34_ = (4.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+333.33)

}

Coef_v34_v23_h_prev_C32::~Coef_v34_v23_h_prev_C32(void)
{

}

void Coef_v34_v23_h_prev_C32::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v34_ - v23_) * C32 * recp_h - i3cb_;
	else
		y = (v34_ - v23_) * C32 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_recp_rSrD41 ---------------------------------------------

Coef_recp_rSrD41::Coef_recp_rSrD41(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD41;
  LiteralName = "recp_rSrD41"; 
  rS41 = INIT_COEF_RS_RD;
  rD41 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD41::~Coef_recp_rSrD41(void)
{


}

void Coef_recp_rSrD41::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS41 + 1/rD41;

}

// ------------------------------------------------------------------

// ---- Coef_recp_rSrD42 ---------------------------------------------

Coef_recp_rSrD42::Coef_recp_rSrD42(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD42;
  LiteralName = "recp_rSrD42"; 
  rS42 = INIT_COEF_RS_RD;
  rD42 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD42::~Coef_recp_rSrD42(void)
{


}

void Coef_recp_rSrD42::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS42 + 1/rD42;

}

// ------------------------------------------------------------------  




// ---- Coef_C4_h ----------------------------------------------------

Coef_C4_h::Coef_C4_h(void)
{
  CoefFuncName = COEF_FUNC_C4_h;
  LiteralName = "C4_h";

}

Coef_C4_h::~Coef_C4_h(void)
{


}

void Coef_C4_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C4 * recp_h;
	else
		y = C4 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C41_h ----------------------------------------------------

Coef_C41_h::Coef_C41_h(void)
{
  CoefFuncName = COEF_FUNC_C41_h;
  LiteralName = "C41_h";

}

Coef_C41_h::~Coef_C41_h(void)
{


}

void Coef_C41_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C41 * recp_h;
	else
		y = C41 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C42_h ----------------------------------------------------

Coef_C42_h::Coef_C42_h(void)
{
  CoefFuncName = COEF_FUNC_C42_h;
  LiteralName = "C42_h";

}

Coef_C42_h::~Coef_C42_h(void)
{


}

void Coef_C42_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C42 * recp_h;
	else
		y = C42 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v41_v44_h_prev_C4 ----------------------------------------------------

Coef_v41_v44_h_prev_C4::Coef_v41_v44_h_prev_C4(void)
{
  CoefFuncName = COEF_FUNC_v41_v44_h_prev_C4;
  LiteralName = "v41_v44_h_prev_C4";

	//Assign initial states...
  v41_ = (3.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+500.00)
  v44_ = (3.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(0.0)

}

Coef_v41_v44_h_prev_C4::~Coef_v41_v44_h_prev_C4(void)
{


}

void Coef_v41_v44_h_prev_C4::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v41_ - v44_) * C4 * recp_h - i4_;
	else
		y = (v41_ - v44_) * C4 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v34_v41_h_prev_C41 ----------------------------------------------------

Coef_v34_v41_h_prev_C41::Coef_v34_v41_h_prev_C41(void)
{
  CoefFuncName = COEF_FUNC_v34_v41_h_prev_C41;
  LiteralName = "v34_v41_h_prev_C41";

	//Assign initial states...
  v41_ = (3.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+500.00)
  v34_ = (4.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+333.33)

}

Coef_v34_v41_h_prev_C41::~Coef_v34_v41_h_prev_C41(void)
{

}

void Coef_v34_v41_h_prev_C41::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v34_ - v41_) * C41 * recp_h - i4ct_;
	else
		y = (v34_ - v41_) * C41 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v44_v34_h_prev_C42 ----------------------------------------------------

Coef_v44_v34_h_prev_C42::Coef_v44_v34_h_prev_C42(void)
{
  CoefFuncName = COEF_FUNC_v44_v34_h_prev_C42;
  LiteralName = "v44_v34_h_prev_C42";

	//Assign initial states...
  v34_ = (4.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(+333.33)
  v44_ = (3.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(0.0)



}

Coef_v44_v34_h_prev_C42::~Coef_v44_v34_h_prev_C42(void)
{

}

void Coef_v44_v34_h_prev_C42::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v44_ - v34_) * C42 * recp_h - i4cb_;
	else
		y = (v44_ - v34_) * C42 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_Lp_h ---------------------------------------------------


Coef_Lp_h::Coef_Lp_h(void)
{
	CoefFuncName = COEF_FUNC_Lp_h;
	LiteralName = "Lp_h";
}

Coef_Lp_h::~Coef_Lp_h(void)
{



}

void Coef_Lp_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * Lp * recp_h;
	else
		y = Lp * recp_h;

}


// ------------------------------------------------------------------

// ---- Coef_Mpn_h ---------------------------------------------------

Coef_Mpn_h::Coef_Mpn_h(void)
{
	CoefFuncName = COEF_FUNC_Mpn_h;
	LiteralName = "Mpn_h";
}

Coef_Mpn_h::~Coef_Mpn_h(void)
{



}

void Coef_Mpn_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * Mpn * recp_h;
	else
		y = Mpn * recp_h;

}


// ------------------------------------------------------------------


// ---- Coef_Mnp_h ---------------------------------------------------

Coef_Mnp_h::Coef_Mnp_h(void)
{
	CoefFuncName = COEF_FUNC_Mnp_h;
	LiteralName = "Mnp_h";
}

Coef_Mnp_h::~Coef_Mnp_h(void)
{



}

void Coef_Mnp_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * Mnp * recp_h;
	else
		y = Mnp * recp_h;

}


// ------------------------------------------------------------------

// ---- Coef_Ln_h ---------------------------------------------------


Coef_Ln_h::Coef_Ln_h(void)
{
	CoefFuncName = COEF_FUNC_Ln_h;
	LiteralName = "Ln_h";
}

Coef_Ln_h::~Coef_Ln_h(void)
{



}

void Coef_Ln_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * Ln * recp_h;
	else
		y = Ln * recp_h;

}


// ------------------------------------------------------------------




// ---- Coef_iLp_iLn_h_prev_Lp_Mpn ----------------------------------

Coef_iLp_iLn_h_prev_Lp_Mpn::Coef_iLp_iLn_h_prev_Lp_Mpn(void)
{
	CoefFuncName = COEF_FUNC_iLp_iLn_h_prev_Lp_Mpn;
	LiteralName = "iLp_iLn_h_prev_Lp_Mpn";


}

Coef_iLp_iLn_h_prev_Lp_Mpn::~Coef_iLp_iLn_h_prev_Lp_Mpn(void)
{


}

void Coef_iLp_iLn_h_prev_Lp_Mpn::CoefFunction(double recp_h, bool DoTrapezoidal)
{
	if(DoTrapezoidal)
		y = 2 * (Lp*iLp_*recp_h + Mpn*iLn_*recp_h) + v44_ - vs_a_;
	else
		y = Lp*iLp_*recp_h + Mpn*iLn_*recp_h;

}


// ------------------------------------------------------------------





// ---- Coef_iLp_iLn_h_prev_Mnp_Ln ----------------------------------

Coef_iLp_iLn_h_prev_Mnp_Ln::Coef_iLp_iLn_h_prev_Mnp_Ln(void)
{
	CoefFuncName = COEF_FUNC_iLp_iLn_h_prev_Mnp_Ln;
	LiteralName = "iLp_iLn_h_prev_Mnp_Ln";

}

Coef_iLp_iLn_h_prev_Mnp_Ln::~Coef_iLp_iLn_h_prev_Mnp_Ln(void)
{


}

void Coef_iLp_iLn_h_prev_Mnp_Ln::CoefFunction(double recp_h, bool DoTrapezoidal)
{
	if(DoTrapezoidal)
		y = 2 * (Mnp*iLp_*recp_h + Ln*iLn_*recp_h) + vs_a_ - v45_;
	else
		y = Mnp*iLp_*recp_h +  Ln*iLn_*recp_h;

}


// ------------------------------------------------------------------




// ---- Coef_ia ----------------------------------------------------

Coef_ia::Coef_ia(void)
{
  CoefFuncName = COEF_FUNC_ia;
  LiteralName = "ia"; 

}

Coef_ia::~Coef_ia(void)
{


}

void Coef_ia::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = ia;

}

// ------------------------------------------------------------------    




// ---- Coef_recp_rSrD51 ---------------------------------------------

Coef_recp_rSrD51::Coef_recp_rSrD51(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD51;
  LiteralName = "recp_rSrD51"; 
  rS51 = INIT_COEF_RS_RD;
  rD51 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD51::~Coef_recp_rSrD51(void)
{


}

void Coef_recp_rSrD51::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS51 + 1/rD51;

}

// ------------------------------------------------------------------

// ---- Coef_recp_rSrD52 ---------------------------------------------

Coef_recp_rSrD52::Coef_recp_rSrD52(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD52;
  LiteralName = "recp_rSrD52"; 
  rS52 = INIT_COEF_RS_RD;
  rD52 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD52::~Coef_recp_rSrD52(void)
{


}

void Coef_recp_rSrD52::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS52 + 1/rD52;

}

// ------------------------------------------------------------------  




// ---- Coef_C5_h ----------------------------------------------------

Coef_C5_h::Coef_C5_h(void)
{
  CoefFuncName = COEF_FUNC_C5_h;
  LiteralName = "C5_h";

}

Coef_C5_h::~Coef_C5_h(void)
{


}

void Coef_C5_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C5 * recp_h;
	else
		y = C5 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C51_h ----------------------------------------------------

Coef_C51_h::Coef_C51_h(void)
{
  CoefFuncName = COEF_FUNC_C51_h;
  LiteralName = "C51_h";

}

Coef_C51_h::~Coef_C51_h(void)
{


}

void Coef_C51_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C51 * recp_h;
	else
		y = C51 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C52_h ----------------------------------------------------

Coef_C52_h::Coef_C52_h(void)
{
  CoefFuncName = COEF_FUNC_C52_h;
  LiteralName = "C52_h";

}

Coef_C52_h::~Coef_C52_h(void)
{


}

void Coef_C52_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C52 * recp_h;
	else
		y = C52 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v51_v56_h_prev_C5 ----------------------------------------------------

Coef_v51_v56_h_prev_C5::Coef_v51_v56_h_prev_C5(void)
{
  CoefFuncName = COEF_FUNC_v51_v56_h_prev_C5;
  LiteralName = "v51_v56_h_prev_C5";

	//Assign initial states...
  v51_ = (2.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+166.66)
  v56_ = (2.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-333.33)

}

Coef_v51_v56_h_prev_C5::~Coef_v51_v56_h_prev_C5(void)
{


}

void Coef_v51_v56_h_prev_C5::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v51_ - v56_) * C5 * recp_h - i5_;
	else
		y = (v51_ - v56_) * C5 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v45_v51_h_prev_C51 ----------------------------------------------------

Coef_v45_v51_h_prev_C51::Coef_v45_v51_h_prev_C51(void)
{
  CoefFuncName = COEF_FUNC_v45_v51_h_prev_C51;
  LiteralName = "v45_v51_h_prev_C51";

	//Assign initial states...
  v51_ = (2.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(+166.66)
  v45_ = (3.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(0.0)

}

Coef_v45_v51_h_prev_C51::~Coef_v45_v51_h_prev_C51(void)
{

}

void Coef_v45_v51_h_prev_C51::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v45_ - v51_) * C51 * recp_h - i5ct_;
	else
		y = (v45_ - v51_) * C51 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v56_v45_h_prev_C52 ----------------------------------------------------

Coef_v56_v45_h_prev_C52::Coef_v56_v45_h_prev_C52(void)
{
  CoefFuncName = COEF_FUNC_v56_v45_h_prev_C52;
  LiteralName = "v56_v45_h_prev_C52";

	//Assign initial states...
  v56_ = (2.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-333.33)
  v45_ = (3.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(0.0)

}

Coef_v56_v45_h_prev_C52::~Coef_v56_v45_h_prev_C52(void)
{


}

void Coef_v56_v45_h_prev_C52::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v56_ - v45_) * C52 * recp_h - i5cb_;
	else
		y = (v56_ - v45_) * C52 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_recp_rSrD61 ---------------------------------------------

Coef_recp_rSrD61::Coef_recp_rSrD61(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD61;
  LiteralName = "recp_rSrD61"; 
  rS61 = INIT_COEF_RS_RD;
  rD61 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD61::~Coef_recp_rSrD61(void)
{


}

void Coef_recp_rSrD61::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS61 + 1/rD61;

}

// ------------------------------------------------------------------

// ---- Coef_recp_rSrD62 ---------------------------------------------

Coef_recp_rSrD62::Coef_recp_rSrD62(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD62;
  LiteralName = "recp_rSrD62"; 
  rS62 = INIT_COEF_RS_RD;
  rD62 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD62::~Coef_recp_rSrD62(void)
{


}

void Coef_recp_rSrD62::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS62 + 1/rD62;

}

// ------------------------------------------------------------------ 



// ---- Coef_C6_h ----------------------------------------------------

Coef_C6_h::Coef_C6_h(void)
{
  CoefFuncName = COEF_FUNC_C6_h;
  LiteralName = "C6_h";

}

Coef_C6_h::~Coef_C6_h(void)
{


}

void Coef_C6_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C6 * recp_h;
	else
		y = C6 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C61_h ----------------------------------------------------

Coef_C61_h::Coef_C61_h(void)
{
  CoefFuncName = COEF_FUNC_C61_h;
  LiteralName = "C61_h";

}

Coef_C61_h::~Coef_C61_h(void)
{


}

void Coef_C61_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C61 * recp_h;
	else
		y = C61 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C62_h ----------------------------------------------------

Coef_C62_h::Coef_C62_h(void)
{
  CoefFuncName = COEF_FUNC_C62_h;
  LiteralName = "C62_h";

}

Coef_C62_h::~Coef_C62_h(void)
{


}

void Coef_C62_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C62 * recp_h;
	else
		y = C62 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v61_v67_h_prev_C6 ----------------------------------------------------

Coef_v61_v67_h_prev_C6::Coef_v61_v67_h_prev_C6(void)
{
  CoefFuncName = COEF_FUNC_v61_v67_h_prev_C6;
  LiteralName = "v61_v67_h_prev_C6";

	//Assign initial states...
  v61_ = (1.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(-166.66)
  v67_ = (1.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-666.66)

}

Coef_v61_v67_h_prev_C6::~Coef_v61_v67_h_prev_C6(void)
{


}

void Coef_v61_v67_h_prev_C6::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v61_ - v67_) * C6 * recp_h - i6_;
	else
		y = (v61_ - v67_) * C6 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v56_v61_h_prev_C61 ----------------------------------------------------

Coef_v56_v61_h_prev_C61::Coef_v56_v61_h_prev_C61(void)
{
  CoefFuncName = COEF_FUNC_v56_v61_h_prev_C61;
  LiteralName = "v56_v61_h_prev_C61";

	//Assign initial states...
  v61_ = (1.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(-166.66)
  v56_ = (2.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-333.33)

}

Coef_v56_v61_h_prev_C61::~Coef_v56_v61_h_prev_C61(void)
{

}

void Coef_v56_v61_h_prev_C61::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v56_ - v61_) * C61 * recp_h - i6ct_;
	else
		y = (v56_ - v61_) * C61 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v67_v56_h_prev_C62 ----------------------------------------------------

Coef_v67_v56_h_prev_C62::Coef_v67_v56_h_prev_C62(void)
{
  CoefFuncName = COEF_FUNC_v67_v56_h_prev_C62;
  LiteralName = "v67_v56_h_prev_C62";

	//Assign initial states...
  v56_ = (2.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-333.33)
  v67_ = (1.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;

}

Coef_v67_v56_h_prev_C62::~Coef_v67_v56_h_prev_C62(void)
{
										 //(-666.66)
}

void Coef_v67_v56_h_prev_C62::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v67_ - v56_) * C62 * recp_h - i6cb_;
	else
		y = (v67_ - v56_) * C62 * recp_h;

}

// ------------------------------------------------------------------






// ---- Coef_recp_rSrD71 ---------------------------------------------

Coef_recp_rSrD71::Coef_recp_rSrD71(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD71;
  LiteralName = "recp_rSrD71";
  rS71 = INIT_COEF_RS_RD;
  rD71 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD71::~Coef_recp_rSrD71(void)
{


}

void Coef_recp_rSrD71::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS71 + 1/rD71;

}


// ------------------------------------------------------------------

// ---- Coef_recp_rSrD72 ---------------------------------------------

Coef_recp_rSrD72::Coef_recp_rSrD72(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD72;
  LiteralName = "recp_rSrD72"; 
  rS72 = INIT_COEF_RS_RD;
  rD72 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD72::~Coef_recp_rSrD72(void)
{


}

void Coef_recp_rSrD72::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS72 + 1/rD72;

}


// ------------------------------------------------------------------





// ---- Coef_C7_h ----------------------------------------------------

Coef_C7_h::Coef_C7_h(void)
{
  CoefFuncName = COEF_FUNC_C7_h;
  LiteralName = "C7_h";

}

Coef_C7_h::~Coef_C7_h(void)
{


}

void Coef_C7_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C7 * recp_h;
	else
		y = C7 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C71_h ----------------------------------------------------

Coef_C71_h::Coef_C71_h(void)
{
  CoefFuncName = COEF_FUNC_C71_h;
  LiteralName = "C71_h";

}

Coef_C71_h::~Coef_C71_h(void)
{


}

void Coef_C71_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C71 * recp_h;
	else
		y = C71 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C72_h ----------------------------------------------------

Coef_C72_h::Coef_C72_h(void)
{
  CoefFuncName = COEF_FUNC_C72_h;
  LiteralName = "C72_h";

}

Coef_C72_h::~Coef_C72_h(void)
{


}

void Coef_C72_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C72 * recp_h;
	else
		y = C72 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_v71_v78_h_prev_C7 ----------------------------------------------------

Coef_v71_v78_h_prev_C7::Coef_v71_v78_h_prev_C7(void)
{
  CoefFuncName = COEF_FUNC_v71_v78_h_prev_C7;
  LiteralName = "v71_v78_h_prev_C7";

  	  //Assign initial states...
  v71_ = (0.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(-500.00)
  v78_ = (0.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-1000.0)

}

Coef_v71_v78_h_prev_C7::~Coef_v71_v78_h_prev_C7(void)
{


}

void Coef_v71_v78_h_prev_C7::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v71_ - v78_) * C7 * recp_h - i7_;
	else
		y = (v71_ - v78_) * C7 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v67_v71_h_prev_C71 ----------------------------------------------------

Coef_v67_v71_h_prev_C71::Coef_v67_v71_h_prev_C71(void)
{
  CoefFuncName = COEF_FUNC_v67_v71_h_prev_C71;
  LiteralName = "v67_v71_h_prev_C71";

	//Assign initial states...
  v71_ = (0.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); //(-500.00)
  v67_ = (1.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-666.66)

}

Coef_v67_v71_h_prev_C71::~Coef_v67_v71_h_prev_C71(void)
{

}

void Coef_v67_v71_h_prev_C71::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v67_ - v71_) * C71 * recp_h - i7ct_;
	else
		y = (v67_ - v71_) * C71 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v78_v67_h_prev_C72 ----------------------------------------------------

Coef_v78_v67_h_prev_C72::Coef_v78_v67_h_prev_C72(void)
{
  CoefFuncName = COEF_FUNC_v78_v67_h_prev_C72;
  LiteralName = "v78_v67_h_prev_C72";

   //Assign initial states...
  v78_ = (0.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-1000.0)
  v67_ = (1.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-666.66)

}

Coef_v78_v67_h_prev_C72::~Coef_v78_v67_h_prev_C72(void)
{

}

void Coef_v78_v67_h_prev_C72::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v78_ - v67_) * C72 * recp_h - i7cb_;
	else
		y = (v78_ - v67_) * C72 * recp_h;

}

// ------------------------------------------------------------------




// ---- Coef_recp_rSrD81 ---------------------------------------------

Coef_recp_rSrD81::Coef_recp_rSrD81(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD81;
  LiteralName = "recp_rSrD81"; 
  rS81 = INIT_COEF_RS_RD;
  rD81 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD81::~Coef_recp_rSrD81(void)
{


}

void Coef_recp_rSrD81::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS81 + 1/rD81;

}



// ------------------------------------------------------------------

// ---- Coef_recp_rSrD82 ---------------------------------------------

Coef_recp_rSrD82::Coef_recp_rSrD82(void)
{
  CoefFuncName = COEF_FUNC_recp_rSrD82;
  LiteralName = "recp_rSrD82"; 
  rS82 = INIT_COEF_RS_RD;
  rD82 = INIT_COEF_RS_RD;

}

Coef_recp_rSrD82::~Coef_recp_rSrD82(void)
{


}

void Coef_recp_rSrD82::CoefFunction(double recp_h, bool DoTrapezoidal)
{

  y = 1/rS82 + 1/rD82;

}



// ------------------------------------------------------------------  




// ---- Coef_C8_h ----------------------------------------------------

Coef_C8_h::Coef_C8_h(void)
{
  CoefFuncName = COEF_FUNC_C8_h;
  LiteralName = "C8_h";

}

Coef_C8_h::~Coef_C8_h(void)
{


}

void Coef_C8_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C8 * recp_h;
	else
		y = C8 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C81_h ----------------------------------------------------

Coef_C81_h::Coef_C81_h(void)
{
  CoefFuncName = COEF_FUNC_C81_h;
  LiteralName = "C81_h";

}

Coef_C81_h::~Coef_C81_h(void)
{


}

void Coef_C81_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C81 * recp_h;
	else
		y = C81 * recp_h;

}

// ------------------------------------------------------------------



// ---- Coef_C82_h --------------------------------------------------

Coef_C82_h::Coef_C82_h(void)
{
  CoefFuncName = COEF_FUNC_C82_h;
  LiteralName = "C78_h";

}

Coef_C82_h::~Coef_C82_h(void)
{


}

void Coef_C82_h::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * C82 * recp_h;
	else
		y = C82 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_v81_h_prev_plus_MinusBus_C8 -----------------------------


Coef_v81_h_prev_plus_MinusBus_C8::Coef_v81_h_prev_plus_MinusBus_C8(void)
{
	  CoefFuncName = COEF_FUNC_v81_h_prev_plus_MinusBus_C8;
	  LiteralName = "v81_h_prev_plus_MinusBus_C8";

	  //Assign initial states...
	  v81_ = MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); 															 //(-500.00)

}

Coef_v81_h_prev_plus_MinusBus_C8::~Coef_v81_h_prev_plus_MinusBus_C8(void)
{



}

void Coef_v81_h_prev_plus_MinusBus_C8::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * v81_ * C8 * recp_h - i8_;
	else
		y = v81_ * C8 * recp_h;
}


// ------------------------------------------------------------------


// ---- Coef_v78_v81_h_prev_C81 ----------------------------------------------------

Coef_v78_v81_h_prev_C81::Coef_v78_v81_h_prev_C81(void)
{
  CoefFuncName = COEF_FUNC_v78_v81_h_prev_C81;
  LiteralName = "v78_v81_h_prev_C81";

  //Assign initial states...
  	v78_ = (0.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-1000.0)
  	v81_ = MINUS_DC_BUS_VOLTAGE + ((PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 4.0); 															 //(-500.00)

}

Coef_v78_v81_h_prev_C81::~Coef_v78_v81_h_prev_C81(void)
{

}

void Coef_v78_v81_h_prev_C81::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = 2 * (v78_ - v81_) * C81 * recp_h - i8ct_;
	else
		y = (v78_ - v81_) * C81 * recp_h;

}

// ------------------------------------------------------------------


// ---- Coef_minus_v78_h_prev_minus_MinusBus_C82 -----------------------------


Coef_minus_v78_h_prev_minus_MinusBus_C82::Coef_minus_v78_h_prev_minus_MinusBus_C82(void)
{
	  CoefFuncName = COEF_FUNC_minus_v78_h_prev_minus_MinusBus_C82;
	  LiteralName = "minus_v78_h_prev_minus_MinusBus_C82";

	//Assign initial states...
	v78_ = (0.0 * (PLUS_DC_BUS_VOLTAGE - MINUS_DC_BUS_VOLTAGE) / 6.0) + MINUS_DC_BUS_VOLTAGE;														 //(-1000.0)

}

Coef_minus_v78_h_prev_minus_MinusBus_C82::~Coef_minus_v78_h_prev_minus_MinusBus_C82(void)
{



}

void Coef_minus_v78_h_prev_minus_MinusBus_C82::CoefFunction(double recp_h, bool DoTrapezoidal)
{

	if(DoTrapezoidal)
		y = - 2 * - v78_ * C82 * recp_h - i8cb_;
	else
		y = - v78_ * C82 * recp_h;
}


// ------------------------------------------------------------------




// ******************************************************************






// **** SwitchObject Functions **************************************

// ---- Switch_rS11 -----------------------------------------------

Switch_rS11::Switch_rS11(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS11;
  LiteralName = "rS11";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD11];

  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS11
  DoProbes = TRUE;
#ifdef DO_PROBE_rS11_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS11_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS11_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}

Switch_rS11::~Switch_rS11(void)
{

}

void Switch_rS11::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD11 *) pCurCoefItem->pCoefObject)->rS11 = y_igbt;
  ((Coef_recp_rSrD11 *) pCurCoefItem->pCoefObject)->rD11 = y_diode;

}

// ------------------------------------------------------------------


// ---- Switch_rS12 -----------------------------------------------

Switch_rS12::Switch_rS12(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS12;
  LiteralName = "rS12";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD12];


  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS12
  DoProbes = TRUE;
#ifdef DO_PROBE_rS12_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS12_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS12_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}

Switch_rS12::~Switch_rS12(void)
{

}

void Switch_rS12::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD12 *) pCurCoefItem->pCoefObject)->rS12 = y_igbt;
  ((Coef_recp_rSrD12 *) pCurCoefItem->pCoefObject)->rD12 = y_diode;

}


// ------------------------------------------------------------------


// ---- Switch_rS21 -----------------------------------------------

Switch_rS21::Switch_rS21(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS21;
  LiteralName = "rS21";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD21];

  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS21
  DoProbes = TRUE;
#ifdef DO_PROBE_rS21_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS21_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS21_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}

Switch_rS21::~Switch_rS21(void)
{

}

void Switch_rS21::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD21 *) pCurCoefItem->pCoefObject)->rS21 = y_igbt;
  ((Coef_recp_rSrD21 *) pCurCoefItem->pCoefObject)->rD21 = y_diode;

}


// ------------------------------------------------------------------


// ---- Switch_rS22 -----------------------------------------------

Switch_rS22::Switch_rS22(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS22;
  LiteralName = "rS22";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD22];


  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS22
  DoProbes = TRUE;
#ifdef DO_PROBE_rS22_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS22_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS22_gate
  DoProbes_gate = TRUE;
#endif 
#endif



}

Switch_rS22::~Switch_rS22(void)
{

}

void Switch_rS22::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD22 *) pCurCoefItem->pCoefObject)->rS22 = y_igbt;
  ((Coef_recp_rSrD22 *) pCurCoefItem->pCoefObject)->rD22 = y_diode;

}



// ------------------------------------------------------------------


// ---- Switch_rS31 -----------------------------------------------

Switch_rS31::Switch_rS31(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS31;
  LiteralName = "rS31";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD31];



  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS31
  DoProbes = TRUE;
#ifdef DO_PROBE_rS31_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS31_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS31_gate
  DoProbes_gate = TRUE;
#endif 
#endif



}

Switch_rS31::~Switch_rS31(void)
{

}

void Switch_rS31::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD31 *) pCurCoefItem->pCoefObject)->rS31 = y_igbt;
  ((Coef_recp_rSrD31 *) pCurCoefItem->pCoefObject)->rD31 = y_diode;

}



// ------------------------------------------------------------------


// ---- Switch_rS32 -----------------------------------------------

Switch_rS32::Switch_rS32(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS32;
  LiteralName = "rS32";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD32];


  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS32
  DoProbes = TRUE;
#ifdef DO_PROBE_rS32_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS32_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS32_gate
  DoProbes_gate = TRUE;
#endif 
#endif

}

Switch_rS32::~Switch_rS32(void)
{

}

void Switch_rS32::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD32 *) pCurCoefItem->pCoefObject)->rS32 = y_igbt;
  ((Coef_recp_rSrD32 *) pCurCoefItem->pCoefObject)->rD32 = y_diode;

}


// ------------------------------------------------------------------


// ---- Switch_rS41 -----------------------------------------------

Switch_rS41::Switch_rS41(void)
{

  CoefObjItem * pCurCoefItem;


  SwitchFuncName = SWITCH_FUNC_rS41;
  LiteralName = "rS41";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD41];



  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS41
  DoProbes = TRUE;
#ifdef DO_PROBE_rS41_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS41_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS41_gate
  DoProbes_gate = TRUE;
#endif 
#endif



}

Switch_rS41::~Switch_rS41(void)
{

}

void Switch_rS41::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD41 *) pCurCoefItem->pCoefObject)->rS41 = y_igbt;
  ((Coef_recp_rSrD41 *) pCurCoefItem->pCoefObject)->rD41 = y_diode;

}



// ------------------------------------------------------------------


// ---- Switch_rS42 -----------------------------------------------

Switch_rS42::Switch_rS42(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS42;
  LiteralName = "rS42";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD42];

  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS42
  DoProbes = TRUE;
#ifdef DO_PROBE_rS42_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS42_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS42_gate
  DoProbes_gate = TRUE;
#endif 
#endif



}

Switch_rS42::~Switch_rS42(void)
{

}

void Switch_rS42::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD42 *) pCurCoefItem->pCoefObject)->rS42 = y_igbt;
  ((Coef_recp_rSrD42 *) pCurCoefItem->pCoefObject)->rD42 = y_diode;

}


// ------------------------------------------------------------------


// ---- Switch_rS51 -----------------------------------------------

Switch_rS51::Switch_rS51(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS51;
  LiteralName = "rS51";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD51];

  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS51
  DoProbes = TRUE;
#ifdef DO_PROBE_rS51_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS51_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS51_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}
Switch_rS51::~Switch_rS51(void)
{

}

void Switch_rS51::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD51 *) pCurCoefItem->pCoefObject)->rS51 = y_igbt;
  ((Coef_recp_rSrD51 *) pCurCoefItem->pCoefObject)->rD51 = y_diode;

}



// ------------------------------------------------------------------


// ---- Switch_rS52 -----------------------------------------------

Switch_rS52::Switch_rS52(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS52;
  LiteralName = "rS52";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD52];


  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS52
  DoProbes = TRUE;
#ifdef DO_PROBE_rS52_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS52_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS52_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}

Switch_rS52::~Switch_rS52(void)
{

}

void Switch_rS52::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD52 *) pCurCoefItem->pCoefObject)->rS52 = y_igbt;
  ((Coef_recp_rSrD52 *) pCurCoefItem->pCoefObject)->rD52 = y_diode;

}




// ------------------------------------------------------------------


// ---- Switch_rS61 -----------------------------------------------

Switch_rS61::Switch_rS61(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS61;
  LiteralName = "rS61";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD61];


  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS61
  DoProbes = TRUE;
#ifdef DO_PROBE_rS61_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS61_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS61_gate
  DoProbes_gate = TRUE;
#endif 
#endif



}

Switch_rS61::~Switch_rS61(void)
{

}

void Switch_rS61::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD61 *) pCurCoefItem->pCoefObject)->rS61 = y_igbt;
  ((Coef_recp_rSrD61 *) pCurCoefItem->pCoefObject)->rD61 = y_diode;

}



// ------------------------------------------------------------------


// ---- Switch_rS62 -----------------------------------------------

Switch_rS62::Switch_rS62(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS62;
  LiteralName = "rS62";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD62];



  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS62
  DoProbes = TRUE;
#ifdef DO_PROBE_rS62_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS62_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS62_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}

Switch_rS62::~Switch_rS62(void)
{

}

void Switch_rS62::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD62 *) pCurCoefItem->pCoefObject)->rS62 = y_igbt;
  ((Coef_recp_rSrD62 *) pCurCoefItem->pCoefObject)->rD62 = y_diode;

}


// ------------------------------------------------------------------


// ---- Switch_rS71 -----------------------------------------------

Switch_rS71::Switch_rS71(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS71;
  LiteralName = "rS71";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD71];

  
  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS71
  DoProbes = TRUE;
#ifdef DO_PROBE_rS71_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS71_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS71_gate
  DoProbes_gate = TRUE;
#endif 
#endif



}

Switch_rS71::~Switch_rS71(void)
{

}

void Switch_rS71::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD71 *) pCurCoefItem->pCoefObject)->rS71 = y_igbt;
  ((Coef_recp_rSrD71 *) pCurCoefItem->pCoefObject)->rD71 = y_diode;

}


// ------------------------------------------------------------------


// ---- Switch_rS72 -----------------------------------------------

Switch_rS72::Switch_rS72(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS72;
  LiteralName = "rS72";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD72];



  IsIgbt = TRUE;
  IsMinusPol = FALSE;

#ifdef PLOT_SWITCHOBJECT_PROBES_rS72
  DoProbes = TRUE;
#ifdef DO_PROBE_rS72_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS72_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS72_gate
  DoProbes_gate = TRUE;
#endif 
#endif



}

Switch_rS72::~Switch_rS72(void)
{

}

void Switch_rS72::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD72 *) pCurCoefItem->pCoefObject)->rS72 = y_igbt;
  ((Coef_recp_rSrD72 *) pCurCoefItem->pCoefObject)->rD72 = y_diode;

}


// ------------------------------------------------------------------



// ---- Switch_rS81 -----------------------------------------------

Switch_rS81::Switch_rS81(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS81;
  LiteralName = "rS81";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD81];


  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS81
  DoProbes = TRUE;
#ifdef DO_PROBE_rS81_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS81_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS81_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}

Switch_rS81::~Switch_rS81(void)
{

}

void Switch_rS81::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD81 *) pCurCoefItem->pCoefObject)->rS81 = y_igbt;
  ((Coef_recp_rSrD81 *) pCurCoefItem->pCoefObject)->rD81 = y_diode;

}


// ------------------------------------------------------------------


// ---- Switch_rS82 -----------------------------------------------

Switch_rS82::Switch_rS82(void)
{

  CoefObjItem * pCurCoefItem;

  SwitchFuncName = SWITCH_FUNC_rS82;
  LiteralName = "rS82";

  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;          
  pCurCoefItem = pCoefObjRValList;
 
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_recp_rSrD82];



  IsIgbt = TRUE;
  IsMinusPol = FALSE;


#ifdef PLOT_SWITCHOBJECT_PROBES_rS82
  DoProbes = TRUE;
#ifdef DO_PROBE_rS82_res
  DoProbes_res = TRUE;
#endif
#ifdef DO_PROBE_rS82_cur
  DoProbes_cur = TRUE;
#endif
#ifdef DO_PROBE_rS82_gate
  DoProbes_gate = TRUE;
#endif 
#endif


}

Switch_rS82::~Switch_rS82(void)
{

}

void Switch_rS82::CoefRValueUpdate(void)
{
  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_recp_rSrD82 *) pCurCoefItem->pCoefObject)->rS82 = y_igbt;
  ((Coef_recp_rSrD82 *) pCurCoefItem->pCoefObject)->rD82 = y_diode;

}


// ------------------------------------------------------------------

   


// ******************************************************************



// **** SpiceObject Functions ***************************************

// ---- Spice_MultiLevel ------------------------------------------

Spice_MultiLevel::Spice_MultiLevel(void)
{
  SpiceObjItem * pCurSpiceItem;
  OdeObjItem * pCurOdeItem;
  SwitchObjItem * pCurSwitchItem;
  CoefObjItem * pCurCoefItem;
  SrcObjItem * pCurSrcItem;
    
  SpiceFuncName = SPICE_FUNC_MultiLevel;
  LiteralName = "MLvl";


#ifdef PLOT_SPICEOBJECT_PROBES_solution_vector
  DoProbes = TRUE;
#endif



  
  InitializeMatrix(69);   //(69 rows, 70 columns)
  
  //Eq. (1)
  a_CoefObj[0][Ind_v11] = &Coef_recp_rSrD11_i;
  a_Static[0][Ind_v11] = -1.0;
  a_CoefObj[0][Ind_i1st] = 0;
  a_Static[0][Ind_i1st] = -1.0;
  a_CoefObj[0][69] = &Coef_recp_rSrD11_i;
  a_Static[0][69] = -PLUS_DC_BUS_VOLTAGE;
  
  //Eq. (2)
  a_CoefObj[1][Ind_v12] = &Coef_recp_rSrD12_i;
  a_Static[1][Ind_v12] = 1.0;
  a_CoefObj[1][Ind_i1sb] = 0;
  a_Static[1][Ind_i1sb] = -1.0;
  a_CoefObj[1][69] = &Coef_recp_rSrD12_i;
  a_Static[1][69] = PLUS_DC_BUS_VOLTAGE;

  //Eq. (3)
  a_CoefObj[2][Ind_ipl] = 0;
  a_Static[2][Ind_ipl] = 1.0;
  a_CoefObj[2][Ind_i1_l] = 0;
  a_Static[2][Ind_i1_l] = 1.0;
  a_CoefObj[2][Ind_i1] = 0;
  a_Static[2][Ind_i1] = -1.0;
  a_CoefObj[2][69] = 0;
  a_Static[2][69] = 0;

  //Eq. (4)
  a_CoefObj[3][Ind_i1st] = 0;
  a_Static[3][Ind_i1st] = 1.0;
  a_CoefObj[3][Ind_i1ct] = 0;
  a_Static[3][Ind_i1ct] = 1.0;
  a_CoefObj[3][Ind_i1] = 0;
  a_Static[3][Ind_i1] = -1.0;
  a_CoefObj[3][69] = 0;
  a_Static[3][69] = 0;

  //Eq. (5)
  a_CoefObj[4][Ind_i1sb] = 0;
  a_Static[4][Ind_i1sb] = 1.0;
  a_CoefObj[4][Ind_i1cb] = 0;
  a_Static[4][Ind_i1cb] = 1.0;
  a_CoefObj[4][Ind_i1_l] = 0;
  a_Static[4][Ind_i1_l] = -1.0;
  a_CoefObj[4][69] = 0;
  a_Static[4][69] = 0;

   //Eq. (6)
  a_CoefObj[5][Ind_v11] = &Coef_C1_h_i;
  a_Static[5][Ind_v11] = 1.0;
  a_CoefObj[5][Ind_v12] = &Coef_C1_h_i;
  a_Static[5][Ind_v12] = -1.0;
  a_CoefObj[5][Ind_i1] = 0;
  a_Static[5][Ind_i1] = -1.0;
  a_CoefObj[5][69] = &Coef_v11_v12_h_prev_C1_i;
  a_Static[5][69] = 1.0;

   //Eq. (7)
  a_CoefObj[6][Ind_v11] = &Coef_C11_h_i;
  a_Static[6][Ind_v11] = -1.0;
  a_CoefObj[6][Ind_i1ct] = 0;
  a_Static[6][Ind_i1ct] = -1.0;
  a_CoefObj[6][69] = &Coef_minus_v11_h_prev_minus_PlusBus_C11_i;
  a_Static[6][69] = 1.0;

   //Eq. (8)
  a_CoefObj[7][Ind_v12] = &Coef_C12_h_i;
  a_Static[7][Ind_v12] = 1.0;
  a_CoefObj[7][Ind_i1cb] = 0;
  a_Static[7][Ind_i1cb] = -1.0;
  a_CoefObj[7][69] = &Coef_v12_h_prev_plus_PlusBus_C12_i;
  a_Static[7][69] = 1.0;

   //Eq. (9)
  a_CoefObj[8][Ind_v12] = &Coef_recp_rSrD21_i;
  a_Static[8][Ind_v12] = 1.0;
  a_CoefObj[8][Ind_v21] = &Coef_recp_rSrD21_i;
  a_Static[8][Ind_v21] = -1.0;
  a_CoefObj[8][Ind_i2st] = 0;
  a_Static[8][Ind_i2st] = -1.0;
  a_CoefObj[8][69] = 0;
  a_Static[8][69] = 0;
  
  //Eq. (10)
  a_CoefObj[9][Ind_v23] = &Coef_recp_rSrD22_i;
  a_Static[9][Ind_v23] = 1.0;
  a_CoefObj[9][Ind_v12] = &Coef_recp_rSrD22_i;
  a_Static[9][Ind_v12] = -1.0;
  a_CoefObj[9][Ind_i2sb] = 0;
  a_Static[9][Ind_i2sb] = -1.0;
  a_CoefObj[9][69] = 0;
  a_Static[9][69] = 0;

  //Eq. (11)
  a_CoefObj[10][Ind_ipl] = 0;
  a_Static[10][Ind_ipl] = 1.0;
  a_CoefObj[10][Ind_i2_l] = 0;
  a_Static[10][Ind_i2_l] = 1.0;
  a_CoefObj[10][Ind_i2] = 0;
  a_Static[10][Ind_i2] = -1.0;
  a_CoefObj[10][69] = 0;
  a_Static[10][69] = 0;

  //Eq. (12)
  a_CoefObj[11][Ind_i2st] = 0;
  a_Static[11][Ind_i2st] = 1.0;
  a_CoefObj[11][Ind_i2ct] = 0;
  a_Static[11][Ind_i2ct] = 1.0;
  a_CoefObj[11][Ind_i2] = 0;
  a_Static[11][Ind_i2] = -1.0;
  a_CoefObj[11][69] = 0;
  a_Static[11][69] = 0;

  //Eq. (13)
  a_CoefObj[12][Ind_i2sb] = 0;
  a_Static[12][Ind_i2sb] = 1.0;
  a_CoefObj[12][Ind_i2cb] = 0;
  a_Static[12][Ind_i2cb] = 1.0;
  a_CoefObj[12][Ind_i2_l] = 0;
  a_Static[12][Ind_i2_l] = -1.0;
  a_CoefObj[12][69] = 0;
  a_Static[12][69] = 0;


  //Eq. (14)
  a_CoefObj[13][Ind_v21] = &Coef_C2_h_i;
  a_Static[13][Ind_v21] = 1.0;
  a_CoefObj[13][Ind_v23] = &Coef_C2_h_i;
  a_Static[13][Ind_v23] = -1.0;
  a_CoefObj[13][Ind_i2] = 0;
  a_Static[13][Ind_i2] = -1.0;
  a_CoefObj[13][69] = &Coef_v21_v23_h_prev_C2_i;
  a_Static[13][69] = 1.0;



  //Eq. (15)
  a_CoefObj[14][Ind_v12] = &Coef_C21_h_i;
  a_Static[14][Ind_v12] = 1.0;
  a_CoefObj[14][Ind_v21] = &Coef_C21_h_i;
  a_Static[14][Ind_v21] = -1.0;
  a_CoefObj[14][Ind_i2ct] = 0;
  a_Static[14][Ind_i2ct] = -1.0;
  a_CoefObj[14][69] = &Coef_v12_v21_h_prev_C21_i;
  a_Static[14][69] = 1.0;


  //Eq. (16)
  a_CoefObj[15][Ind_v23] = &Coef_C22_h_i;
  a_Static[15][Ind_v23] = 1.0;
  a_CoefObj[15][Ind_v12] = &Coef_C22_h_i;
  a_Static[15][Ind_v12] = -1.0;
  a_CoefObj[15][Ind_i2cb] = 0;
  a_Static[15][Ind_i2cb] = -1.0;
  a_CoefObj[15][69] = &Coef_v23_v12_h_prev_C22_i;
  a_Static[15][69] = 1.0;


  //Eq. (17)
  a_CoefObj[16][Ind_v23] = &Coef_recp_rSrD31_i;
  a_Static[16][Ind_v23] = 1.0;
  a_CoefObj[16][Ind_v31] = &Coef_recp_rSrD31_i;
  a_Static[16][Ind_v31] = -1.0;
  a_CoefObj[16][Ind_i3st] = 0;
  a_Static[16][Ind_i3st] = -1.0;
  a_CoefObj[16][69] = 0;
  a_Static[16][69] = 0;
  
  //Eq. (18)
  a_CoefObj[17][Ind_v34] = &Coef_recp_rSrD32_i;
  a_Static[17][Ind_v34] = 1.0;
  a_CoefObj[17][Ind_v23] = &Coef_recp_rSrD32_i;
  a_Static[17][Ind_v23] = -1.0;
  a_CoefObj[17][Ind_i3sb] = 0;
  a_Static[17][Ind_i3sb] = -1.0;
  a_CoefObj[17][69] = 0;
  a_Static[17][69] = 0;

  
  //Eq. (19)
  a_CoefObj[18][Ind_ipl] = 0;
  a_Static[18][Ind_ipl] = 1.0;
  a_CoefObj[18][Ind_i3_l] = 0;
  a_Static[18][Ind_i3_l] = 1.0;
  a_CoefObj[18][Ind_i3] = 0;
  a_Static[18][Ind_i3] = -1.0;
  a_CoefObj[18][69] = 0;
  a_Static[18][69] = 0;


  //Eq. (20)
  a_CoefObj[19][Ind_i3st] = 0;
  a_Static[19][Ind_i3st] = 1.0;
  a_CoefObj[19][Ind_i3ct] = 0;
  a_Static[19][Ind_i3ct] = 1.0;
  a_CoefObj[19][Ind_i3] = 0;
  a_Static[19][Ind_i3] = -1.0;
  a_CoefObj[19][69] = 0;
  a_Static[19][69] = 0;


  //Eq. (21)
  a_CoefObj[20][Ind_i3sb] = 0;
  a_Static[20][Ind_i3sb] = 1.0;
  a_CoefObj[20][Ind_i3cb] = 0;
  a_Static[20][Ind_i3cb] = 1.0;
  a_CoefObj[20][Ind_i3_l] = 0;
  a_Static[20][Ind_i3_l] = -1.0;
  a_CoefObj[20][69] = 0;
  a_Static[20][69] = 0;


   //Eq. (22)
  a_CoefObj[21][Ind_v31] = &Coef_C3_h_i;
  a_Static[21][Ind_v31] = 1.0;
  a_CoefObj[21][Ind_v34] = &Coef_C3_h_i;
  a_Static[21][Ind_v34] = -1.0;
  a_CoefObj[21][Ind_i3] = 0;
  a_Static[21][Ind_i3] = -1.0;
  a_CoefObj[21][69] = &Coef_v31_v34_h_prev_C3_i;
  a_Static[21][69] = 1.0;


  //Eq. (23)
  a_CoefObj[22][Ind_v23] = &Coef_C31_h_i;
  a_Static[22][Ind_v23] = 1.0;
  a_CoefObj[22][Ind_v31] = &Coef_C31_h_i;
  a_Static[22][Ind_v31] = -1.0;
  a_CoefObj[22][Ind_i3ct] = 0;
  a_Static[22][Ind_i3ct] = -1.0;
  a_CoefObj[22][69] = &Coef_v23_v31_h_prev_C31_i;
  a_Static[22][69] = 1.0;


  //Eq. (24)
  a_CoefObj[23][Ind_v34] = &Coef_C32_h_i;
  a_Static[23][Ind_v34] = 1.0;
  a_CoefObj[23][Ind_v23] = &Coef_C32_h_i;
  a_Static[23][Ind_v23] = -1.0;
  a_CoefObj[23][Ind_i3cb] = 0;
  a_Static[23][Ind_i3cb] = -1.0;
  a_CoefObj[23][69] = &Coef_v34_v23_h_prev_C32_i;
  a_Static[23][69] = 1.0;


  //Eq. (25)
  a_CoefObj[24][Ind_v34] = &Coef_recp_rSrD41_i;
  a_Static[24][Ind_v34] = 1.0;
  a_CoefObj[24][Ind_v41] = &Coef_recp_rSrD41_i;
  a_Static[24][Ind_v41] = -1.0;
  a_CoefObj[24][Ind_i4st] = 0;
  a_Static[24][Ind_i4st] = -1.0;
  a_CoefObj[24][69] = 0;
  a_Static[24][69] = 0;
  
  //Eq. (26)
  a_CoefObj[25][Ind_v44] = &Coef_recp_rSrD42_i;
  a_Static[25][Ind_v44] = 1.0;
  a_CoefObj[25][Ind_v34] = &Coef_recp_rSrD42_i;
  a_Static[25][Ind_v34] = -1.0;
  a_CoefObj[25][Ind_i4sb] = 0;
  a_Static[25][Ind_i4sb] = -1.0;
  a_CoefObj[25][69] = 0;
  a_Static[25][69] = 0;

  //Eq. (27)
  a_CoefObj[26][Ind_ipl] = 0;
  a_Static[26][Ind_ipl] = 1.0;
  a_CoefObj[26][Ind_i4_l] = 0;
  a_Static[26][Ind_i4_l] = 1.0;
  a_CoefObj[26][Ind_i4] = 0;
  a_Static[26][Ind_i4] = -1.0;
  a_CoefObj[26][69] = 0;
  a_Static[26][69] = 0;


  //Eq. (28)
  a_CoefObj[27][Ind_i4st] = 0;
  a_Static[27][Ind_i4st] = 1.0;
  a_CoefObj[27][Ind_i4ct] = 0;
  a_Static[27][Ind_i4ct] = 1.0;
  a_CoefObj[27][Ind_i4] = 0;
  a_Static[27][Ind_i4] = -1.0;
  a_CoefObj[27][69] = 0;
  a_Static[27][69] = 0;


  //Eq. (29)
  a_CoefObj[28][Ind_i4sb] = 0;
  a_Static[28][Ind_i4sb] = 1.0;
  a_CoefObj[28][Ind_i4cb] = 0;
  a_Static[28][Ind_i4cb] = 1.0;
  a_CoefObj[28][Ind_i4_l] = 0;
  a_Static[28][Ind_i4_l] = -1.0;
  a_CoefObj[28][69] = 0;
  a_Static[28][69] = 0;


  //Eq. (30)
  a_CoefObj[29][Ind_v41] = &Coef_C4_h_i;
  a_Static[29][Ind_v41] = 1.0;
  a_CoefObj[29][Ind_v44] = &Coef_C4_h_i;
  a_Static[29][Ind_v44] = -1.0;
  a_CoefObj[29][Ind_i4] = 0;
  a_Static[29][Ind_i4] = -1.0;
  a_CoefObj[29][69] = &Coef_v41_v44_h_prev_C4_i;
  a_Static[29][69] = 1.0;


  //Eq. (31)
  a_CoefObj[30][Ind_v34] = &Coef_C41_h_i;
  a_Static[30][Ind_v34] = 1.0;
  a_CoefObj[30][Ind_v41] = &Coef_C41_h_i;
  a_Static[30][Ind_v41] = -1.0;
  a_CoefObj[30][Ind_i4ct] = 0;
  a_Static[30][Ind_i4ct] = -1.0;
  a_CoefObj[30][69] = &Coef_v34_v41_h_prev_C41_i;
  a_Static[30][69] = 1.0;


  //Eq. (32)
  a_CoefObj[31][Ind_v44] = &Coef_C42_h_i;
  a_Static[31][Ind_v44] = 1.0;
  a_CoefObj[31][Ind_v34] = &Coef_C42_h_i;
  a_Static[31][Ind_v34] = -1.0;
  a_CoefObj[31][Ind_i4cb] = 0;
  a_Static[31][Ind_i4cb] = -1.0;
  a_CoefObj[31][69] = &Coef_v44_v34_h_prev_C42_i;
  a_Static[31][69] = 1.0;

  
  //Eq. (33)
  a_CoefObj[32][Ind_v44] = 0;
  a_Static[32][Ind_v44] = RECEP_Rp;
  a_CoefObj[32][Ind_vs_a] = 0;
  a_Static[32][Ind_vs_a] = -RECEP_Rp;
  a_CoefObj[32][Ind_iLp] = 0;
  a_Static[32][Ind_iLp] = 1.0;
  a_CoefObj[32][Ind_ipl] = 0;
  a_Static[32][Ind_ipl] = -1.0;
  a_CoefObj[32][69] = 0;
  a_Static[32][69] = 0;
  
  //Eq. (34)
  a_CoefObj[33][Ind_v45] = 0;
  a_Static[33][Ind_v45] = -RECEP_Rn;
  a_CoefObj[33][Ind_vs_a] = 0;
  a_Static[33][Ind_vs_a] = RECEP_Rn;
  a_CoefObj[33][Ind_iLn] = 0;
  a_Static[33][Ind_iLn] = 1.0;
  a_CoefObj[33][Ind_inl] = 0;
  a_Static[33][Ind_inl] = -1.0;
  a_CoefObj[33][69] = 0;
  a_Static[33][69] = 0;


  
  //Eq. (35)
  a_CoefObj[34][Ind_iLp] = &Coef_Lp_h_i;
  a_Static[34][Ind_iLp] = 1.0;
  a_CoefObj[34][Ind_iLn] = &Coef_Mpn_h_i;
  a_Static[34][Ind_iLn] = 1.0;
  a_CoefObj[34][Ind_v44] = 0;
  a_Static[34][Ind_v44] = -1.0;
  a_CoefObj[34][Ind_vs_a] = 0;
  a_Static[34][Ind_vs_a] = 1.0;
  a_CoefObj[34][69] = &Coef_iLp_iLn_h_prev_Lp_Mpn_i;
  a_Static[34][69] = 1.0;


  //Eq. (36)
  a_CoefObj[35][Ind_iLp] = &Coef_Mnp_h_i;
  a_Static[35][Ind_iLp] = 1.0;
  a_CoefObj[35][Ind_iLn] = &Coef_Ln_h_i;
  a_Static[35][Ind_iLn] = 1.0;
  a_CoefObj[35][Ind_vs_a] = 0;
  a_Static[35][Ind_vs_a] = -1.0;
  a_CoefObj[35][Ind_v45] = 0;
  a_Static[35][Ind_v45] = 1.0;
  a_CoefObj[35][69] = &Coef_iLp_iLn_h_prev_Mnp_Ln_i;
  a_Static[35][69] = 1.0;


  //Eq. (37)
  a_CoefObj[36][Ind_ipl] = 0;
  a_Static[36][Ind_ipl] = 1.0;
  a_CoefObj[36][Ind_inl] = 0;
  a_Static[36][Ind_inl] = -1.0;
  a_CoefObj[36][69] = &Coef_ia_i;
  a_Static[36][69] = 1.0;


   
  //Eq. (38)
  a_CoefObj[37][Ind_v45] = &Coef_recp_rSrD51_i;
  a_Static[37][Ind_v45] = 1.0;
  a_CoefObj[37][Ind_v51] = &Coef_recp_rSrD51_i;
  a_Static[37][Ind_v51] = -1.0;
  a_CoefObj[37][Ind_i5st] = 0;
  a_Static[37][Ind_i5st] = -1.0;
  a_CoefObj[37][69] = 0;
  a_Static[37][69] = 0;
  

  //Eq. (39)
  a_CoefObj[38][Ind_v56] = &Coef_recp_rSrD52_i;
  a_Static[38][Ind_v56] = 1.0;
  a_CoefObj[38][Ind_v45] = &Coef_recp_rSrD52_i;
  a_Static[38][Ind_v45] = -1.0;
  a_CoefObj[38][Ind_i5sb] = 0;
  a_Static[38][Ind_i5sb] = -1.0;
  a_CoefObj[38][69] = 0;
  a_Static[38][69] = 0;

  //Eq. (40)
  a_CoefObj[39][Ind_inl] = 0;
  a_Static[39][Ind_inl] = 1.0;
  a_CoefObj[39][Ind_i5_l] = 0;
  a_Static[39][Ind_i5_l] = 1.0;
  a_CoefObj[39][Ind_i5] = 0;
  a_Static[39][Ind_i5] = -1.0;
  a_CoefObj[39][69] = 0;
  a_Static[39][69] = 0;

  //Eq. (41)
  a_CoefObj[40][Ind_i5st] = 0;
  a_Static[40][Ind_i5st] = 1.0;
  a_CoefObj[40][Ind_i5ct] = 0;
  a_Static[40][Ind_i5ct] = 1.0;
  a_CoefObj[40][Ind_i5] = 0;
  a_Static[40][Ind_i5] = -1.0;
  a_CoefObj[40][69] = 0;
  a_Static[40][69] = 0;


  //Eq. (42)
  a_CoefObj[41][Ind_i5sb] = 0;
  a_Static[41][Ind_i5sb] = 1.0;
  a_CoefObj[41][Ind_i5cb] = 0;
  a_Static[41][Ind_i5cb] = 1.0;
  a_CoefObj[41][Ind_i5_l] = 0;
  a_Static[41][Ind_i5_l] = -1.0;
  a_CoefObj[41][69] = 0;
  a_Static[41][69] = 0;


  //Eq. (43)
  a_CoefObj[42][Ind_v51] = &Coef_C5_h_i;
  a_Static[42][Ind_v51] = 1.0;
  a_CoefObj[42][Ind_v56] = &Coef_C5_h_i;
  a_Static[42][Ind_v56] = -1.0;
  a_CoefObj[42][Ind_i5] = 0;
  a_Static[42][Ind_i5] = -1.0;
  a_CoefObj[42][69] = &Coef_v51_v56_h_prev_C5_i;
  a_Static[42][69] = 1.0;


  //Eq. (44)
  a_CoefObj[43][Ind_v45] = &Coef_C51_h_i;
  a_Static[43][Ind_v45] = 1.0;
  a_CoefObj[43][Ind_v51] = &Coef_C51_h_i;
  a_Static[43][Ind_v51] = -1.0;
  a_CoefObj[43][Ind_i5ct] = 0;
  a_Static[43][Ind_i5ct] = -1.0;
  a_CoefObj[43][69] = &Coef_v45_v51_h_prev_C51_i;
  a_Static[43][69] = 1.0;


  //Eq. (45)
  a_CoefObj[44][Ind_v56] = &Coef_C52_h_i;
  a_Static[44][Ind_v56] = 1.0;
  a_CoefObj[44][Ind_v45] = &Coef_C52_h_i;
  a_Static[44][Ind_v45] = -1.0;
  a_CoefObj[44][Ind_i5cb] = 0;
  a_Static[44][Ind_i5cb] = -1.0;
  a_CoefObj[44][69] = &Coef_v56_v45_h_prev_C52_i;
  a_Static[44][69] = 1.0;



  //Eq. (46)
  a_CoefObj[45][Ind_v56] = &Coef_recp_rSrD61_i;
  a_Static[45][Ind_v56] = 1.0;
  a_CoefObj[45][Ind_v61] = &Coef_recp_rSrD61_i;
  a_Static[45][Ind_v61] = -1.0;
  a_CoefObj[45][Ind_i6st] = 0;
  a_Static[45][Ind_i6st] = -1.0;
  a_CoefObj[45][69] = 0;
  a_Static[45][69] = 0;

  //Eq. (47)
  a_CoefObj[46][Ind_v67] = &Coef_recp_rSrD62_i;
  a_Static[46][Ind_v67] = 1.0;
  a_CoefObj[46][Ind_v56] = &Coef_recp_rSrD62_i;
  a_Static[46][Ind_v56] = -1.0;
  a_CoefObj[46][Ind_i6sb] = 0;
  a_Static[46][Ind_i6sb] = -1.0;
  a_CoefObj[46][69] = 0;
  a_Static[46][69] = 0;

  //Eq. (48)
  a_CoefObj[47][Ind_inl] = 0;
  a_Static[47][Ind_inl] = 1.0;
  a_CoefObj[47][Ind_i6_l] = 0;
  a_Static[47][Ind_i6_l] = 1.0;
  a_CoefObj[47][Ind_i6] = 0;
  a_Static[47][Ind_i6] = -1.0;
  a_CoefObj[47][69] = 0;
  a_Static[47][69] = 0;

  //Eq. (49)
  a_CoefObj[48][Ind_i6st] = 0;
  a_Static[48][Ind_i6st] = 1.0;
  a_CoefObj[48][Ind_i6ct] = 0;
  a_Static[48][Ind_i6ct] = 1.0;
  a_CoefObj[48][Ind_i6] = 0;
  a_Static[48][Ind_i6] = -1.0;
  a_CoefObj[48][69] = 0;
  a_Static[48][69] = 0;


  //Eq. (50)
  a_CoefObj[49][Ind_i6sb] = 0;
  a_Static[49][Ind_i6sb] = 1.0;
  a_CoefObj[49][Ind_i6cb] = 0;
  a_Static[49][Ind_i6cb] = 1.0;
  a_CoefObj[49][Ind_i6_l] = 0;
  a_Static[49][Ind_i6_l] = -1.0;
  a_CoefObj[49][69] = 0;
  a_Static[49][69] = 0;


  //Eq. (51)
  a_CoefObj[50][Ind_v61] =  &Coef_C6_h_i;
  a_Static[50][Ind_v61] = 1.0;
  a_CoefObj[50][Ind_v67] =  &Coef_C6_h_i;
  a_Static[50][Ind_v67] = -1.0;
  a_CoefObj[50][Ind_i6] = 0;
  a_Static[50][Ind_i6] = -1.0;
  a_CoefObj[50][69] = &Coef_v61_v67_h_prev_C6_i;
  a_Static[50][69] = 1.0;

  //Eq. (52)
  a_CoefObj[51][Ind_v56] = &Coef_C61_h_i;
  a_Static[51][Ind_v56] = 1.0;
  a_CoefObj[51][Ind_v61] = &Coef_C61_h_i;
  a_Static[51][Ind_v61] = -1.0;
  a_CoefObj[51][Ind_i6ct] = 0;
  a_Static[51][Ind_i6ct] = -1.0;
  a_CoefObj[51][69] = &Coef_v56_v61_h_prev_C61_i;
  a_Static[51][69] = 1.0;


  //Eq. (53)
  a_CoefObj[52][Ind_v67] = &Coef_C62_h_i;
  a_Static[52][Ind_v67] = 1.0;
  a_CoefObj[52][Ind_v56] = &Coef_C62_h_i;
  a_Static[52][Ind_v56] = -1.0;
  a_CoefObj[52][Ind_i6cb] = 0;
  a_Static[52][Ind_i6cb] = -1.0;
  a_CoefObj[52][69] = &Coef_v67_v56_h_prev_C62_i;
  a_Static[52][69] = 1.0;


  //Eq. (54)
  a_CoefObj[53][Ind_v67] = &Coef_recp_rSrD71_i;
  a_Static[53][Ind_v67] = 1.0;
  a_CoefObj[53][Ind_v71] = &Coef_recp_rSrD71_i;
  a_Static[53][Ind_v71] = -1.0;
  a_CoefObj[53][Ind_i7st] = 0;
  a_Static[53][Ind_i7st] = -1.0;
  a_CoefObj[53][69] = 0;
  a_Static[53][69] = 0;

  //Eq. (55)
  a_CoefObj[54][Ind_v78] = &Coef_recp_rSrD72_i;
  a_Static[54][Ind_v78] = 1.0;
  a_CoefObj[54][Ind_v67] = &Coef_recp_rSrD72_i;
  a_Static[54][Ind_v67] = -1.0;
  a_CoefObj[54][Ind_i7sb] = 0;
  a_Static[54][Ind_i7sb] = -1.0;
  a_CoefObj[54][69] = 0;
  a_Static[54][69] = 0;

  //Eq. (56)
  a_CoefObj[55][Ind_inl] = 0;
  a_Static[55][Ind_inl] = 1.0;
  a_CoefObj[55][Ind_i7_l] = 0;
  a_Static[55][Ind_i7_l] = 1.0;
  a_CoefObj[55][Ind_i7] = 0;
  a_Static[55][Ind_i7] = -1.0;
  a_CoefObj[55][69] = 0;
  a_Static[55][69] = 0;

  //Eq. (57)
  a_CoefObj[56][Ind_i7st] = 0;
  a_Static[56][Ind_i7st] = 1.0;
  a_CoefObj[56][Ind_i7ct] = 0;
  a_Static[56][Ind_i7ct] = 1.0;
  a_CoefObj[56][Ind_i7] = 0;
  a_Static[56][Ind_i7] = -1.0;
  a_CoefObj[56][69] = 0;
  a_Static[56][69] = 0;


  //Eq. (58)
  a_CoefObj[57][Ind_i7sb] = 0;
  a_Static[57][Ind_i7sb] = 1.0;
  a_CoefObj[57][Ind_i7cb] = 0;
  a_Static[57][Ind_i7cb] = 1.0;
  a_CoefObj[57][Ind_i7_l] = 0;
  a_Static[57][Ind_i7_l] = -1.0;
  a_CoefObj[57][69] = 0;
  a_Static[57][69] = 0;


  //Eq. (59)
  a_CoefObj[58][Ind_v71] = &Coef_C7_h_i;
  a_Static[58][Ind_v71] = 1.0;
  a_CoefObj[58][Ind_v78] = &Coef_C7_h_i;
  a_Static[58][Ind_v78] = -1.0;
  a_CoefObj[58][Ind_i7] = 0;
  a_Static[58][Ind_i7] = -1.0;
  a_CoefObj[58][69] = &Coef_v71_v78_h_prev_C7_i;
  a_Static[58][69] = 1.0;

  //Eq. (60)
  a_CoefObj[59][Ind_v67] = &Coef_C71_h_i;
  a_Static[59][Ind_v67] = 1.0;
  a_CoefObj[59][Ind_v71] = &Coef_C71_h_i;
  a_Static[59][Ind_v71] = -1.0;
  a_CoefObj[59][Ind_i7ct] = 0;
  a_Static[59][Ind_i7ct] = -1.0;
  a_CoefObj[59][69] = &Coef_v67_v71_h_prev_C71_i;
  a_Static[59][69] = 1.0;


  //Eq. (61)
  a_CoefObj[60][Ind_v78] = &Coef_C72_h_i;
  a_Static[60][Ind_v78] = 1.0;
  a_CoefObj[60][Ind_v67] = &Coef_C72_h_i;
  a_Static[60][Ind_v67] = -1.0;
  a_CoefObj[60][Ind_i7cb] = 0;
  a_Static[60][Ind_i7cb] = -1.0;
  a_CoefObj[60][69] = &Coef_v78_v67_h_prev_C72_i;
  a_Static[60][69] = 1.0;
  

  //Eq. (62)
  a_CoefObj[61][Ind_v78] = &Coef_recp_rSrD81_i;
  a_Static[61][Ind_v78] = 1.0;
  a_CoefObj[61][Ind_v81] = &Coef_recp_rSrD81_i;
  a_Static[61][Ind_v81] = -1.0;
  a_CoefObj[61][Ind_i8st] = 0;
  a_Static[61][Ind_i8st] = -1.0;
  a_CoefObj[61][69] = 0;
  a_Static[61][69] = 0;


  //Eq. (63)
  a_CoefObj[62][Ind_v78] = &Coef_recp_rSrD82_i;
  a_Static[62][Ind_v78] = -1.0;
  a_CoefObj[62][Ind_i8sb] = 0;
  a_Static[62][Ind_i8sb] = -1.0;
  a_CoefObj[62][69] = &Coef_recp_rSrD82_i;
  a_Static[62][69] = - MINUS_DC_BUS_VOLTAGE;

  //Eq. (64)
  a_CoefObj[63][Ind_inl] = 0;
  a_Static[63][Ind_inl] = 1.0;
  a_CoefObj[63][Ind_i8_l] = 0;
  a_Static[63][Ind_i8_l] = 1.0;
  a_CoefObj[63][Ind_i8] = 0;
  a_Static[63][Ind_i8] = -1.0;
  a_CoefObj[63][69] = 0;
  a_Static[63][69] = 0;

  //Eq. (65)
  a_CoefObj[64][Ind_i8st] = 0;
  a_Static[64][Ind_i8st] = 1.0;
  a_CoefObj[64][Ind_i8ct] = 0;
  a_Static[64][Ind_i8ct] = 1.0;
  a_CoefObj[64][Ind_i8] = 0;
  a_Static[64][Ind_i8] = -1.0;
  a_CoefObj[64][69] = 0;
  a_Static[64][69] = 0;


  //Eq. (66)
  a_CoefObj[65][Ind_i8sb] = 0;
  a_Static[65][Ind_i8sb] = 1.0;
  a_CoefObj[65][Ind_i8cb] = 0;
  a_Static[65][Ind_i8cb] = 1.0;
  a_CoefObj[65][Ind_i8_l] = 0;
  a_Static[65][Ind_i8_l] = -1.0;
  a_CoefObj[65][69] = 0;
  a_Static[65][69] = 0;



  //Eq. (67)
  a_CoefObj[66][Ind_v81] = &Coef_C8_h_i;
  a_Static[66][Ind_v81] = 1.0;
  a_CoefObj[66][Ind_i8] = 0;
  a_Static[66][Ind_i8] = -1.0;
  a_CoefObj[66][69] = &Coef_v81_h_prev_plus_MinusBus_C8_i;
  a_Static[66][69] = 1.0;

  //Eq. (68)
  a_CoefObj[67][Ind_v78] = &Coef_C81_h_i;
  a_Static[67][Ind_v78] = 1.0;
  a_CoefObj[67][Ind_v81] = &Coef_C81_h_i;
  a_Static[67][Ind_v81] = -1.0;
  a_CoefObj[67][Ind_i8ct] = 0;
  a_Static[67][Ind_i8ct] = -1.0;
  a_CoefObj[67][69] = &Coef_v78_v81_h_prev_C81_i;
  a_Static[67][69] = 1.0;

  //Eq. (69)
  a_CoefObj[68][Ind_v78] = &Coef_C82_h_i;
  a_Static[68][Ind_v78] = -1.0;
  a_CoefObj[68][Ind_i8cb] = 0;
  a_Static[68][Ind_i8cb] = -1.0;
  a_CoefObj[68][69] = &Coef_minus_v78_h_prev_minus_MinusBus_C82_i;
  a_Static[68][69] = 1.0;




  
  //build the ODE Rvalue list.

  pOdeObjRValList = new OdeObjItem;          
  pCurOdeItem = pOdeObjRValList;
 
  pCurOdeItem->pOdeObject = OdeObjectList[ODE_FUNC_ia_ib_1];    




  //build the SWITCH Rvalue list.

  pSwitchObjRValList = new SwitchObjItem;
  pCurSwitchItem = pSwitchObjRValList;

  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS11];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;  
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS12];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS21];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS22];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS31];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS32];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS41];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS42];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS51];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS52];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS61];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS62];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS71];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS72];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS81];
  pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem; 
  pCurSwitchItem->pSwitchObject = SwitchObjectList[SWITCH_FUNC_rS82];


  //build the COEF Rvalue list.

  pCoefObjRValList = new CoefObjItem;
  pCurCoefItem = pCoefObjRValList;

  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v11_v12_h_prev_C1];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_minus_v11_h_prev_minus_PlusBus_C11];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v12_h_prev_plus_PlusBus_C12];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v21_v23_h_prev_C2];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v12_v21_h_prev_C21];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v23_v12_h_prev_C22];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v31_v34_h_prev_C3];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v23_v31_h_prev_C31];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v34_v23_h_prev_C32];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v41_v44_h_prev_C4];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v34_v41_h_prev_C41];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v44_v34_h_prev_C42];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_iLp_iLn_h_prev_Lp_Mpn];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_iLp_iLn_h_prev_Mnp_Ln];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v51_v56_h_prev_C5];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v45_v51_h_prev_C51];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v56_v45_h_prev_C52];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v61_v67_h_prev_C6];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v56_v61_h_prev_C61];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v67_v56_h_prev_C62];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v71_v78_h_prev_C7];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v67_v71_h_prev_C71];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v78_v67_h_prev_C72];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v81_h_prev_plus_MinusBus_C8];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_v78_v81_h_prev_C81];
  pCurCoefItem->pNextCoefItem = new CoefObjItem;
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  pCurCoefItem->pCoefObject = CoefObjectList[COEF_FUNC_minus_v78_h_prev_minus_MinusBus_C82];


  //build the SRC Rvalue list.

   pSrcObjRValList = new SrcObjItem;
   pCurSrcItem = pSrcObjRValList;

   pCurSrcItem->pSrcObject = SrcObjectList[SRC_FUNC_Vref];






#ifdef PLOT_SPICEOBJECT_PROBES_solution_vector
  DoProbes = TRUE;
#endif
#ifdef DO_PROBE_v11    
  pDoProbes_[Ind_v11] = TRUE;
  pProbeName[Ind_v11] = "v11";
#endif
#ifdef DO_PROBE_v12   
  pDoProbes_[Ind_v12] = TRUE;
  pProbeName[Ind_v12] = "v12";
#endif 
#ifdef DO_PROBE_v21   
  pDoProbes_[Ind_v21] = TRUE;
  pProbeName[Ind_v21] = "v21";
#endif 
#ifdef DO_PROBE_v23   
  pDoProbes_[Ind_v23] = TRUE;
  pProbeName[Ind_v23] = "v23";
#endif 
#ifdef DO_PROBE_v31   
  pDoProbes_[Ind_v31] = TRUE;
  pProbeName[Ind_v31] = "v31";
#endif 
#ifdef DO_PROBE_v34 
  pDoProbes_[Ind_v34] = TRUE;
  pProbeName[Ind_v34] = "v34";
#endif   
#ifdef DO_PROBE_v41  
  pDoProbes_[Ind_v41] = TRUE;
  pProbeName[Ind_v41] = "v41";
#endif  
#ifdef DO_PROBE_v44  
  pDoProbes_[Ind_v44] = TRUE;
  pProbeName[Ind_v44] = "v44";
#endif  
#ifdef DO_PROBE_v45   
  pDoProbes_[Ind_v45] = TRUE;
  pProbeName[Ind_v45] = "v45";
#endif 
#ifdef DO_PROBE_vs_a   
  pDoProbes_[Ind_vs_a] = TRUE;
  pProbeName[Ind_vs_a] = "vs_a";
#endif
#ifdef DO_PROBE_v51    
  pDoProbes_[Ind_v51] = TRUE;
  pProbeName[Ind_v51] = "v51";
#endif
#ifdef DO_PROBE_v56   
  pDoProbes_[Ind_v56] = TRUE;
  pProbeName[Ind_v56] = "v56";
#endif 
#ifdef DO_PROBE_v61   
  pDoProbes_[Ind_v61] = TRUE;
  pProbeName[Ind_v61] = "v61";
#endif 
#ifdef DO_PROBE_v67    
  pDoProbes_[Ind_v67] = TRUE;
  pProbeName[Ind_v67] = "v67";
#endif
#ifdef DO_PROBE_v71    
  pDoProbes_[Ind_v71] = TRUE;
  pProbeName[Ind_v71] = "v71";
#endif
#ifdef DO_PROBE_v78    
  pDoProbes_[Ind_v78] = TRUE;
  pProbeName[Ind_v78] = "v78";
#endif
#ifdef DO_PROBE_v81    
  pDoProbes_[Ind_v81] = TRUE;
  pProbeName[Ind_v81] = "v81";
#endif
#ifdef DO_PROBE_i1    
  pDoProbes_[Ind_i1] = TRUE;
  pProbeName[Ind_i1] = "i1";
#endif
#ifdef DO_PROBE_i1_l  
  pDoProbes_[Ind_i1_l] = TRUE;
  pProbeName[Ind_i1_l] = "i1_l";
#endif
#ifdef DO_PROBE_ipl   
  pDoProbes_[Ind_ipl] = TRUE;
  pProbeName[Ind_ipl] = "ipl";
#endif 
#ifdef DO_PROBE_i1st
  pDoProbes_[Ind_i1st] = TRUE;
  pProbeName[Ind_i1st] = "i1st";
#endif
#ifdef DO_PROBE_i1ct
  pDoProbes_[Ind_i1ct] = TRUE;
  pProbeName[Ind_i1ct] = "i1ct";
#endif
#ifdef DO_PROBE_i1sb
  pDoProbes_[Ind_i1sb] = TRUE;
  pProbeName[Ind_i1sb] = "i1sb";
#endif
#ifdef DO_PROBE_i1cb
  pDoProbes_[Ind_i1cb] = TRUE;
  pProbeName[Ind_i1cb] = "i1cb";
#endif
#ifdef DO_PROBE_i2    
  pDoProbes_[Ind_i2] = TRUE;
  pProbeName[Ind_i2] = "i2";
#endif 
#ifdef DO_PROBE_i2_l    
  pDoProbes_[Ind_i2_l] = TRUE;
  pProbeName[Ind_i2_l] = "i2_l";
#endif 
#ifdef DO_PROBE_i2st
  pDoProbes_[Ind_i2st] = TRUE;
  pProbeName[Ind_i2st] = "i2st";
#endif
#ifdef DO_PROBE_i2ct
  pDoProbes_[Ind_i2ct] = TRUE;
  pProbeName[Ind_i2ct] = "i2ct";
#endif
#ifdef DO_PROBE_i2sb
  pDoProbes_[Ind_i2sb] = TRUE;
  pProbeName[Ind_i2sb] = "i2sb";
#endif
#ifdef DO_PROBE_i2cb
  pDoProbes_[Ind_i2cb] = TRUE;
  pProbeName[Ind_i2cb] = "i2cb";
#endif
#ifdef DO_PROBE_i3   
  pDoProbes_[Ind_i3] = TRUE;
  pProbeName[Ind_i3] = "i3";
#endif  
#ifdef DO_PROBE_i3_l   
  pDoProbes_[Ind_i3_l] = TRUE;
  pProbeName[Ind_i3_l] = "i3_l";
#endif 
#ifdef DO_PROBE_i3st
  pDoProbes_[Ind_i3st] = TRUE;
  pProbeName[Ind_i3st] = "i3st";
#endif
#ifdef DO_PROBE_i3ct
  pDoProbes_[Ind_i3ct] = TRUE;
  pProbeName[Ind_i3ct] = "i3ct";
#endif
#ifdef DO_PROBE_i3sb
  pDoProbes_[Ind_i3sb] = TRUE;
  pProbeName[Ind_i3sb] = "i3sb";
#endif
#ifdef DO_PROBE_i3cb
  pDoProbes_[Ind_i3cb] = TRUE;
  pProbeName[Ind_i3cb] = "i3cb";
#endif
#ifdef DO_PROBE_i4 
  pDoProbes_[Ind_i4] = TRUE;
  pProbeName[Ind_i4] = "i4";
#endif    
#ifdef DO_PROBE_i4_l 
  pDoProbes_[Ind_i4_l] = TRUE;
  pProbeName[Ind_i4_l] = "i4_l";
#endif 
#ifdef DO_PROBE_i4st
  pDoProbes_[Ind_i4st] = TRUE;
  pProbeName[Ind_i4st] = "i4st";
#endif
#ifdef DO_PROBE_i4ct
  pDoProbes_[Ind_i4ct] = TRUE;
  pProbeName[Ind_i4ct] = "i4ct";
#endif
#ifdef DO_PROBE_i4sb
  pDoProbes_[Ind_i4sb] = TRUE;
  pProbeName[Ind_i4sb] = "i4sb";
#endif
#ifdef DO_PROBE_i4cb
  pDoProbes_[Ind_i4cb] = TRUE;
  pProbeName[Ind_i4cb] = "i4cb";
#endif
#ifdef DO_PROBE_i5  
  pDoProbes_[Ind_i5] = TRUE;
  pProbeName[Ind_i5] = "i5";
#endif   
#ifdef DO_PROBE_i5_l  
  pDoProbes_[Ind_i5_l] = TRUE;
  pProbeName[Ind_i5_l] = "i5_l";
#endif   
#ifdef DO_PROBE_i5st
  pDoProbes_[Ind_i5st] = TRUE;
  pProbeName[Ind_i5st] = "i5st";
#endif
#ifdef DO_PROBE_i5ct
  pDoProbes_[Ind_i5ct] = TRUE;
  pProbeName[Ind_i5ct] = "i5ct";
#endif
#ifdef DO_PROBE_i5sb
  pDoProbes_[Ind_i5sb] = TRUE;
  pProbeName[Ind_i5sb] = "i5sb";
#endif
#ifdef DO_PROBE_i5cb
  pDoProbes_[Ind_i5cb] = TRUE;
  pProbeName[Ind_i5cb] = "i5cb";
#endif
#ifdef DO_PROBE_inl  
  pDoProbes_[Ind_inl] = TRUE;
  pProbeName[Ind_inl] = "inl";
#endif  
#ifdef DO_PROBE_i6   
 pDoProbes_[Ind_i6] = TRUE;
 pProbeName[Ind_i6] = "i6";
#endif  
#ifdef DO_PROBE_i6_l   
 pDoProbes_[Ind_i6_l] = TRUE;
 pProbeName[Ind_i6_l] = "i6_l";
#endif  
#ifdef DO_PROBE_i6st
  pDoProbes_[Ind_i6st] = TRUE;
  pProbeName[Ind_i6st] = "i6st";
#endif
#ifdef DO_PROBE_i6ct
  pDoProbes_[Ind_i6ct] = TRUE;
  pProbeName[Ind_i6ct] = "i6ct";
#endif
#ifdef DO_PROBE_i6sb
  pDoProbes_[Ind_i6sb] = TRUE;
  pProbeName[Ind_i6sb] = "i6sb";
#endif
#ifdef DO_PROBE_i6cb
  pDoProbes_[Ind_i6cb] = TRUE;
  pProbeName[Ind_i6cb] = "i6cb";
#endif
#ifdef DO_PROBE_i7  
 pDoProbes_[Ind_i7] = TRUE;
 pProbeName[Ind_i7] = "i7";
#endif   
#ifdef DO_PROBE_i7_l  
 pDoProbes_[Ind_i7_l] = TRUE;
 pProbeName[Ind_i7_l] = "i7_l";
#endif
#ifdef DO_PROBE_i7st
  pDoProbes_[Ind_i7st] = TRUE;
  pProbeName[Ind_i7st] = "i7st";
#endif
#ifdef DO_PROBE_i7ct
  pDoProbes_[Ind_i7ct] = TRUE;
  pProbeName[Ind_i7ct] = "i7ct";
#endif
#ifdef DO_PROBE_i7sb
  pDoProbes_[Ind_i7sb] = TRUE;
  pProbeName[Ind_i7sb] = "i7sb";
#endif
#ifdef DO_PROBE_i7cb
  pDoProbes_[Ind_i7cb] = TRUE;
  pProbeName[Ind_i7cb] = "i7cb";
#endif
#ifdef DO_PROBE_i8  
  pDoProbes_[Ind_i8] = TRUE;
  pProbeName[Ind_i8] = "i8";
#endif
#ifdef DO_PROBE_i8_l  
  pDoProbes_[Ind_i8_l] = TRUE;
  pProbeName[Ind_i8_l] = "i8_l";
#endif
#ifdef DO_PROBE_i8st
  pDoProbes_[Ind_i8st] = TRUE;
  pProbeName[Ind_i8st] = "i8st";
#endif
#ifdef DO_PROBE_i8ct
  pDoProbes_[Ind_i8ct] = TRUE;
  pProbeName[Ind_i8ct] = "i8ct";
#endif
#ifdef DO_PROBE_i8sb
  pDoProbes_[Ind_i8sb] = TRUE;
  pProbeName[Ind_i8sb] = "i8sb";
#endif
#ifdef DO_PROBE_i8cb
  pDoProbes_[Ind_i8cb] = TRUE;
  pProbeName[Ind_i8cb] = "i8cb";
#endif


#ifdef DUMP_VS_A_OUTPUT_SIGNAL
#ifdef ENABLE_OMEGA_CTRL
	fp_vs_a_output = fopen("vs_a_output_omega.dat", "w");
#else
	fp_vs_a_output = fopen("vs_a_output_no_omega.dat", "w");
#endif


#endif


}

Spice_MultiLevel::~Spice_MultiLevel(void)
{
#ifdef DUMP_VS_A_OUTPUT_SIGNAL
	fclose(fp_vs_a_output);
#endif
}



void Spice_MultiLevel::OdeRValueUpdate(void)
{

  OdeObjItem * pCurOdeItem;

  //Update Rvalues
  pCurOdeItem = pOdeObjRValList;

  ((class ia_ib_1 *) pCurOdeItem->pOdeObject)->vs_a = a[Ind_vs_a][69];

#ifdef DUMP_VS_A_OUTPUT_SIGNAL

#ifdef DUMP_IPL_SIGNAL_INSTEAD_OF_VS_A_SIGNAL
  fprintf(fp_vs_a_output, "%g\n", a[Ind_ipl][69]);
#else
  fprintf(fp_vs_a_output, "%g\n", a[Ind_vs_a][69]);
#endif


#endif

  //These updates added for MultLevelDCTest 3 Phase.
  ipl_bus_current = a[Ind_ipl][69];
  vs_plotdata =  a[Ind_vs_a][69];

}

void Spice_MultiLevel::SwitchRValueUpdate(void)
{
  SwitchObjItem * pCurSwitchItem;

  //Update Rvalues
  pCurSwitchItem = pSwitchObjRValList;

  ((class Switch_rS11 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i1st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS12 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i1sb][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS21 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i2st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS22 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i2sb][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS31 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i3st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS32 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i3sb][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS41 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i4st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS42 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i4sb][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS51 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i5st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS52 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i5sb][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS61 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i6st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS62 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i6sb][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS71 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i7st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS72 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i7sb][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS81 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i8st][69];
  pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  ((class Switch_rS82 *) pCurSwitchItem->pSwitchObject)->DevCur = a[Ind_i8sb][69];




}


void Spice_MultiLevel::CoefRValueUpdate(void)
{

  CoefObjItem * pCurCoefItem;

  //Update Rvalues
  pCurCoefItem = pCoefObjRValList;

  ((Coef_v11_v12_h_prev_C1 *) pCurCoefItem->pCoefObject)->v11_ = a[Ind_v11][69];
  ((Coef_v11_v12_h_prev_C1 *) pCurCoefItem->pCoefObject)->v12_ = a[Ind_v12][69];
  ((Coef_v11_v12_h_prev_C1 *) pCurCoefItem->pCoefObject)->i1_ = a[Ind_i1][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_minus_v11_h_prev_minus_PlusBus_C11 *) pCurCoefItem->pCoefObject)->v11_ = a[Ind_v11][69];
  ((Coef_minus_v11_h_prev_minus_PlusBus_C11 *) pCurCoefItem->pCoefObject)->i1ct_ = a[Ind_i1ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v12_h_prev_plus_PlusBus_C12 *) pCurCoefItem->pCoefObject)->v12_ = a[Ind_v12][69];
  ((Coef_v12_h_prev_plus_PlusBus_C12 *) pCurCoefItem->pCoefObject)->i1cb_ = a[Ind_i1cb][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v21_v23_h_prev_C2 *) pCurCoefItem->pCoefObject)->v21_ = a[Ind_v21][69];
  ((Coef_v21_v23_h_prev_C2 *) pCurCoefItem->pCoefObject)->v23_ = a[Ind_v23][69];
  ((Coef_v21_v23_h_prev_C2 *) pCurCoefItem->pCoefObject)->i2_ = a[Ind_i2][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v12_v21_h_prev_C21 *) pCurCoefItem->pCoefObject)->v12_ = a[Ind_v12][69];
  ((Coef_v12_v21_h_prev_C21 *) pCurCoefItem->pCoefObject)->v21_ = a[Ind_v21][69];
  ((Coef_v12_v21_h_prev_C21 *) pCurCoefItem->pCoefObject)->i2ct_ = a[Ind_i2ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v23_v12_h_prev_C22 *) pCurCoefItem->pCoefObject)->v23_ = a[Ind_v23][69];
  ((Coef_v23_v12_h_prev_C22 *) pCurCoefItem->pCoefObject)->v12_ = a[Ind_v12][69];
  ((Coef_v23_v12_h_prev_C22 *) pCurCoefItem->pCoefObject)->i2cb_ = a[Ind_i2cb][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v31_v34_h_prev_C3 *) pCurCoefItem->pCoefObject)->v31_ = a[Ind_v31][69];
  ((Coef_v31_v34_h_prev_C3 *) pCurCoefItem->pCoefObject)->v34_ = a[Ind_v34][69];
  ((Coef_v31_v34_h_prev_C3 *) pCurCoefItem->pCoefObject)->i3_ = a[Ind_i3][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v23_v31_h_prev_C31 *) pCurCoefItem->pCoefObject)->v23_ = a[Ind_v23][69];
  ((Coef_v23_v31_h_prev_C31 *) pCurCoefItem->pCoefObject)->v31_ = a[Ind_v31][69];
  ((Coef_v23_v31_h_prev_C31 *) pCurCoefItem->pCoefObject)->i3ct_ = a[Ind_i3ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v34_v23_h_prev_C32 *) pCurCoefItem->pCoefObject)->v34_ = a[Ind_v34][69];
  ((Coef_v34_v23_h_prev_C32 *) pCurCoefItem->pCoefObject)->v23_ = a[Ind_v23][69];
  ((Coef_v34_v23_h_prev_C32 *) pCurCoefItem->pCoefObject)->i3cb_ = a[Ind_i3cb][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v41_v44_h_prev_C4 *) pCurCoefItem->pCoefObject)->v41_ = a[Ind_v41][69];
  ((Coef_v41_v44_h_prev_C4 *) pCurCoefItem->pCoefObject)->v44_ = a[Ind_v44][69];
  ((Coef_v41_v44_h_prev_C4 *) pCurCoefItem->pCoefObject)->i4_ = a[Ind_i4][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v34_v41_h_prev_C41 *) pCurCoefItem->pCoefObject)->v34_ = a[Ind_v34][69];
  ((Coef_v34_v41_h_prev_C41 *) pCurCoefItem->pCoefObject)->v41_ = a[Ind_v41][69];
  ((Coef_v34_v41_h_prev_C41 *) pCurCoefItem->pCoefObject)->i4ct_ = a[Ind_i4ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v44_v34_h_prev_C42 *) pCurCoefItem->pCoefObject)->v44_ = a[Ind_v44][69];
  ((Coef_v44_v34_h_prev_C42 *) pCurCoefItem->pCoefObject)->v34_ = a[Ind_v34][69];
  ((Coef_v44_v34_h_prev_C42 *) pCurCoefItem->pCoefObject)->i4cb_ = a[Ind_i4cb][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;


  ((Coef_iLp_iLn_h_prev_Lp_Mpn *) pCurCoefItem->pCoefObject)->iLp_ = a[Ind_iLp][69];
  ((Coef_iLp_iLn_h_prev_Lp_Mpn *) pCurCoefItem->pCoefObject)->iLn_ = a[Ind_iLn][69];
  ((Coef_iLp_iLn_h_prev_Lp_Mpn *) pCurCoefItem->pCoefObject)->v44_ = a[Ind_v44][69];
  ((Coef_iLp_iLn_h_prev_Lp_Mpn *) pCurCoefItem->pCoefObject)->vs_a_ = a[Ind_vs_a][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_iLp_iLn_h_prev_Mnp_Ln *) pCurCoefItem->pCoefObject)->iLp_ = a[Ind_iLp][69];
  ((Coef_iLp_iLn_h_prev_Mnp_Ln *) pCurCoefItem->pCoefObject)->iLn_ = a[Ind_iLn][69];
  ((Coef_iLp_iLn_h_prev_Mnp_Ln *) pCurCoefItem->pCoefObject)->vs_a_ = a[Ind_vs_a][69];
  ((Coef_iLp_iLn_h_prev_Mnp_Ln *) pCurCoefItem->pCoefObject)->v45_ = a[Ind_v45][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;


  ((Coef_v51_v56_h_prev_C5 *) pCurCoefItem->pCoefObject)->v51_ = a[Ind_v51][69];
  ((Coef_v51_v56_h_prev_C5 *) pCurCoefItem->pCoefObject)->v56_ = a[Ind_v56][69];
  ((Coef_v51_v56_h_prev_C5 *) pCurCoefItem->pCoefObject)->i5_ = a[Ind_i5][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v45_v51_h_prev_C51 *) pCurCoefItem->pCoefObject)->v45_ = a[Ind_v45][69];
  ((Coef_v45_v51_h_prev_C51 *) pCurCoefItem->pCoefObject)->v51_ = a[Ind_v51][69];
  ((Coef_v45_v51_h_prev_C51 *) pCurCoefItem->pCoefObject)->i5ct_ = a[Ind_i5ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v56_v45_h_prev_C52 *) pCurCoefItem->pCoefObject)->v56_ = a[Ind_v56][69];
  ((Coef_v56_v45_h_prev_C52 *) pCurCoefItem->pCoefObject)->v45_ = a[Ind_v45][69];
  ((Coef_v56_v45_h_prev_C52 *) pCurCoefItem->pCoefObject)->i5cb_ = a[Ind_i5cb][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v61_v67_h_prev_C6 *) pCurCoefItem->pCoefObject)->v61_ = a[Ind_v61][69];
  ((Coef_v61_v67_h_prev_C6 *) pCurCoefItem->pCoefObject)->v67_ = a[Ind_v67][69];
  ((Coef_v61_v67_h_prev_C6 *) pCurCoefItem->pCoefObject)->i6_ = a[Ind_i6][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v56_v61_h_prev_C61 *) pCurCoefItem->pCoefObject)->v56_ = a[Ind_v56][69];
  ((Coef_v56_v61_h_prev_C61 *) pCurCoefItem->pCoefObject)->v61_ = a[Ind_v61][69];
  ((Coef_v56_v61_h_prev_C61 *) pCurCoefItem->pCoefObject)->i6ct_ = a[Ind_i6ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v67_v56_h_prev_C62 *) pCurCoefItem->pCoefObject)->v67_ = a[Ind_v67][69];
  ((Coef_v67_v56_h_prev_C62 *) pCurCoefItem->pCoefObject)->v56_ = a[Ind_v56][69];
  ((Coef_v67_v56_h_prev_C62 *) pCurCoefItem->pCoefObject)->i6cb_ = a[Ind_i6cb][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v71_v78_h_prev_C7 *) pCurCoefItem->pCoefObject)->v71_ = a[Ind_v71][69];
  ((Coef_v71_v78_h_prev_C7 *) pCurCoefItem->pCoefObject)->v78_ = a[Ind_v78][69];
  ((Coef_v71_v78_h_prev_C7 *) pCurCoefItem->pCoefObject)->i7_ = a[Ind_i7][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v67_v71_h_prev_C71 *) pCurCoefItem->pCoefObject)->v67_ = a[Ind_v67][69];
  ((Coef_v67_v71_h_prev_C71 *) pCurCoefItem->pCoefObject)->v71_ = a[Ind_v71][69];
  ((Coef_v67_v71_h_prev_C71 *) pCurCoefItem->pCoefObject)->i7ct_ = a[Ind_i7ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v78_v67_h_prev_C72 *) pCurCoefItem->pCoefObject)->v78_ = a[Ind_v78][69];
  ((Coef_v78_v67_h_prev_C72 *) pCurCoefItem->pCoefObject)->v67_ = a[Ind_v67][69];
  ((Coef_v78_v67_h_prev_C72 *) pCurCoefItem->pCoefObject)->i7cb_ = a[Ind_i7cb][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v81_h_prev_plus_MinusBus_C8 *) pCurCoefItem->pCoefObject)->v81_ = a[Ind_v81][69];
  ((Coef_v81_h_prev_plus_MinusBus_C8 *) pCurCoefItem->pCoefObject)->i8_ = a[Ind_i8][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_v78_v81_h_prev_C81 *) pCurCoefItem->pCoefObject)->v78_ = a[Ind_v78][69];
  ((Coef_v78_v81_h_prev_C81 *) pCurCoefItem->pCoefObject)->v81_ = a[Ind_v81][69];
  ((Coef_v78_v81_h_prev_C81 *) pCurCoefItem->pCoefObject)->i8ct_ = a[Ind_i8ct][69];
  pCurCoefItem = pCurCoefItem->pNextCoefItem;
  ((Coef_minus_v78_h_prev_minus_MinusBus_C82 *) pCurCoefItem->pCoefObject)->v78_ = a[Ind_v78][69];
  ((Coef_minus_v78_h_prev_minus_MinusBus_C82 *) pCurCoefItem->pCoefObject)->i8cb_ = a[Ind_i8cb][69];

}

void Spice_MultiLevel::SrcRValueUpdate(void)
{
  SrcObjItem * pCurSrcItem;


  //Update Rvalues
  pCurSrcItem = pSrcObjRValList;

  ((Vref *) pCurSrcItem->pSrcObject)->v11_v12 = a[Ind_v11][69] - a[Ind_v12][69];
  ((Vref *) pCurSrcItem->pSrcObject)->v21_v23 = a[Ind_v21][69] - a[Ind_v23][69];
  ((Vref *) pCurSrcItem->pSrcObject)->v31_v34 = a[Ind_v31][69] - a[Ind_v34][69];
  ((Vref *) pCurSrcItem->pSrcObject)->v41_v44 = a[Ind_v41][69] - a[Ind_v44][69];
  ((Vref *) pCurSrcItem->pSrcObject)->v51_v56 = a[Ind_v51][69] - a[Ind_v56][69];
  ((Vref *) pCurSrcItem->pSrcObject)->v61_v67 = a[Ind_v61][69] - a[Ind_v67][69];
  ((Vref *) pCurSrcItem->pSrcObject)->v71_v78 = a[Ind_v71][69] - a[Ind_v78][69];
  ((Vref *) pCurSrcItem->pSrcObject)->v81_v88 = a[Ind_v81][69] - MINUS_DC_BUS_VOLTAGE;


}



// ---------------------------------------------------------------

// **************************************************************

 
                                                                                  

// **** SimuObject ************************************************




SimuObject Simulation;

SimuObject::SimuObject(void)
{
  OdeObjItem * pCurOdeItem;
  SrcObjItem * pCurSrcItem;
  CtrlObjItem * pCurCtrlItem;
  CtrlObjGroup * pCurCtrlGroup;
  CoefObjItem * pCurCoefItem;         
  SpiceObjItem * pCurSpiceItem;
  SwitchObjItem * pCurSwitchItem;


  int i;


  SimuTime = .2;

  
  MinimumPlotTime = 0;
  MaximumPlotTime = 1;



  ExceptionTime = .05;

  //For this example the setting of "RelTol" is important. If it is set to large
  //the system will become unstable. 
  RelTol =  .0000000001;
  //Setting "AbsTol" too small will make a system with a decaying oscillation
  //take too long to finish.
  AbsTol =  .0000000000001;
  //typical error correction setting as per "RungeKutta_java.htm" (never set higher then "1.0")
  Safety = .98;
//  h_start = .000000001;
// !!!!!!!!!!!! TEMPORARY !!!!!!!!!!!!!!!!!!1
  h_start = .00000064;		//App_NgSpiceTest.cir produces 93868 steps over 60 mSec. We will match this here
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!1

  h = h_start;


// !!!!!!!!!!!! TEMPORARY !!!!!!!!!!!!!!!!!!1
  OdeSimuFixed = true;		//Run fixed mode. We do not have adaptive time stepping workin yet for Spice mode.


// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!1



  //set to a value other then "0" to clamp maximum "h";
//  h_max = .0000002;
// !!!!!!!!!!!! TEMPORARY !!!!!!!!!!!!!!!!!!1
  h_max = .00000064;
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


  //set to store and plot "h" parameter.
  Do_h_Plot = 1;
  
  TimeQuantum = h;
  CtrlTimeQuantum = QUANTUM_PERIOD;

  //set tp force CTRL update on first ODE update.
  CtrlTimeAccumulator = 0;

  SrcPeriodQuantum = QUANTUM_PERIOD;
  SrcPeriodAccumulator = 0;
 

  //build the SRC equation list
  pSrcEquationList = new SrcObjItem;
  pCurSrcItem = pSrcEquationList;
  i = 0;
  while(SrcObjectList[i]){
    pCurSrcItem->pSrcObject = SrcObjectList[i];
    i++;
    if(!SrcObjectList[i])
      break;
    pCurSrcItem->pNextSrcItem = new SrcObjItem;
    pCurSrcItem = pCurSrcItem->pNextSrcItem;


  }

  //build the ODE equation list.
  pOdeEquationList = new OdeObjItem;
  pCurOdeItem = pOdeEquationList;
  i = 0;
  while(1){
    pCurOdeItem->pOdeObject = OdeObjectList[i];
    i++;
    if(!OdeObjectList[i])
      break;
    pCurOdeItem->pNextOdeItem = new OdeObjItem;
    pCurOdeItem = pCurOdeItem->pNextOdeItem;
  }



  if(CtrlObjectList[0]){

    pCtrlGroupList = new CtrlObjGroup;
    pCurCtrlGroup = pCtrlGroupList;

    pCurCtrlGroup->pCtrlEquationList = new CtrlObjItem;
    pCurCtrlItem = pCurCtrlGroup->pCtrlEquationList;
    //set tp force CTRL update on first ODE update.
    pCurCtrlGroup->QuantumCount = 1;
    pCurCtrlGroup->QuantumNum = CtrlObjectQuantum[0];

    i = 0;
    while(1){
      pCurCtrlItem->pCtrlObject = CtrlObjectList[i];   
      i++;
      if(!CtrlObjectList[i])                 
	break;

      if(CtrlObjectQuantum[i] == CtrlObjectQuantum[i-1]){
	pCurCtrlItem->pNextCtrlItem = new CtrlObjItem;
	pCurCtrlItem = pCurCtrlItem->pNextCtrlItem;
      }
      else{
	//create a new group...
	pCtrlGroupList->pNextCtrlGroup = new CtrlObjGroup;
	pCurCtrlGroup = pCtrlGroupList->pNextCtrlGroup;

	pCurCtrlGroup->pCtrlEquationList = new CtrlObjItem;
	pCurCtrlItem = pCurCtrlGroup->pCtrlEquationList;
	//set tp force CTRL update on first ODE update.
	pCurCtrlGroup->QuantumCount = 1;
	pCurCtrlGroup->QuantumNum = CtrlObjectQuantum[i];  
      }
    }
  }

  //build the COEF equation list.              //NOTE: The COEF list is used only for exception processing.
  pCoefEquationList = new CoefObjItem;
  pCurCoefItem = pCoefEquationList;
  i = 0;
  while(1){
    pCurCoefItem->pCoefObject = CoefObjectList[i];
    i++;
    if(!CoefObjectList[i])
      break;
    pCurCoefItem->pNextCoefItem = new CoefObjItem;
    pCurCoefItem = pCurCoefItem->pNextCoefItem;
  }



  //build the SPICE equation list.
  pSpiceEquationList = new SpiceObjItem;
  pCurSpiceItem = pSpiceEquationList;
  i = 0;
  while(1){
    pCurSpiceItem->pSpiceObject = SpiceObjectList[i];
    i++;
    if(!SpiceObjectList[i])
      break;
    pCurSpiceItem->pNextSpiceItem = new SpiceObjItem;
    pCurSpiceItem = pCurSpiceItem->pNextSpiceItem;
  }
 
  //build the SWITCH equation list.
  pSwitchEquationList = new SwitchObjItem;
  pCurSwitchItem = pSwitchEquationList;
  i = 0;
  while(1){
    pCurSwitchItem->pSwitchObject = SwitchObjectList[i];
    i++;
    if(!SwitchObjectList[i])
      break;
    pCurSwitchItem->pNextSwitchItem = new SwitchObjItem;
    pCurSwitchItem = pCurSwitchItem->pNextSwitchItem;
  }





}

SimuObject::~SimuObject(void)
{

  //(ideally we should delete all make allocation make in the constructor, here)

}



// ************************************************************






// ***** This section added to implement 3 Phase simulation **************************************************



FILE * phase_a_fp;
FILE * phase_b_fp;
FILE * phase_c_fp;

FILE * bus_current_fp;
FILE * bus_voltage_fp;

FILE * debug_phase_a_fp;
FILE * debug_phase_b_fp;
FILE * debug_phase_c_fp;


unsigned int PhaseInstance;    //0, 1, or 2 for Phase A, B, or C respectively.


// Recreation of Cell parameters defined by "Figure_45_a", "Figure_45_b", and ""Figure_45_c" above.
// (NOTE: "#define Figure_45_c" near line 1675 should be un-commented above.)

double RailToRailStateTable_a[][6] = {{PWM_GAIN*(1.0),	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(1.0),	PWM_GAIN*(0),  		PWM_GAIN*(0),	    0.0,	   	0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(0.25),	PWM_GAIN*(0),  		PWM_GAIN*(0),	   -1.0,	    0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-0.5),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(0.25),	PWM_GAIN*(0),  		PWM_GAIN*(0),	    1.0,	    0.0,		0.0}};	//315 Degrees

double RailToRailStateTable_b[][6] = {{PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},  	//  0 Degress
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 45 Degrees
                                    {PWM_GAIN*(1.0),   	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(0.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0}};	//315 Degrees

double RailToRailStateTable_c[][6] = {{PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,	    0.0,		0.0},   //  0 Degress
									{PWM_GAIN*(-0.25),	PWM_GAIN*(0),   	PWM_GAIN*(0),	   -1.0,	    0.0,		0.0},  	// 45 Degrees
                                    {PWM_GAIN*(0.5),  	PWM_GAIN*(0),		PWM_GAIN*(0),	   	0.0,		0.0,		0.0},	// 90 Degrees
                                    {PWM_GAIN*(-0.25), 	PWM_GAIN*(0),   	PWM_GAIN*(0),	    1.0,	    0.0,		0.0},	//135 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),		PWM_GAIN*(0),	    0.0,	    0.0,		0.0},	//180 Degrees
                                    {PWM_GAIN*(-1.0),	PWM_GAIN*(0),   	PWM_GAIN*(0),		0.0,		0.0,		0.0},	//225 Degrees
                                    {PWM_GAIN*(-1.0), 	PWM_GAIN*(0), 		PWM_GAIN*(0),	    0.0,		0.0,		0.0},	//270 Degrees
                                    {PWM_GAIN*(-1.0),  	PWM_GAIN*(0),  		PWM_GAIN*(0),		0.0,		0.0,		0.0}};	//315 Degrees


void InitializeCellParameters(void)
{
	if(PhaseInstance == 0)
	{
		memcpy(RailToRailStateTable, RailToRailStateTable_a, sizeof(RailToRailStateTable));
		CONSTANT_CURRENT_SOURCE_LOAD = 20.0;

	}
	else if(PhaseInstance == 1)
	{
		memcpy(RailToRailStateTable, RailToRailStateTable_b, sizeof(RailToRailStateTable));
		CONSTANT_CURRENT_SOURCE_LOAD = .001;
	}
	else  //(Phase C, PhaseInstance == 2)
	{
		memcpy(RailToRailStateTable, RailToRailStateTable_c, sizeof(RailToRailStateTable));
		CONSTANT_CURRENT_SOURCE_LOAD = -20.0;

	}



}

double ipl_b_bus_current;
double ipl_c_bus_current;
	//(Like the single phase simulation, we start this current (and "iln") at zero.)
double ipl_a_bus_current_ = 0;
double ipl_b_bus_current_ = 0;
double ipl_c_bus_current_ = 0;


double PLUS_DC_BUS_VOLTAGE_3PHASE;

#define PLUS_DC_BUS_VOLTAGE_MAIN 	1000.0

#define L_Bus .25e-3
#define R_Bus .001

//We are doing  "fixed" time as defined in "SimuObject::SimuObject()" with "h_start" set at .00000064.
#define RECP_H  ( (double) 1.0 / .00000064)

void DoPlusDCBusSimulation(void)
{
	double bus_current = ipl_b_bus_current + ipl_c_bus_current + ipl_bus_current;  //("ipl_bus_current" in context with phase "a").
	double bus_current_ = ipl_b_bus_current_ + ipl_c_bus_current_ + ipl_a_bus_current_;

	PLUS_DC_BUS_VOLTAGE_3PHASE =
			PLUS_DC_BUS_VOLTAGE_MAIN - L_Bus * (bus_current - bus_current_ )  * RECP_H -
			R_Bus * bus_current;

	ipl_a_bus_current_ = ipl_bus_current;  //("ipl_bus_current" in context with phase "a").
	ipl_b_bus_current_ = ipl_b_bus_current;
	ipl_c_bus_current_ = ipl_c_bus_current;

	fprintf(bus_current_fp, "%g\n", bus_current);
	fprintf(bus_voltage_fp, "%.9g\n", PLUS_DC_BUS_VOLTAGE_3PHASE);  //(".9" precision improves differenciation noise slightly.)
	//
	//  NOTE: This problem looks to differenciation noise an not malfunction in the simulation.
	//
	//				Use "plot_bus_voltage_filtered.m" to see the signal with most of this noise filtered out.
	//
	//  This simulation looks to be correct. For the next step do the following.
	//
	//
	//			Modify debug_phase_a_fp, debug_phase_b_fp, debug_phase_c_fp and "plot_debug_dat.m"  to formally plot the individual
	//          bus currents.
	//
	//          Add simulation code to modify the "-" Bus voltage for simulation. Use a "define" statement to select plotting the "+" Bus
	//          and "-" Bus data seperately.
	//
	//
	//
	//
	//

}



	//This function modified from the original single phase simulation using "fork()" and other related functions.
bool ExecuteSimulation(void)
{

	//NOTE: See "ReadMe-MEA.txt" in this directory for references to examples for fork(), pipe() and "sem_<xxx>" commnads.

  int pid[2];
  int phase_b_read_pipefd[2];    //(NOTE: Kernel lockup unless we make two pipes for phase b and c.)
  int phase_b_write_pipefd[2];
  int phase_c_read_pipefd[2];
  int phase_c_write_pipefd[2];

  Simulation.OdeSimuType = ODE_SIMU_56;


  //Create the Phase a and Phase b pipes.
  pipe(phase_b_read_pipefd);
  pipe(phase_b_write_pipefd);
  pipe(phase_c_read_pipefd);
  pipe(phase_c_write_pipefd);
   
  // create a semaphore for each child: value=0
  sem_t *sem1 = sem_open("Phase_b_sync", O_CREAT|O_EXCL);
  sem_t *sem2 = sem_open("Phase_c_sync", O_CREAT|O_EXCL);

  pid[0]=fork();
  if(pid[0]==0){
	//This should be context for "Phase b" simulation

	  PhaseInstance = 1;
	  phase_b_fp = fopen("Phase_b_data.dat", "w");
	  debug_phase_b_fp = fopen("Debug_b_data.dat", "w");

	  close(phase_b_read_pipefd[1]);
	  close(phase_b_write_pipefd[0]);


	  InitializeCellParameters();


	  while(Simulation.t < Simulation.SimuTime){
		  if(!Simulation.DoOneInteration())
			  break;

		  fprintf(phase_b_fp, "%g\n", vs_plotdata);

		   //***** Use this for various tests (other phases must match selection). *******************
		   fprintf(debug_phase_b_fp, "%g\n", ipl_bus_current);


		   // ************************************************************************************************

		  write(phase_b_write_pipefd[1], &ipl_bus_current, sizeof(ipl_bus_current));
		  read(phase_b_read_pipefd[0], &PLUS_DC_BUS_VOLTAGE, sizeof(PLUS_DC_BUS_VOLTAGE));

	  }

	  fclose(phase_b_fp);
	  fclose(debug_phase_b_fp);

	  // wait on semaphore => blocks if value <= 0
	  sem_wait(sem1);


  }
  else{
       pid[1]=fork();
       if(pid[1]==0){
		//This should be context for "Phase c" simulation

    	   PhaseInstance = 2;
    	   phase_c_fp = fopen("Phase_c_data.dat", "w");
    	   debug_phase_c_fp = fopen("Debug_c_data.dat", "w");

    	   close(phase_c_read_pipefd[1]);
    	   close(phase_c_write_pipefd[0]);

    	   InitializeCellParameters();

    	   while(Simulation.t < Simulation.SimuTime){
    		   if(!Simulation.DoOneInteration())
    			   break;

    		   fprintf(phase_c_fp, "%g\n", vs_plotdata);

    		   //***** Use this for various tests (other phases must match selection).******************
    		   fprintf(debug_phase_c_fp, "%g\n", ipl_bus_current);


    		   // ************************************************************************************************

			   write(phase_c_write_pipefd[1], &ipl_bus_current, sizeof(ipl_bus_current));
			   read(phase_c_read_pipefd[0], &PLUS_DC_BUS_VOLTAGE, sizeof(PLUS_DC_BUS_VOLTAGE));

    	   }

    	   fclose(phase_c_fp);
    	   fclose(debug_phase_c_fp);

    	   // wait on semaphore => blocks if value <= 0
    	   sem_wait(sem2);


       }
       else{
    	   //This is the parent as is the context for "Phase a" simulation.

    	   PhaseInstance = 0;
    	   phase_a_fp = fopen("Phase_a_data.dat", "w");
    	   debug_phase_a_fp = fopen("Debug_a_data.dat", "w");

    	   bus_current_fp = fopen("Bus_current.dat", "w");;
    	   bus_voltage_fp = fopen("Bus_voltage.dat", "w");;


    	   close(phase_b_read_pipefd[0]);
    	   close(phase_b_write_pipefd[1]);

       	   close(phase_c_read_pipefd[0]);
           close(phase_c_write_pipefd[1]);


    	   InitializeCellParameters();

    	   while(Simulation.t < Simulation.SimuTime){
    		    if(!Simulation.DoOneInteration())
    		      break;

    		    fprintf(phase_a_fp, "%g\n", vs_plotdata);

    			 //***** Use this for various tests (other phases must match selection). ******************
    			 fprintf(debug_phase_a_fp, "%g\n", ipl_bus_current);


    			 // ************************************************************************************************


       		    read(phase_b_write_pipefd[0], &ipl_b_bus_current, sizeof(ipl_b_bus_current));
        		read(phase_c_write_pipefd[0], &ipl_c_bus_current, sizeof(ipl_c_bus_current));

        		DoPlusDCBusSimulation();

        		PLUS_DC_BUS_VOLTAGE =  PLUS_DC_BUS_VOLTAGE_3PHASE; //(Updating phase "a" here.)
        		write(phase_b_read_pipefd[1], &PLUS_DC_BUS_VOLTAGE_3PHASE, sizeof(PLUS_DC_BUS_VOLTAGE_3PHASE));
        		write(phase_c_read_pipefd[1], &PLUS_DC_BUS_VOLTAGE_3PHASE, sizeof(PLUS_DC_BUS_VOLTAGE_3PHASE));

    	   }

    	   fclose(phase_a_fp);
    	   fclose(debug_phase_a_fp);
    	   fclose(bus_current_fp);
    	   fclose(bus_voltage_fp);

           // signal semaphore1 (increment) releasing Phase b process.
           sem_post(sem1);
           // signal semaphore2 (increment) releasing Phase c process.
           sem_post(sem2);


           // wait for Phase b and Phase c processes to exit.
           waitpid(pid[0],0, 0);
           waitpid(pid[1],0, 0);




       }


  }


  return TRUE;
}

// ***********************************************************************************************************

void PlotResults(string TagNamesToPlot[], double ScaleFactors[], double MinPlotTimes[], double MaxPlotTimes[], 
		 double PlotTimeSteps[], double PlotTimeOffset)
{
  Simulation.PlotSimuResults(TagNamesToPlot, ScaleFactors, MinPlotTimes, MaxPlotTimes, PlotTimeSteps, PlotTimeOffset);


}
